/*
 * cmsfs-fuse - CMS EDF filesystem support for Linux
 * DASD specific functions.
 *
 * Copyright IBM Corp. 2010
 * Author(s): Jan Glauber <jang@linux.vnet.ibm.com>
 */

#define _GNU_SOURCE
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include "helper.h"
#include "edf.h"
#include "cmsfs-fuse.h"

typedef struct dasd_info {
	unsigned int devno;         /* S/390 devno */
	unsigned int real_devno;    /* for aliases */
	unsigned int schid;         /* S/390 subchannel identifier */
	unsigned int cu_type   :16; /* from SenseID */
	unsigned int cu_model  :8;  /* from SenseID */
	unsigned int dev_type  :16; /* from SenseID */
	unsigned int dev_model :8;  /* from SenseID */
	unsigned int open_count;
	unsigned int req_queue_len;
	unsigned int chanq_len;     /* length of chanq */
	char type[4];               /* from discipline.name, 'none' for unknown */
	unsigned int status;        /* current device level */
	unsigned int label_block;   /* where to find the VOLSER */
	unsigned int FBA_layout;    /* fixed block size (like AIXVOL) */
	unsigned int characteristics_size;
	unsigned int confdata_size;
	char characteristics[64];   /* from read_device_characteristics */
	char configuration_data[256]; /* from read_configuration_data */
} dasd_info_t;

typedef struct dasd_info2 {
	unsigned int devno;         /* S/390 devno */
	unsigned int real_devno;    /* for aliases */
	unsigned int schid;         /* S/390 subchannel identifier */
	unsigned int cu_type   :16; /* from SenseID */
	unsigned int cu_model  :8;  /* from SenseID */
	unsigned int dev_type  :16; /* from SenseID */
	unsigned int dev_model :8;  /* from SenseID */
	unsigned int open_count;
	unsigned int req_queue_len;
	unsigned int chanq_len;     /* length of chanq */
	char type[4];               /* from discipline.name, 'none' for unknown */
	unsigned int status;        /* current device level */
	unsigned int label_block;   /* where to find the VOLSER */
	unsigned int FBA_layout;    /* fixed block size (like AIXVOL) */
	unsigned int characteristics_size;
	unsigned int confdata_size;
	char characteristics[64];   /* from read_device_characteristics */
	char configuration_data[256]; /* from read_configuration_data */
	unsigned int format;          /* format info like formatted/cdl/ldl/... */
	unsigned int features;        /* dasd features like 'ro',...            */
	unsigned int reserved[8];
} dasd_info2_t;

#define HDIO_GETGEO	0x0301
#define BLKSSZGET	_IO(0x12, 104)
#define BIODASDINFO	_IOR('D', 1, dasd_info_t)
#define BIODASDINFO2	_IOR('D', 3, dasd_info2_t)

/* CMS disk label starts with ASCII string "CMS1" */
#define VOL_LABEL_EBCDIC 0xc3d4e2f1

#define DASD_FORMAT_LDL		1

static int disk_supported(int fd, struct cmsfs *cmsfs)
{
	unsigned int cms_id = VOL_LABEL_EBCDIC;
	struct cms_label label;
	int offset, rc;

	/* check that this is a ldl disk */
	if (cmsfs->format != DASD_FORMAT_LDL) {
		fprintf(stderr, COMP "Disk not LDL formatted\n");
		return 0;
	}

	/* label is on block number 3 */
	offset = 2 * cmsfs->blksize;

	rc = lseek(fd, offset, SEEK_SET);
	if (rc < 0) {
		perror(COMP "lseek failed");
		return 0;
	}

	rc = read(fd, &label, sizeof(label));
	if (rc < 0) {
		perror(COMP "read failed");
		return 0;
	}

	/* check that the label contains the CMS1 string */
	if (memcmp(label.id, &cms_id, sizeof(cms_id)) != 0) {
		fprintf(stderr, COMP "Disk is not a CMS disk\n");
		return 0;
	}

	DEBUG("  DOP: %d", label.dop);
	/* block number 5 means 0x4000... */
	cmsfs->fdir = (label.dop - 1) * cmsfs->blksize;
	DEBUG("  fdir: %lx", cmsfs->fdir);
	/* get disk usage for statfs */
	cmsfs->total_blocks = label.total_blocks;
	cmsfs->used_blocks = label.used_blocks;
	DEBUG("  Total blocks: %d  Used blocks: %d",
		cmsfs->total_blocks, cmsfs->used_blocks);
	return 1;
}

static void get_device_info_bdev(int fd, struct cmsfs *cmsfs)
{
	struct dasd_info2 *info = NULL;

	if (ioctl(fd, BLKSSZGET, &cmsfs->blksize) != 0)
		DIE("ioctl error get blocksize\n");

	info = malloc(sizeof(struct dasd_info2));
	if (info == NULL)
		DIE_PERROR("malloc failed");

	/* get disk information */
	if (ioctl(fd, BIODASDINFO2, info) == 0) {
		/* INFO2 failed - try INFO using the same (larger) buffer */
		if (ioctl(fd, BIODASDINFO, info) != 0)
			DIE("ioctl dasd info failed\n");
	}
	cmsfs->format = info->format;
	free(info);
}

static int blocksizes[] = { 4096, 512, 2048, 1024 };

static void get_device_info_file(int fd, struct cmsfs *cmsfs)
{
	unsigned int cms_id = VOL_LABEL_EBCDIC;
	unsigned int i;
	char label[4];
	off_t offset;
	int rc;

	cmsfs->blksize = 0;

	/*
	 * Read the blocksize from label. Unfortunately the blocksize
	 * position depends on the blocksize... time for some heuristics.
	 */
	for (i = 0; i < ARRAY_SIZE(blocksizes); i++) {
		offset = blocksizes[i] * 2;

		rc = lseek(fd, offset, SEEK_SET);
		if (rc < 0)
			DIE_PERROR("lseek failed");

		rc = read(fd, &label, 4);
		if (rc < 0)
			DIE_PERROR("read failed");

		/* check if the label contains the CMS1 string */
		if (memcmp(label, &cms_id, sizeof(cms_id)) == 0) {
			cmsfs->blksize = blocksizes[i];
			break;
		}
	}

	if (!cmsfs->blksize)
		DIE("Error detecting blocksize from file!\n");

	/*
	 * Hardcoded since the label doesn't contain that info.
	 * Checking the disk identifier must be sufficient.
	 */
	cmsfs->format = DASD_FORMAT_LDL;
}

int get_device_info(struct cmsfs *cmsfs)
{
	struct stat stat;
	int fd;

	/*
	 * Open writable, if write access is not granted fall back to
	 * read only.
	 */
	fd = open(cmsfs->device, O_RDWR);
	if (fd < 0) {
		if (errno == EROFS || errno == EACCES) {
			cmsfs->readonly = 1;
			fd = open(cmsfs->device, O_RDONLY);
			if (fd < 0)
				DIE_PERROR("open failed");
		} else
			DIE_PERROR("open failed");
	}

	if (fstat(fd, &stat) < 0)
		DIE_PERROR("fstat failed");

	if (S_ISBLK(stat.st_mode))
		get_device_info_bdev(fd, cmsfs);
	else if (S_ISREG(stat.st_mode))
		get_device_info_file(fd, cmsfs);
	else
		goto error;

	if (!disk_supported(fd, cmsfs))
		goto error;
	return fd;

error:
	DIE("Unsupported disk\n");
}
