//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [rat_ecl] = CL_gm_eclipseCheck(pos_obs, pos1, pos2, sr1, sr2, output)
// Hidden fraction of a body (eclipsed by another body), as seen from an observer
//
// Calling Sequence
// [rat_ecl] = CL_gm_eclipseCheck(pos_obs, pos1, pos2, sr1, sr2)
//
// Description
// <itemizedlist><listitem>
// <p>Computes the hidden fraction of a body (eclipsed by another body), as seen from an observer:</p>
// <p>- 1.0 : 100% of the first body is eclipsed </p>
// <p>- 0.7 : 70% of the first body is eclipsed </p>
// <p>- 0.0 : 0% of the first body is eclipsed </p>
// <p></p>
// <p>The two bodies are assumed spherical</p>
// <p><inlinemediaobject><imageobject><imagedata fileref="eclipse_check.gif"/></imageobject></inlinemediaobject></p>
// </listitem>
// </itemizedlist>
//
// Parameters
// pos_obs: Position of observer (3xN or 3x1)
// pos1: Position of centre of sphere1 (3xN or 3x1)
// pos2: Position of centre of sphere2 (3xN or 3x1)
// sr1: Radius of sphere1 (1x1 or 1xN)
// sr2: Radius of sphere2 (1x1 or 1xN)
// rat_ecl: Hidden fraction of body1, as seen from pos_obs (3xN)
//
// Authors
// CNES - DCT/SB
//
// See also
//
// Examples
// // Fraction of Sun eclipsed by Earth (from an observer in space):
// theta = linspace(165*%CL_deg2rad, 175*%CL_deg2rad, 100); 
// pos_obs = 42000.e3 * [ cos(theta) ; sin(theta) ; zeros(theta) ];
// pos1 = %CL_au * [ 1 ; 0 ; 0 ]; // Sun
// sr1 = %CL_radiusSun;
// pos2 = [0 ; 0 ; 0]; // Earth
// sr2 = %CL_eqRad;
// rat_ecl = CL_gm_eclipseCheck(pos_obs, pos1, pos2, sr1, sr2); 
// scf();
//  plot(theta*%CL_rad2deg, rat_ecl, "o"); 

// Declarations:

function ang = ang_inter_2_cones(alpha1,alpha2,alpha)
// Solid angle of the intersection of two cones
//
// alpha1: apparent radius of cone 1
// alpha2: apparent radius of cone 2
// alpha: angular separation between axes of the cones
// ang: Solid angle of the intersection of the cones
//

  I = find(alpha1 < 0 | alpha2 < 0 | alpha < 0);
  if (I ~= []) ; CL__error("negative angles"); end;

  ang = %nan * ones(alpha);
    
  // Sort alpha1, alpha2 so that alpha1 < alpha2
  alpha_1 = min(alpha1, alpha2);
  alpha2 = max(alpha1, alpha2);
  alpha1 = alpha_1;

  ca1 = cos(alpha1);
  ca2 = cos(alpha2);

  // Angular separation between axes of the cones such that smaller cone is included in bigger cone
  I = find(alpha <= alpha2 - alpha1);
  ang(I) = 2*%pi * (1 - ca1(I)); // intersection = smaller cone
    
  // Angular separation such that there is no intersection
  I = find(alpha >= alpha1 + alpha2)
  ang(I) = 0;
  
  // Standard case
  I = find(alpha < alpha1 + alpha2 & alpha > alpha2 - alpha1);
    
  if (I <> [])
    X   = (alpha1(I) + alpha2(I) + alpha(I)) / 2; 
    sina1 = sin(X - alpha1(I)); 
    sina2 = sin(X - alpha2(I)); 
    sina  = sin(X - alpha(I)); 
    R   = sqrt(sina1 .* sina2 .* sina ./ sin(X)); 

    beta1 = 2 * atan(R, sina2); 
    beta2 = 2 * atan(R, sina1); 
    betaa = 2 * atan(R, sina); 

    // Solid angle of a cone is 2*pi*(1-cos(alpha1)).
    // Solid angle of a "portion" of a cone is beta1*(1-cos(alpha1))
    // Solid angle of a triangle of sides alpha,alpha1,alpha2 is: betaa+beta1+beta2-pi
    // So the solid angle of intersection is:

    ang(I) = 2 * (beta1 .* (1 - ca1(I)) + beta2 .* (1 - ca2(I)) - (betaa + beta1 + beta2 - %pi)); 
  end

endfunction
  
// -------------------------------------------------------------

// Code :
  
[lhs,rhs]=argn();
  
[pos_obs, pos1, pos2, sr1, sr2, N] = CL__checkInputs(pos_obs,3, pos1,3, pos2,3, sr1,1 ,sr2,1);

// Check that radii are positive
I = find(sr1 < 0 | sr2 < 0);
if (I~=[]); CL__error("Radius of spheres must be positive"); end;
  
// Check that observer is outside both spheres :
norm1 = CL_norm(pos_obs-pos1);
norm2 = CL_norm(pos_obs-pos2);
I = find(norm1 < sr1 | norm2 < sr2);
if (I~=[]); CL__error("Observer must be outside both spheres"); end;
  
// -------------------------------------------------------------

alpha1 = asin( sr1 ./ norm1 ); // apparent radius of sphere1
alpha2 = asin( sr2 ./ norm2 ); // apparent radius of sphere2

// separation angle between the two spheres (as seen from obs)
alpha = CL_vectAngle( pos1 - pos_obs , pos2 - pos_obs ); 
  
// Solid angle of body1 occulted by body2
ang_inters = ang_inter_2_cones(alpha1,alpha2,alpha);
  
// If body2 was behind body1 : body1 is not occulted
I = find( norm2 > norm1 );
ang_inters(I) = 0;
  
// Solid angle of body1 : 
ang_tot = 2*%pi * (1 - cos(alpha1));
 
// ratio of body1 eclipsed
rat_ecl = %nan * ones(ang_inters); 
I = find(ang_tot > 0); 
rat_ecl(I) = ang_inters(I) ./ ang_tot(I);


endfunction






