//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [d_ecc_dt,d_pom_dt] = CL_op_frozenOrbitDer(sma,ecc,inc,pom,er,mu,zonals)
// Derivatives of eccentricity and argument of periapsis with respect to time
//
// Calling Sequence
// [d_ecc_dt,d_pom_dt] = CL_op_frozenOrbitDer(sma,ecc,inc,pom, [er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// <p>Computes the derivatives of eccentricity and argument of periapsis 
// with respect to time that correspond to the gravitational effects due to J2 and J3.</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// sma: Semi-major axis [m] (1x1 or 1xN)
// ecc: Eccentricity (1x1 or 1xN)
// inc: Inclination [rad] (1x1 or 1xN)
// pom: Argument of pariapsis [rad] (1x1 or 1xN)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonal coefficients J1 to Jn, troncated to J3. Default is %CL_j1jn(1:3)). (1xNz)
// d_ecc_dt: Derivative of eccentricity [s^-1] (1xN)
// d_pom_dt: Derivative of argument of periapsis [rad/s] (1xN)
//
// Authors
// CNES - DCT/SB
// 
// Bibliography
// 1) "Frozen orbits in the J2+J3 problem", Krystyna Kiedron and Richard Cook, AAS 91-426. 
//
// Examples
// sma = [%CL_eqRad+350.e3 %CL_eqRad+700.e3];
// inc = CL_deg2rad([51.6 91.6]);
// [ecc,pom] = CL_op_frozenOrbit(sma,inc);
// ecc = 0.999*ecc;
// pom = 0.999*pom;
// [d_ecc_dt,d_pom_dt] = CL_op_frozenOrbitDer(sma,ecc,inc,pom)


	function [d_ecc_dt,d_pom_dt] = CL_op_perGelD(a,ecc,xi,pom,er,mu,j2,j3)

	// de/dt and d(pom)/dt due to J2 and J3

	// Declarations:

	// Code:

	sinxi = sin(xi)
	cosxi = cos(xi)
	cospom = cos(pom)
	sinpom = sin(pom)

	xn = CL_kp_params('mm',a,mu) // mean motion

	// calculation of eccentricity derivative
	exp1 = 1.5*xn .* (er./a).^3 .* j3 ./ (1-ecc.^2).^2
	exp2 = sinxi .* cospom .* (1.25.*(sinxi).^2 - 1)
	d_ecc_dt = exp1.*exp2

	// calculation of periapsis argument derivative
	exp11 = -0.75*xn .* (er./a).^2 .* j2 ./ (1-ecc.^2).^2 .* (1-5*cosxi.^2)
	exp22 = -1.5*xn .* (er./a).^3 .* j3 ./ (ecc.*(1-ecc.^2).^3) .* sinpom./sinxi
	exp3 = (1.25*sinxi.^2 - 1) .* sinxi.^2
	exp4 = ecc.^2 .* (1-8.75*sinxi.^2 .* cosxi.^2)
	d_pom_dt = exp11 + exp22.*(exp3+exp4)

	endfunction


// Declarations:
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_mu')) then global %CL_mu; end;
if(~exists('%CL_j1jn')) then global %CL_j1jn; end;


// Code:

Nsma = size(sma, 2);  
Ninc = size(inc, 2);  
Necc = size(ecc, 2);  
Npom = size(pom, 2);  
N = max(Nsma, Ninc, Necc, Npom); 

coherence = (Nsma==N|Nsma==1) & (Ninc==N|Ninc==1) & (Necc==N|Necc==1) & (Npom==N|Npom==1); 
if (~coherence); 
  CL__error('Invalid sizes of input arguments'); 
end

if (N <> 1)
  if (Nsma == 1); sma = sma * ones(1,N); end
  if (Ninc == 1); inc = inc * ones(1,N); end
  if (Necc == 1); ecc = ecc * ones(1,N); end
  if (Npom == 1); pom = pom * ones(1,N); end
end

if ~exists('er','local'); er = %CL_eqRad; end
if ~exists('mu','local'); mu = %CL_mu; end
if ~exists('zonals','local'); zonals = %CL_j1jn(1:3); end


if (length(zonals) < 3)
  CL__error('zonals must be a vector of size 3 (or more)');
end

[d_ecc_dt,d_pom_dt] = CL_op_perGelD(sma,ecc,inc,pom,er,mu,zonals(2),zonals(3))


endfunction
