/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Written (W) 1999-2008 Soeren Sonnenburg
 * Written (W) 1999-2008 Gunnar Raetsch
 * Copyright (C) 1999-2008 Fraunhofer Institute FIRST and Max-Planck-Society
 */

#ifndef _WEIGHTEDCOMMWORDSTRINGKERNEL_H___
#define _WEIGHTEDCOMMWORDSTRINGKERNEL_H___

#include "lib/common.h"
#include "lib/Mathematics.h"
#include "kernel/CommWordStringKernel.h"

class CCommWordStringKernel;

/** kernel WeightedCommWordString */
class CWeightedCommWordStringKernel: public CCommWordStringKernel
{
	public:
		/** constructor
		 *
		 * @param size cache size
		 * @param use_sign if sign shall be used
		 * @param normalization_ type of normalization
		 */
		CWeightedCommWordStringKernel(INT size, bool use_sign,
			ENormalizationType normalization_=FULL_NORMALIZATION);

		/** constructor
		 *
		 * @param l features of left-hand side
		 * @param r features of right-hand side
		 * @param use_sign if sign shall be used
		 * @param normalization_ type of normalization
		 * @param size cache size
		 */
		CWeightedCommWordStringKernel(
			CStringFeatures<WORD>* l, CStringFeatures<WORD>* r,
			bool use_sign=false,
			ENormalizationType normalization_=FULL_NORMALIZATION,
			INT size=10);

		virtual ~CWeightedCommWordStringKernel();

		/** initialize kernel
		 *
		 * @param l features of left-hand side
		 * @param r features of right-hand side
		 * @return if initializing was successful
		 */
		virtual bool init(CFeatures* l, CFeatures* r);

		/** clean up kernel */
		virtual void cleanup();

		/** compute optimized
	 	*
	 	* @param idx index to compute
	 	* @return optimized value at given index
	 	*/
		virtual DREAL compute_optimized(INT idx);

		/** add to normal
		 *
		 * @param idx where to add
		 * @param weight what to add
		 */
		virtual void add_to_normal(INT idx, DREAL weight);

		/** merge normal */
		void merge_normal();

		/** set weighted degree weights
		 *
		 * @return if setting was successful
		 */
		bool set_wd_weights();

		/** set custom weights (swig compatible)
		 *
		 * @param w weights
		 * @param d degree (must match number of weights)
		 * @return if setting was successful
		 */
		bool set_weights(DREAL* w, INT d);

		/** return what type of kernel we are
		 *
		 * @return kernel type WEIGHTEDCOMMWORDSTRING
		 */
		virtual EKernelType get_kernel_type() { return K_WEIGHTEDCOMMWORDSTRING; }

		/** return the kernel's name
		 *
		 * @return name WeightedCommWordString
		 */
		virtual const CHAR* get_name() { return "WeightedCommWordString"; }

		/** return feature type the kernel can deal with
		 *
		 * @return feature type WORD
		 */
		inline virtual EFeatureType get_feature_type() { return F_WORD; }

		/** compute scoring
		 *
		 * @param max_degree maximum degree
		 * @param num_feat number of features
		 * @param num_sym number of symbols
		 * @param target target
		 * @param num_suppvec number of support vectors
		 * @param IDX IDX
		 * @param alphas alphas
		 * @param do_init if initialization shall be performed
		 * @return computed score
		 */
		virtual DREAL* compute_scoring(INT max_degree, INT& num_feat,
			INT& num_sym, DREAL* target, INT num_suppvec, INT* IDX,
			DREAL* alphas, bool do_init=true);

	protected:
		/** helper for compute
		 *
		 * @param idx_a index a
		 * @param idx_b index b
		 * @param do_sort if sorting shall be performed
		 */
		virtual DREAL compute_helper(INT idx_a, INT idx_b, bool do_sort);

		/** degree */
		INT degree;

		/** weights for each of the subkernels of degree 1...d */
		DREAL* weights;
};

#endif /* _WEIGHTEDCOMMWORDSTRINGKERNEL_H__ */
