/////////////////////////////////////////////////////////////////////////////
// File:        pin.h
// Author:      Cesar Mauri Loba (cesar at crea-si dot com)
// Copyright:   (C) 2010 Cesar Mauri Loba - CREA Software Systems
// 
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
/////////////////////////////////////////////////////////////////////////////
#ifndef SPCORE_PIN_H
#define SPCORE_PIN_H

#include "include/spcore/baseobj.h"
#include <assert.h>

namespace spcore {

class CTypeAny;

///**
//Interface for input pins
//This interface declares operations that should be available to work 
//with pins but not to implement it. Use CInputPin to create your 
//specific (i.e. typed) pin implementation.
//*/
class IInputPin : public IBaseObject {
protected:
	virtual ~IInputPin() {}

public:
    // Return the internal type ID for the type name of this pin, 
    // which is execution dependent, or  TYPE_ANY if the type name doesn't exists
    virtual int GetTypeID() const = 0;

    //Return the name of the pin
    virtual const char* GetName() const = 0;

    enum EIPinProperties {
      ALLOW_WRITE = 1,
      ALLOW_READ = 2

    };

    ///**
    // Get the properties of the pin.
    //
    // \return a combination of flags
    // \return ALLOW_WRITE: calls to Send method allowed
    // \return ALLOW_READ: calls to Read method allowed
    //*/
    virtual unsigned int GetProperties() const = 0;

    ///**
    //Send a message to the pin. Intended only for calls outside the graph.
    //\return int: 0 -> message successfully sent, -1 -> type mismatch
    //*/
    virtual int Send(SmartPtr<const CTypeAny> message) = 0;

	/**
		Read the content of the pin. Always returns a copy of the content
		of a NULL pointer
	*/
    virtual SmartPtr<const CTypeAny> Read() const = 0;

	/**
		Rename the pin.

		\param name, new name of the pin
	*/
	virtual void Rename (const char * name) = 0;

	/**
		Allow to change the type of the pin. The only change allowed
		is to turn an any pin into a typed one. The change
		might fail if the pin is already connected.

		\param type_name, string with the new of the new type
		\return  0 -> type changed correctly
				-1 -> type cannot be changed
				-2 -> type name does not exists
	*/
	virtual int ChangeType (const char * type_name) = 0;
};

/**
//Interface for output pins
//This interface declares operations that should be available to work 
//with pins but not to implement it. Use COutputPin to create your 
//specific (i.e. typed pin implementation)
*/

class IOutputPin : public IBaseObject {
protected:
	virtual ~IOutputPin() {}


public:
	/*
    // Return the internal type ID for the type name of this pin, 
    // which is execution dependent, or  TYPE_ANY if the type name doesn't exists
	*/
    virtual int GetTypeID() const = 0;

	/*
		Name of the pin
		\return a pointer to a internal char array which might not be freed
	*/
    virtual const char* GetName() const = 0;

    ///**
    //Connects this with a consumer input pin. 
    //
    //\return 0 if all correct of <0 when error
    //*/
    virtual int Connect(IInputPin & consumer) = 0;

    ///**
    // Disconnects. If the two pins where not connected does nothing.
    //*/
    virtual void Disconnect(const IInputPin & consumer) = 0;

    // Checks whether two pins can be connected taking into account
    // the type of each one.
    virtual bool CanConnect(const IInputPin & dst) const = 0;

    ///**
    // Sends a message to the connected pins. To be used by the component implementation.
    // \return 0 if correct or <0 when error
    //*/
    virtual int Send(SmartPtr<const CTypeAny> message) = 0;

	/**
		Return the number of consumers currently connected
		\return number of consumers 
	*/
	virtual unsigned int GetNumComsumers() const = 0;

	/**
		Rename the pin.

		\param name, new name of the pin
	*/
	virtual void Rename (const char * name) = 0;

	/**
		Allow to change the type of the pin. The only change allowed
		is to turn an any pin into a typed one. The change
		might fail if the pin is already connected.

		\param type_name, string with the new of the new type
		\return  0 -> type changed correctly
				-1 -> type cannot be changed
				-2 -> type name does not exists
	*/
	virtual int ChangeType (const char * type_name) = 0;

};

} // namespace spcore
#endif
