#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include "list.h"
#include "common.h"
#include "buffer.h"

void InitBufferManager(BUFFERMANAGER *man, char *name, size_t data_size){
    DPRINT(10, "name=%s, data_size=%d\n", name, data_size);
    safecpy(man->name, name);
    man->count_max = 15;
    man->count = 0;
    man->data_size = data_size;
    init_list(&man->list);
/*    pthread_mutex_init(&man->mutex, NULL); */
}

void DestroyBufferManager(BUFFERMANAGER *man){
    DPRINT(10, "name=%s\n", man->name);
    DestroyBufferList(man);
/*    pthread_mutex_destroy(&man->mutex); */
}

int SetMaxBufferCount(BUFFERMANAGER *man, int count){
    DPRINT(7, "name=%s, count=%d\n", man->name, count);
    if (count < 4) return 0;
    pthread_mutex_lock(&man->mutex);
    man->count_max = count;
    pthread_mutex_unlock(&man->mutex);
    return 1;
}

BUFFER* NewBuffer(BUFFERMANAGER *man){
    BUFFER *buf;

    if (man->count >= man->count_max) return NULL;
    if ((buf = malloc(sizeof(BUFFER) + man->data_size)) == NULL) return NULL;
    memset(buf, 0, sizeof(BUFFER) + man->data_size);
    buf->man = man;
    add_to_list(&man->list, buf);
    man->count++;
    DPRINT(10, "name=%s, count=%d, buffer=%p\n", man->name, man->count, buf);
    return buf;
}

void DeleteBuffer(BUFFER *buf){
    DPRINT(10, "name=%s, count=%d, buffer=%p\n", buf->man->name, buf->man->count, buf);
    remove_from_list(&buf->man->list, buf);
    buf->man->count--;
    free(buf);
    return;
}

void DestroyBufferList(BUFFERMANAGER *man){
    pthread_mutex_lock(&man->mutex);
    while(!is_list_empty(&man->list))
	DeleteBuffer(first_elem(&man->list));
    DPRINT(1, "%s buffers still in use: %d\n", man->name, man->count);
    pthread_mutex_unlock(&man->mutex);
}

BUFFER* GetBuffer(BUFFERMANAGER *man){
    BUFFER	*buf;

    pthread_mutex_lock(&man->mutex);
    if (is_list_empty(&man->list)) buf = NewBuffer(man);
    else buf = first_elem(&man->list);
    if (buf != NULL) remove_from_list(&man->list, buf);
    pthread_mutex_unlock(&man->mutex);
    DPRINT(10, "name=%s, buffer=%p\n", man->name, buf);
    return buf;
}

void ReleaseBuffer(BUFFER *buf){
    BUFFERMANAGER	*man;

    man = buf->man;
    DPRINT(10, "name=%s, buffer=%p\n", man->name, buf);
    pthread_mutex_lock(&man->mutex);
    add_to_list(&man->list, buf);
    if (man->count > man->count_max) DeleteBuffer(buf);
    pthread_mutex_unlock(&man->mutex);
    return;
}

void AllocateBuffers(BUFFERMANAGER *man){
    pthread_mutex_lock(&man->mutex);
    while(man->count < man->count_max)
	if (NewBuffer(man) == NULL) break;
    pthread_mutex_unlock(&man->mutex);
}

size_t BufferSize(BUFFER *buf){
    return buf->man->data_size;
}
