/*
 * ffmpeg2raw.c -- Convert ffmpeg support a/v files to Raw DV
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <string.h>

#include <ffmpeg/common.h>
#include <ffmpeg/avformat.h>

#include "dvutils.h"

#undef printf
#undef fprintf

/** The ffmpeg2raw conversion process works as follows:

	* open the specified file and obtain decoders for a 
	  video and audio stream
	* determine the frame multiplication factor (for PAL
	  this is 25/fps and for NTSC, 29.97/fps)
	* for each packet received, determine if it's video or
	  audio
		* when a video frame is obtained, convert to RGB24
		  and pass to the dvframe queue
		* when an audio packet is obtained, decode and pass
		  all the data to the audio queue - this is then 
		  split into DV audio frame sizes and with each
		  frame added the flush is called
			* the flush determines that at least one video 
			  frame is available and the requisite number of
			  audio frames for that frame are present (using
			  the multiplication factor evaluated above). The
			  flush correctly determines when video frames 
			  should be dropped or duplicated and encodes to 
			  dv if necessary.
	* flush and close on stream finish
*/

/** Allocate and initialise an AVFrame.
*/

AVFrame *alloc_picture(int pix_fmt, int width, int height)
{
    AVFrame *picture;
    uint8_t *picture_buf;
    int size;
    
    picture = avcodec_alloc_frame();
    if (!picture)
        return NULL;
    size = avpicture_get_size(pix_fmt, width, height);
    picture_buf = av_malloc(size);
    if (!picture_buf) 
	{
        av_free(picture);
        return NULL;
    }
    avpicture_fill((AVPicture *)picture, picture_buf, 
                   pix_fmt, width, height);
    return picture;
}

typedef struct ff2raw
{
	AVFormatContext *context;
	int video_index;
	int audio_index;
	int frequency;
	int channels;
	int scale;
	int wide;
	int pal;
	int twopass;
	int preview;
	double start;
	double end;
	int ppm_output;
	int audio_output;
	char *filter;
	int scaler;
	int every;
	int nodupe;
	int disable_audio;
	double fps;
}
*ff2raw;

ff2raw ff2raw_init( )
{
	ff2raw this = calloc( 1, sizeof( *this ) );
	if ( this != NULL )
	{
		this->pal = 1;
		this->video_index = -1;
		this->audio_index = -1;
		this->frequency = 48000;
		this->channels = 2;
		this->start = 0;
		this->end = -1;
		this->scaler = 4;
		this->every = 1;
	}
	return this;
}

void ff2raw_output( ff2raw this )
{
	int i;
	AVCodecContext *aenc = NULL;
	AVCodecContext *venc = NULL; 
	AVStream *astream = NULL;
	AVStream *vstream = NULL;
	AVCodec *acodec = NULL;
	AVCodec *vcodec = NULL;
	ReSampleContext *resample = NULL;
	double fps = 0.0;

    for( i = 0; i < this->context->nb_streams; i++ ) 
	{
        AVCodecContext *enc = &this->context->streams[ i ]->codec;
        switch( enc->codec_type ) 
		{
        	case CODEC_TYPE_VIDEO:
            	if ( this->video_index < 0 )
                	this->video_index = i;
            	break;
        	case CODEC_TYPE_AUDIO:
            	if ( this->audio_index < 0 && !this->disable_audio )
                	this->audio_index = i;
            	break;
        	default:
            	break;
        }
    }

	if ( this->video_index >= 0 )
	{
		vstream = this->context->streams[ this->video_index ];
		venc = &this->context->streams[ this->video_index ]->codec;
		vcodec = avcodec_find_decoder( venc->codec_id );
		fps = (double) venc->time_base.den / (double) venc->time_base.num;
		if ( fps > 10000 ) fps /= 1000;
		if ( vcodec == NULL || avcodec_open( venc, vcodec ) < 0 )
			this->video_index = -1;
		if ( this->fps != 0.0 )
			fps = this->fps;
	}

	if ( this->audio_index >= 0 )
	{
		astream = this->context->streams[ this->audio_index ];
 		aenc = &this->context->streams[ this->audio_index ]->codec;
		acodec = avcodec_find_decoder( aenc->codec_id );
		if ( this->channels != aenc->channels && aenc->codec_id == CODEC_ID_AC3)
			aenc->channels = this->channels;
		if ( acodec != NULL && avcodec_open( aenc, acodec ) >= 0 )
			resample = audio_resample_init( this->channels, aenc->channels, 
											this->frequency, aenc->sample_rate);
		else
			this->audio_index = -1;
	}

    if ( this->video_index >= 0 ) 
	{
		AVPacket pkt;
		int ret;
		AVFrame real_frame;
		AVFrame *frame = &real_frame;
		AVFrame *output = alloc_picture( PIX_FMT_RGB24, vstream->codec->width, vstream->codec->height );
		int len;
		int len1;
		int got_picture;
		uint8_t *ptr;
		double multiplier = ( this->pal ? 25.00 : 29.97 ) / fps;
		int16_t audio_buf[(AVCODEC_MAX_AUDIO_FRAME_SIZE * 4) ]; 
		int16_t resampled[(AVCODEC_MAX_AUDIO_FRAME_SIZE * 4) ]; 
		int first = 1;

		dvframes_info info;
		info.pal = this->pal;
		info.wide = this->wide;
		info.frequency = this->frequency;
		info.scale = this->scale;
		info.twopass = this->twopass;
		info.preview = this->preview;
		info.start = this->start;
		info.end = this->end;
		info.ppm_output = this->ppm_output;
		info.audio_output = this->audio_output;
		info.filter = this->filter;
		info.scaler = this->scaler;
		info.every = this->every;
		info.width = venc->width;
		info.height = venc->height;

		dvframes_init( info, multiplier );

		do
		{
			ret = av_read_frame( this->context, &pkt );

			ptr = pkt.data;
			len = pkt.size;

			if ( ret >=0 && pkt.stream_index == this->video_index )
			{
				if ( len == 0 && this->nodupe == 0 && !first )
				{
					if ( dvframes_add_video( NULL, venc->width, venc->height, 0 ) )
					{
						ret = -1;
						fprintf( stderr, "No dv frames available\n" );
						break;
					}
				}

				while ( len > 0 )
				{
					len1 = avcodec_decode_video( &vstream->codec, frame, &got_picture, ptr, len);
					if ( len1 < 0 )
						break;
					if ( got_picture )
					{
						//double video_pts = (double)pkt.pts * this->context->pts_num / this->context->pts_den;
						//if ( video_pts > 0 )
							//fprintf( stderr, "video_pts = %f\n", video_pts );

        				img_convert((AVPicture *)output, PIX_FMT_RGB24, (AVPicture *)frame, 
									venc->pix_fmt, venc->width, venc->height);

						first = 0;

						if ( dvframes_add_video( output->data[ 0 ], venc->width, venc->height, output->linesize[ 0 ] ) )
						{
							ret = -1;
							fprintf( stderr, "No dv frames available\n" );
							break;
						}
					}
					ptr += len1;
					len -= len1;
				}
			}
			else if ( ret >=0 && pkt.stream_index == this->audio_index )
			{
				int data_size;

				while ( len > 0 )
				{
            		len1 = avcodec_decode_audio(&astream->codec, audio_buf, &data_size, ptr, len );
					len -= len1;
					ptr += len1;
					if ( data_size > 0 )
					{
        				int size_out = audio_resample( resample, resampled, audio_buf, 
													   data_size / (aenc->channels * 2));
						//double audio_pts = (double)pkt.pts * this->context->pts_num / this->context->pts_den;
						//if ( audio_pts > 0 )
							//fprintf( stderr, "audio_pts = %f\n", audio_pts );

						if ( dvframes_add_audio( resampled, size_out ) )
						{
							ret = -1;
							fprintf( stderr, "No audio frames available\n" );
							break;
						}
					}
				}
			}

			dvframes_flush( this->audio_index == -1 );
			av_free_packet( &pkt );
		}
		while ( ret >= 0 );

		dvframes_close( );
    }
	else
	{
		fprintf( stderr, "No video or audio stream found\n" );
	}
}

void ff2raw_close( ff2raw this )
{
	av_free( this );
}

void show_stats( AVFormatContext *ic, const char *url, int pal )
{
    int i;

    printf( "T: ");
    if (ic->duration != AV_NOPTS_VALUE) 
	{
        int hours, mins, secs, us;
        secs = ic->duration / AV_TIME_BASE;
        us = ic->duration % AV_TIME_BASE;
        mins = secs / 60;
        secs %= 60;
        hours = mins / 60;
        mins %= 60;
        printf("%02d:%02d:%02d.%01d\n", hours, mins, secs, (10 * us) / AV_TIME_BASE);
    } 
	else 
	{
        printf("N/A\n");
    }

    printf( "F: ");
    if (ic->duration != AV_NOPTS_VALUE) 
	{
        double frames = ( ( ic->duration / AV_TIME_BASE ) * ( pal ? 25 : 29.97 ) ) ; 
        printf("%.0f\n", frames );
    } 
	else 
	{
        printf("N/A\n");
    }

    for(i=0;i<ic->nb_streams;i++) 
	{
        AVStream *st = ic->streams[i];
		AVCodecContext *enc = &st->codec;
		AVCodec *p = avcodec_find_decoder(enc->codec_id);
		if ( p )
        	printf("%c%d: %s\n", enc->codec_type == CODEC_TYPE_VIDEO ? 'V' : 'A', i, p->name );
    }
}

void print_usage( )
{
	fprintf( stderr, "ffmpeg2raw [ options ] input\n"
					 "Where options are:\n" 
					 "      -n         - NTSC output (default: PAL)\n"
					 "      -ppm       - output unscaled ppm instead of raw dv (no audio)\n" 
					 "      -a         - scale image and maintain aspect ratio\n"
					 "      -s         - scale image and ignore aspect ratio\n"
					 "      -w         - wide screen output (default: off)\n"
					 "      -2         - 2 pass encoding (default: off)\n"
					 "      -f freq    - audio frequency (default: 48khz)\n" 
					 "      -at track  - select audio track (default: first found)\n" 
					 "      -vt track  - select video track (default: first found)\n" 
					 "      -o offset  - starting offset in seconds (default: 0)\n" 
					 "      -e offset  - ending offset in seconds (default: all)\n" 
					 "      -every n   - encode every nth frame (default: 1)\n" 
					 "      -scaler q  - scaler quality [0 - 4] (default: 4 [highest])\n" 
					 "      -vhook cmd - video hook\n" 
					 "      -an        - disable audio\n"
					 "      -r fps     - override the input frame rate detection\n" 
					 );

	exit( 0 );
}

int main( int argc, char **argv )
{
	int i;
	ff2raw convert = ff2raw_init( );
	int stats = 0;

	av_register_all( );

	if ( argc == 1 )
		print_usage( );

	for ( i = 1; i < argc; i ++ )
	{
		if ( !strcmp( argv[ i ], "--help" ) )
			print_usage( );
		else if ( !strcmp( argv[ i ], "--stats" ) )
			stats = 1;
		else if ( !strcmp( argv[ i ], "-n" ) )
			convert->pal = 0;
		else if ( !strcmp( argv[ i ], "-2" ) )
			convert->twopass = 1;
		else if ( !strcmp( argv[ i ], "-p" ) )
			convert->preview = 1;
		else if ( !strcmp( argv[ i ], "-a" ) )
			convert->scale = 1;
		else if ( !strcmp( argv[ i ], "-s" ) )
			convert->scale = 2;
		else if ( !strcmp( argv[ i ], "-w" ) )
			convert->wide = 1;
		else if ( !strcmp( argv[ i ], "-raw" ) )
			convert->ppm_output = convert->audio_output = 1;
		else if ( !strcmp( argv[ i ], "-ppm" ) )
			convert->ppm_output = 1;
		else if ( !strcmp( argv[ i ], "-audio" ) )
			convert->audio_output = 1;
		else if ( !strcmp( argv[ i ], "-at" ) )
			convert->audio_index = atoi( argv[ ++ i ] );
		else if ( !strcmp( argv[ i ], "-vt" ) )
			convert->video_index = atoi( argv[ ++ i ] );
		else if ( !strcmp( argv[ i ], "-f" ) )
			convert->frequency = atoi( argv[ ++ i ] );
		else if ( !strcmp( argv[ i ], "-o" ) )
			convert->start = atof( argv[ ++ i ] );
		else if ( !strcmp( argv[ i ], "-e" ) )
			convert->end = atof( argv[ ++ i ] );
		else if ( !strcmp( argv[ i ], "-vhook" ) )
			convert->filter = argv[ ++ i ];
		else if ( !strcmp( argv[ i ], "-scaler" ) )
			convert->scaler = atoi( argv[ ++ i ] );
		else if ( !strcmp( argv[ i ], "-every" ) )
			convert->every = atoi( argv[ ++ i ] );
		else if ( !strcmp( argv[ i ], "-nodupe" ) )
			convert->nodupe = 1;
		else if ( !strcmp( argv[ i ], "-an" ) )
			convert->disable_audio = 1;
		else if ( !strcmp( argv[ i ], "-r" ) )
			convert->fps = atof( argv[ ++ i ] );
		else if ( av_open_input_file( &convert->context, argv[ i ], NULL, 0, NULL ) >= 0 )
		{
			if ( av_find_stream_info( convert->context ) >= 0 )
			{
				if ( !stats )
				{
					dump_format( convert->context, 0, argv[ i ], 0 );
					ff2raw_output( convert );
					convert->audio_index = convert->video_index = -1;
				}
				else
				{
					show_stats( convert->context, argv[ i ], convert->pal );
				}
			}
			else
			{
				fprintf( stderr, "Unable to obtain stream info\n" );
			}
			av_close_input_file( convert->context );
		}
		else
		{
			fprintf( stderr, "Unable to open file %s\n", argv[ i ] );
		}
	}

	ff2raw_close( convert );

	return 0;
}

