/*
 * AudioWav.h -- Audio IO via Wav
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef _KINO_AUDIO_WAV_
#define _KINO_AUDIO_WAV_

#include <stdio.h>
#include "BufferReader.h"
#include "BufferWriter.h"
#include "AudioImporter.h"
#include "AudioExtractor.h"
#include <frame.h>
#include "Threader.h"

typedef struct WavHeader
{
	char riff[ 4 ];
	uint32_t riff_length;
	char type[ 4 ];
	char format[ 4 ];
	uint32_t format_length;
	int16_t filler;
	int16_t channels;
	uint32_t rate;
	uint32_t bytespersecond;
	int16_t bytespersample;
	int16_t bitspersample;
	char data[ 4 ];
	uint32_t data_length;
}
WAVHeaderType, *WAVHeader;

class WavData :
	public AudioImporter,
	protected BufferReader, 
	protected BufferWriter
{
	private:
		struct WavHeader header;
		int16_t temp[ 20480 ];
		
	public:
		WavData( );
		void SetInfo( int16_t channels, int rate, int bytespersample );
		int GetLength( );
		int GetChannels( );
		int GetFrequency( );
		int GetBytesPerSample( );
		int ReadHeader( );
		int WriteHeader( );
		bool IsWav( );
		bool Get( int16_t **data, int length );
		bool Set( int16_t *data, int length );
	
		virtual bool Open( string input ) { return false; }
		virtual int Read( uint8_t *data, int size ) { return 0; }
		virtual int Write( uint8_t *data, int size ) { return 0; }
		virtual bool Seek( long offset ) { return false; }
		virtual bool Close( ) { return false; }
};

class WavImporter : 
	public WavData
{
	private:
		int fd;
	public:
		virtual bool Open( string input );
		virtual int Read( uint8_t *data, int size );
		virtual bool Seek( long offset );
		virtual bool Close( );
};

class WavThreadedReader :
	public WavData,
	private Threader
{
	private:
		int fd;
	public:
		virtual bool Open( string input );
		virtual int Read( uint8_t *data, int size );
		virtual bool Seek( long offset );
		virtual bool Close( );
		void Thread( );
};
	
class WavExporter :
	public AudioExtractor,
	public WavData
{
	private:
		int fd;
		AudioResample *resampler;
	
	public:
		WavExporter( string file );
		int Write( uint8_t *data, int size );
		bool Initialise( Frame & );
		bool Output( Frame & );
		bool Flush( );
};

class Mp2Exporter :
	public AudioExtractor,
	public WavData
{
	private:
		string filename;
		FILE *file;
		int fd;
		AudioResample *resampler;
	
	public:
		Mp2Exporter( string file );
		int Write( uint8_t *data, int size );
		bool Initialise( Frame & );
		bool Output( Frame & );
		bool Flush( );
};

#endif
