#===================================
#          stackapplet
#  Copyright 2010 - Nathan Osman
#
#     Contains the main class
#      used by StackApplet
#
#   StackApplet is released under
#        the MIT license
#===================================

# The following modules are required
# for supporting translations
import locale
import gettext

# Bind the domain
gettext.bindtextdomain("stackapplet","/usr/share/locale")
gettext.textdomain("stackapplet")

# Figure out the system locale
(lang, enc) = locale.getdefaultlocale()

# Set it to a sensible default if we couldn't
# figure it out otherwise
if lang == None:
	lang = "en_US"

# Now create the translation object and
# register the translation function
tr = gettext.translation("stackapplet",
                         "/usr/share/locale",
                         [lang],
                         fallback=True)
_ = tr.gettext

# Global definitions
__application__ = _("StackApplet")
__authors__     = ["Nathan Osman", "Isaiah Heyer"]
__artists__     = ["Marco Ceppi"]
__translators__ = _("translator-credits")
__copyright__   = _("Copyright (c) 2010, Nathan Osman")
__license__     = _('''Copyright (c) 2010 Nathan Osman

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.''')
__version__     = "1.4"

# modules required for using GTK
import pygtk
pygtk.require('2.0')
import gtk
import gobject

# modules required for OS level stuff
import os
import urllib2
import time

# modules required for using AppIndicators
# keep in mind that we may need to fallback to
# our own replacement if the original module
# isn't found
try:
	import appindicator
except ImportError:
	import appindicator_replacement as appindicator

# module that we use for preferences etc.
import prefs_dialog

# module for accessing the StackExchange API
import stack_api

# module for loading / storing preferences
import config_store

# module for opening webpages	
import webbrowser

# if libnotify is available, we can
# alert the user through notifications
try:
    import pynotify
    CAN_DISPLAY_ALERTS = 1;
    
except ImportError:

    # otherwise, we can't display alerts
    CAN_DISPLAY_ALERTS = 0;

# This is the root class that contains
# all of the core logic for StackApplet
class stackapplet:

	# Our constructor
	def __init__ (self):
	
		# Call our initialization function
		self.reset_defaults()
		
		# Create an AppIndicator
		self.ind = appindicator.Indicator ("example-simple-client",
		                                   "stackapplet_grey",
		                                   appindicator.CATEGORY_APPLICATION_STATUS)
		
		self.ind.set_status (appindicator.STATUS_ACTIVE)
		self.ind.set_attention_icon("stackapplet")
		
		# Begin creating the menu
		self.create_menu()
	
	# Resets all of the variables pertaining
	# to registered accounts
	def reset_defaults(self):
		
		self.timer = None
		
		self.account_list = []
		self.total_notifications = 0
	
	# Creates the menu for the application
	def create_menu(self):
		
		# The root menu
		menu = gtk.Menu()
		
		menu_item_acc_header = gtk.MenuItem(_("Accounts"))
		menu.append(menu_item_acc_header)
		menu_item_acc_header.set_sensitive(False)
		menu_item_acc_header.show()
		
		num_added = self.add_accounts_to_menu(menu)
		
		if num_added:
			
			# Add a seperator
			separator = gtk.SeparatorMenuItem()
		
			separator.show()
			menu.append(separator)
		else:
			
			menu_item_acc_header.set_label(_("No Accounts"))
		
		# Add the other items
		menu_item_header = gtk.MenuItem(_("Tools"))
		
		# For these items, we can use the stock
		# GTK menuitems
		menu_item_ref    = gtk.ImageMenuItem(stock_id=gtk.STOCK_REFRESH)
		menu_item_prefs  = gtk.ImageMenuItem(stock_id=gtk.STOCK_PREFERENCES)
		menu_item_about  = gtk.ImageMenuItem(stock_id=gtk.STOCK_ABOUT)
		menu_item_quit   = gtk.ImageMenuItem(stock_id=gtk.STOCK_QUIT)
		
		# Add them all to the menu
		menu.append(menu_item_header)
		menu.append(menu_item_ref)
		menu.append(menu_item_prefs)
		menu.append(menu_item_about)
		menu.append(menu_item_quit)
		
		menu_item_header.set_sensitive(False)
		menu_item_ref  .connect("activate", self.menu_ref_activate)
		menu_item_prefs.connect("activate", self.menu_prefs_activate)
		menu_item_about.connect("activate", self.menu_about_activate)
		menu_item_quit .connect("activate", self.menu_quit_activate)
		
		menu_item_header.show()
		menu_item_ref.show()
		menu_item_prefs.show()
		menu_item_about.show()
		menu_item_quit.show()
		
		self.ind.set_menu(menu)
		
		# Now do the update
		self.update()
	
	def get_image(self,site):
		
		# Figure out the image's path
		img_fn = "/usr/share/pixmaps/" + site + ".png"
		
		# See if it exists there
		if not os.path.isfile(img_fn):
			
			# Now see if we've retrieved this one before
			img_path = os.path.join(os.getenv("HOME"), '.stackapplet')
			img_path = os.path.join(img_path, 'logos')
			img_fn = os.path.join(img_path, site + ".png")
			
			if not os.path.isfile(img_fn):
				
				# We need to download the file and
				# save it.
				img_path = os.path.join(config_store.CONFIG_FILE_PATH,"logos")
				
				# Make sure the path exists
				if not os.path.isdir(img_path):
					os.makedirs(img_path)
				
				# Now download the logos
				site_data = stack_api.api.fetch_stats(site)
				site_icon = site_data['statistics'][0]['site']['icon_url']
				
				# Now fetch the file
				request_data  = urllib2.urlopen(site_icon).read()
				
				new_img = os.path.join(img_path,site + ".png")
				
				f = open(new_img, "wb")
				f.write(request_data)
				f.close()
				
				return new_img
			
			return img_fn
		else:
			
			return img_fn
	
	# Fetch the list of sites / accounts now
	def add_accounts_to_menu(self,menu):
		
		# Start by loading the refresh rate
		prefs = config_store.config_store()
		prefs.load_settings()
		
		self.refresh_rate = prefs.get_val("refresh_rate",300000)
		
		# Get the icon to use
		if prefs.get_val("theme","dark") == 'dark':
			icon_to_use = 'stackapplet_grey'
		else:
			icon_to_use = 'stackapplet_light'
		
		self.ind.set_icon(icon_to_use)
		
		accounts = prefs.get_val("accounts",{})
		
		num_accounts_added = 0
		
		# Now we need to add them to the menu
		for account in accounts:
			
			# Create the menu
			'''item = gtk.ImageMenuItem()'''
			item = gtk.MenuItem()
			item.set_label(_("Please wait..."))
			
			'''img = gtk.Image()
			img_fn = self.get_image(account["site"])
			img.set_from_file(img_fn)
			img.show()
			item.set_image(img)'''
			
			item.show()
			
			menu.append(item)
			
			# We need to store this menu item
			# for easy access to later when the
			# item gets updated
			new_account = { "menu"              : item,
			                "site"              : account["site"],
			                "site_name"         : account["site_name"],
			                "user_id"           : account["user_id"],
			                "last_reputation"   : 0,
			                "last_comment_id"   : 0,
			                "last_timestamp"    : 0,
			                "unread_reputation" : 0,
			                "unread_comments"   : 0,
			                "unread_answers"    : 0,
			                "notification"      : 0 }
			
			self.account_list.append(new_account)
			
			item.connect("activate", self.menu_item_activate, new_account)
			
			num_accounts_added += 1
		
		return num_accounts_added
	
	def update(self):
	
		global CAN_DISPLAY_ALERTS
		
		# kill the current timer
		if(not self.timer == None):
			gobject.source_remove(self.timer)
		
		for account in self.account_list:
			
			# We need to wrap everything in a try block
			# in case there is network failure or some
			# other exception thrown by the URL handling
			# code.
			
			try:
			
				# Start by fetching the user's data
				user = stack_api.api.fetch_user_profile(account["site"],account["user_id"])["users"][0]
			
				if (not account["last_reputation"] == 0) and (account["last_reputation"] != user["reputation"]):
				
					diff = user["reputation"] - account["last_reputation"]
				
					if CAN_DISPLAY_ALERTS:
				
						# We can alert the user - they gained / lost
						# reputation
				
						if diff > 0:
							msg = _("gained")
				
						else:
							msg = _("lost")
						
						notif_msg = _("You have {message} {amount} reputation.").format(message=msg, amount=abs(diff))
						
						n = pynotify.Notification(account["site_name"] + ":",
								          notif_msg,
								          self.get_image(account["site"]))
						n.show()
				
					# Even without alerts, we can still
					# make the icon 'glow'
					self.ind.set_status(appindicator.STATUS_ATTENTION)
				
					# Set a few variables
				
					# Only set this IF there was no prior
					# notification
					if not account["notification"]:
				
						account["notification"] = 1
						self.total_notifications += 1
				
					account["unread_reputation"] += diff
			
				account["last_reputation"] = user["reputation"]
			
				# Now we fetch the recent comments made to
				# the user (via the /mentioned API method)
				comments = stack_api.api.fetch_user_mentioned(account["site"],account["user_id"])["comments"]
			
				# See if there are any comments
				if (len(comments) > 0):
				
					# Now see if there are any NEW comments
					if (not account["last_comment_id"] == 0) and (not account["last_comment_id"] == comments[0]["comment_id"]):
					
						if CAN_DISPLAY_ALERTS:
							
							notif_msg = _("%s has posted a comment to you.") % (comments[0]["owner"]["display_name"])
							
							n = pynotify.Notification(account["site_name"] + ":",
									          notif_msg,
									          self.get_image(account["site"]))
							n.show()
					
						# Even without alerts, we can still
						# make the icon 'glow'
						self.ind.set_status(appindicator.STATUS_ATTENTION)
					
						if not account["notification"]:
					
							account["notification"] = 1
							self.total_notifications += 1
					
						account["unread_comments"] += 1
			
					account["last_comment_id"] = comments[0]["comment_id"]
				
				# Now do the check for new answers...
				
				# This is a two step process that requires we first get the
				# user's questions on a given site
				users_questions = stack_api.api.fetch_users_questions(account["site"],account["user_id"])["questions"]
				
				# Loop thru the questions, gathering IDs
				id_array = []
				
				for question in users_questions:
					id_array.append(str(question["question_id"]))
				
				# Only do this if there are any
				if len(id_array) and account["last_timestamp"]:
					
					id_str = ";".join(id_array)
					
					# Now query the answers for the questions
					answers = stack_api.api.fetch_answers_to_questions(account["site"],id_str,account["last_timestamp"])["answers"]
					
					# Now see if there are any
					if len(answers):
						
						# Do the notification
						if CAN_DISPLAY_ALERTS:
							
							notif_msg = _("One of your questions has been answered.")
							
							n = pynotify.Notification(account["site_name"] + ":",
									          notif_msg,
									          self.get_image(account["site"]))
							n.show()
						
						# Now display the notification
						self.ind.set_status(appindicator.STATUS_ATTENTION)
					
						if not account["notification"]:
					
							account["notification"] = 1
							self.total_notifications += 1
						
						account["unread_answers"] += len(answers)
				
				# Store the current timestamp
				# for later use
				account["last_timestamp"] = int(time.time())
			
				account["menu"].set_label(self.get_menu_caption(account))
			
			except urllib2.URLError:
				
				# We encountered an error
				print _("There was an error retrieving data from %s.") % (account["site_name"])
		
		# We need to set the timer so we can
		# repeat this in a few minutes / seconds / whatever
		
		self.timer = gobject.timeout_add(self.refresh_rate,self.update)
	
	def new_settings(self,object):
		
		self.reset_defaults()
		self.create_menu()
	
	# Called when an account item is clicked
	def menu_item_activate(self,widget,account):
	
		if account["notification"]:
			
			# We need to reset the item
			account["notification"]      = 0
			account["unread_reputation"] = 0
			account["unread_comments"]   = 0
			account["unread_answers"]    = 0
			
			# Reset the title of the item
			account["menu"].set_label(self.get_menu_caption(account))
			
			self.total_notifications -= 1
			
			if not self.total_notifications:
				self.ind.set_status(appindicator.STATUS_ACTIVE)
		
		else:
			
			# Otherwise, open the page with the stuff on it
			webbrowser.open("http://" + account["site"] + ".com/users/recent/" + str(account["user_id"]))
	
	# This function gets called to determine the
	# caption that a menu item should have
	def get_menu_caption(self,account):
		
		if account["notification"]:
			caption = "* "
		else:
			caption = ""
		
		caption += account["site_name"] + " / " + str(account["last_reputation"])
		
		if account["unread_reputation"]:
			
			rep_msg = _(" [%d reputation]") % (account["unread_reputation"])
			caption += rep_msg
		
		if account["unread_comments"]:
			
			comment_msg = _(" [%d comment(s)]") % (account["unread_comments"])
			caption += comment_msg
		
		if account["unread_answers"]:
			
			comment_msg = _(" [%d answers(s)]") % (account["unread_answers"])
			caption += comment_msg
		
		return caption
	
	# Called when the refresh menu item is clicked
	def menu_ref_activate(self,widget):
	
		self.update()
	
	# Called when the preferences menu item is clicked
	def menu_prefs_activate(self,widget):
	
		# It is possible during the initialization process
		# that an exception will be thrown.
		
		try:
		
			prefs = prefs_dialog.prefs_dialog()
		
			# Now it's simply a matter of waiting
			# until the signal is sent indicating
			# that we need to refresh the list of accounts.
			prefs.connect('new_settings', self.new_settings)
			
		except Exception as inst:
			
			# Tell the user
			dialog = gtk.MessageDialog(None, gtk.DIALOG_MODAL,gtk.MESSAGE_INFO,gtk.BUTTONS_OK,
					           str(inst))
			dialog.set_title(_("An error has occurred:"))
			dialog.connect("response",lambda x,y: dialog.destroy())
			dialog.run()
	
	def menu_about_activate(self,widget):
		
		global __application__,__author__,__copyright__,__license__,__version__
		
		# Display the about dialog box
		dialog = gtk.AboutDialog()
		
		dialog.set_name(__application__)
		dialog.set_authors(__authors__)
		dialog.set_artists(__artists__)
		dialog.set_translator_credits(__translators__)
		dialog.set_copyright(__copyright__)
		dialog.set_license(__license__)
		dialog.set_version(__version__)
		dialog.set_logo_icon_name("stackapplet")
		
		dialog.connect("response",lambda x,y: dialog.destroy())
		
		dialog.run()

	# Called when the quit menu item is clicked
	def menu_quit_activate(self,widget):
	
		gtk.main_quit()

# First check to see if this is the first
# run and the previous configuration exists
LEGACY_FILE_NAME = os.path.join(os.path.expanduser("~"),".stackapplet")

if os.path.isfile(LEGACY_FILE_NAME):
	
	# Import our conversion module
	# and do the conversion
	print _("Legacy config file detected, converting...")
	
	import import_13
	
	import_13.Convert()

# Create the primary instance and
# enter the main GTK loop

if __name__ == "__main__":

	stackapplet_instance = stackapplet()
	
	gtk.main()
