/*  This file is part of "xprintmon"
 *  Copyright (C) 2006 Bernhard R. Link
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1301  USA
 */
#include <config.h>

#include <assert.h>
#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <stdarg.h>
#include <alloca.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/AsciiText.h>

#include "global.h"

struct action_dialog {
	struct child *action;
	Widget parent, popup, form, cancelBtn, closeBtn, text;
	int displayed;
	XtCallbackProc callback; XtPointer callback_data;
};

static void actioncallback(void *privdata, const char *buffer, size_t newsize, bool finished) {
	struct action_dialog *dialog = privdata;

	assert( dialog->action != NULL );
	assert( newsize >= (size_t)dialog->displayed );
	if( dialog->popup != NULL && newsize > (size_t)dialog->displayed) {
		XawTextBlock text;
		int len, lastline;

		len = newsize;

		while( len > 0 && ( buffer[len-1] == '\n'
					|| buffer[len-1] == '\r'
					|| buffer[len-1] == '\t'
					|| buffer[len-1] == ' ' ) )
			len--;
		if( len > dialog->displayed ) {
			text.firstPos = dialog->displayed;
			text.length = len-dialog->displayed;
			text.ptr = (char*)buffer;
			text.format = XawFmt8Bit;
			XtVaSetValues(dialog->text,
					XtNeditType,
						(XtPointer)XawtextAppend,
					NULL);
			XawTextReplace(dialog->text,
					dialog->displayed, dialog->displayed,
					&text);
			dialog->displayed = len;
			for( lastline = len-1; lastline >= 0 ; lastline-- )
				if( buffer[lastline] == '\n' )
					break;
			lastline++;
			XtVaSetValues(dialog->text,
					XtNeditType,
						(XtPointer)XawtextRead,
					XtNinsertPosition,
						(XtPointer)(lastline),
					NULL);
		}
	}
	if( finished ) {
		abortCollector(dialog->action);
		dialog->action = NULL;
		if( dialog->callback != NULL ) {
			dialog->callback(dialog->parent,
			                 dialog->callback_data,
			                 NULL);
		}
		if( dialog->popup == NULL ) {
			free(dialog);
			return;
		} else {
			XtVaSetValues(dialog->closeBtn,
					XtNsensitive,	(XtPointer)True,
					NULL);
			XtVaSetValues(dialog->cancelBtn,
					XtNsensitive,	(XtPointer)False,
					NULL);
			XtInstallAccelerators(dialog->popup, dialog->closeBtn);
			XtInstallAccelerators(dialog->text, dialog->closeBtn);
		}
	}
}

static void action_destroy(Widget w UNUSED, XtPointer privdata,
		XtPointer cad UNUSED) {
	struct action_dialog *dialog = privdata;

	if( dialog->action != NULL )
		abortCollector(dialog->action);
	dialog->action = NULL;
	dialog->popup = NULL;
	free(dialog);
}
static void action_cancel(Widget w UNUSED, XtPointer privdata,
		XtPointer cad UNUSED) {
	struct action_dialog *dialog = privdata;

	if( dialog->action != NULL )
		killCollector(dialog->action);
}
static void action_close(Widget w UNUSED, XtPointer privdata,
		XtPointer cad UNUSED) {
	struct action_dialog *dialog = privdata;

	if( dialog->action != NULL )
		abortCollector(dialog->action);
	dialog->action = NULL;
	XtDestroyWidget(dialog->popup);
}

void notified_logged_action(XtAppContext context, Widget parent, XtCallbackProc callback, XtPointer privdata, const char *command, ...) {
	va_list ap;
	size_t count = 0;
	const char **arguments, **p, *a;
	struct action_dialog *dialog;

	dialog = calloc(1,sizeof(struct action_dialog));
	if( dialog == NULL )
		return;
	dialog->parent = parent;
	dialog->callback = callback;
	dialog->callback_data = privdata;

	va_start(ap, command);
	while( va_arg(ap, char*) != NULL )
		count++;
	va_end(ap);
	arguments = alloca(sizeof(char*)*(count+2));
	p = arguments;
	*(p++) = command;
	va_start(ap, command);
	while( (a = va_arg(ap, char*)) != NULL )
		*(p++) = a;
	*p = NULL;
	assert( (size_t)(p-arguments) == count+1);

	dialog->popup = XtVaCreatePopupShell(command,
			transientShellWidgetClass,
			// topLevelShellWidgetClass,
			parent,
			XtNsensitive,		(XtPointer)True,
			NULL);
	dialog->form = XtVaCreateManagedWidget("logform",
			formWidgetClass, dialog->popup,
			NULL);
	dialog->text = XtVaCreateManagedWidget("text",
			asciiTextWidgetClass, dialog->form,
			XtNdisplayCaret,	(XtPointer)False,
			XtNscrollVertical,	(XtPointer)XawtextScrollAlways,
			XtNeditType,		(XtPointer)XawtextRead,
			XtNuseStringInPlace,	(XtPointer)False,
			XtNtop,		(XtPointer)XawChainTop,
			XtNbottom,	(XtPointer)XawChainBottom,
			XtNleft,	(XtPointer)XawChainLeft,
			XtNright,	(XtPointer)XawChainRight,
			NULL);
	dialog->cancelBtn = XtVaCreateManagedWidget("cancel",
			commandWidgetClass, dialog->form,
			XtNfromVert,	(XtPointer)dialog->text,
			XtNfromHoriz,	(XtPointer)NULL,
			XtNtop,		(XtPointer)XawChainBottom,
			XtNbottom,	(XtPointer)XawChainBottom,
			XtNleft,	(XtPointer)XawChainLeft,
			XtNright,	(XtPointer)XawChainLeft,
			NULL);
	dialog->closeBtn = XtVaCreateManagedWidget("close",
			commandWidgetClass, dialog->form,
			XtNfromHoriz,	(XtPointer)dialog->cancelBtn,
			XtNfromVert,	(XtPointer)dialog->text,
			XtNtop,		(XtPointer)XawChainBottom,
			XtNbottom,	(XtPointer)XawChainBottom,
			XtNleft,	(XtPointer)XawChainLeft,
			XtNright,	(XtPointer)XawChainLeft,
			XtNsensitive,	(XtPointer)False,
			NULL);
	XtAddCallback(dialog->cancelBtn, XtNcallback, action_cancel, dialog);
	XtAddCallback(dialog->closeBtn, XtNcallback, action_close, dialog);
	XtAddCallback(dialog->popup, XtNdestroyCallback, action_destroy, dialog);
	XtInstallAllAccelerators(dialog->popup, dialog->popup);
	XtInstallAllAccelerators(dialog->text, dialog->popup);
	XtSetKeyboardFocus(dialog->popup, dialog->text);
	XtPopup(dialog->popup, XtGrabNone);
	setDeleteHandler(dialog->popup, wmdDestroy);
	dialog->action = newCollector(context, actioncallback, dialog,
			command, (char**)arguments);
}
