/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.painter.noise;

import java.awt.Color;
import java.awt.image.*;

import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.PerlinNoiseGenerator;
import org.jvnet.substance.utils.SubstanceColorUtilities;

/**
 * Factory for creating noise images. This class is part of officially supported
 * API.
 * 
 * @author Kirill Grouchnikov.
 */
public class NoiseFactory {
	/**
	 * Returns a noise image.
	 * 
	 * @param theme1
	 *            The first theme to use for rendering the image.
	 * @param theme2
	 *            The second theme to use for rendering the image.
	 * @param width
	 *            Image width.
	 * @param height
	 *            Image height.
	 * @param xFactor
	 *            X stretch factor.
	 * @param yFactor
	 *            Y stretch factor.
	 * @param hasConstantZ
	 *            Indication whether the Z is constant.
	 * @param noiseFilter
	 *            Noise filter to apply.
	 * @param toBlur
	 *            Indication whether the resulting image should be blurred.
	 * @param isPreview
	 *            Indication whether the image is in preview mode.
	 * @return Noise image.
	 */
	public static BufferedImage getNoiseImage(SubstanceTheme theme1,
			SubstanceTheme theme2, int width, int height, double xFactor,
			double yFactor, boolean hasConstantZ, NoiseFilter noiseFilter,
			boolean toBlur, boolean isPreview) {
		Color c1_1 = SubstanceColorUtilities.getWatermarkDarkColor(theme1);
		Color c2_1 = SubstanceColorUtilities.getWatermarkStampColor(theme1);
		Color c3_1 = SubstanceColorUtilities.getWatermarkLightColor(theme1);
		Color c1_2 = SubstanceColorUtilities.getWatermarkDarkColor(theme2);
		Color c2_2 = SubstanceColorUtilities.getWatermarkStampColor(theme2);
		Color c3_2 = SubstanceColorUtilities.getWatermarkLightColor(theme2);

		Color c1 = SubstanceColorUtilities
				.getInterpolatedColor(c1_1, c1_2, 0.1);
		Color c2 = SubstanceColorUtilities
				.getInterpolatedColor(c2_1, c2_2, 0.5);
		Color c3 = SubstanceColorUtilities
				.getInterpolatedColor(c3_1, c3_2, 0.9);
		if (isPreview) {
			c1 = SubstanceColorUtilities.getAlphaColor(c1, 255);
			c2 = SubstanceColorUtilities.getAlphaColor(c2, 255);
			c3 = SubstanceColorUtilities.getAlphaColor(c3, 255);
		}

		BufferedImage dst = new BufferedImage(width, height,
				BufferedImage.TYPE_INT_ARGB);

		// Borrow from Sebastien Petrucci fast blur code - direct access
		// to the raster data
		int[] dstBuffer = ((DataBufferInt) dst.getRaster().getDataBuffer())
				.getData();

		double m2 = xFactor * width * xFactor * width + yFactor * height
				* yFactor * height;
		int pos = 0;
		for (int j = 0; j < height; j++) {
			double jj = yFactor * j;
			for (int i = 0; i < width; i++) {
				double ii = xFactor * i;
				double z = hasConstantZ ? 1.0 : Math.sqrt(m2 - ii * ii - jj
						* jj);
				double noise = 0.5 + 0.5 * PerlinNoiseGenerator
						.noise(ii, jj, z);
				if (noiseFilter != null)
					noise = noiseFilter.apply(i, j, z, noise);

				// dstBuffer[pos++] = (noise < 0.5) ? SubstanceColorUtilities
				// .getInterpolatedRGB(c2, c1, 2.0 * noise)
				// : SubstanceColorUtilities.getInterpolatedRGB(c3, c2,
				// 2.0 * (noise - .5));
				dstBuffer[pos++] = SubstanceColorUtilities.getInterpolatedRGB(
						c3, c1, 2.0 * noise);
			}
		}
		if (toBlur) {
			ConvolveOp convolve = new ConvolveOp(new Kernel(3, 3, new float[] {
					.08f, .08f, .08f, .08f, .38f, .08f, .08f, .08f, .08f }),
					ConvolveOp.EDGE_NO_OP, null);
			dst = convolve.filter(dst, null);
		}
		return dst;
	}
}
