package test.contrib.classloader;

import java.io.IOException;
import java.net.URL;
import java.security.*;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.jar.Attributes.Name;

import sun.misc.Resource;

public class UniversalClassLoader extends SecureClassLoader {

	/* The search path list for classes and resources */
	private final Classpath classpath;

	/* The access context to be used when loading classes and resources */
	private AccessControlContext acc;

	// private UID uid = null;

	/**
	 * Constructs a new UniversalClassLoader for the given Classpath. The
	 * Classpaths will be searched in the order specified for classes and
	 * resources after first searching in the specified parent class loader.
	 * <p>
	 * If there is a security manager, this method first calls the security
	 * manager's <code>checkCreateClassLoader</code> method to ensure creation
	 * of a class loader is allowed.
	 * 
	 * @param path
	 *            the main classpath
	 * @param parent
	 *            the parent classloader
	 */
	public UniversalClassLoader(Classpath path, ClassLoader parent) {
		super(parent);
		SecurityManager security = System.getSecurityManager();
		if (security != null) {
			security.checkCreateClassLoader();
		}
		acc = AccessController.getContext();
		classpath = path;
	}

	/**
	 * @see java.lang.ClassLoader#findLibrary(java.lang.String)
	 */
	@Override
	protected String findLibrary(String libname) {
		return classpath.findLibrary(libname);
	}

	/**
	 * Adds a new Classpath to this Classloader
	 * 
	 * @param path
	 *            the new classpath
	 * @return TRUE if added successfuly and FALSE if the classpath already
	 *         exist or NULL
	 */
	public synchronized boolean addPath(Classpath path) {
		return classpath.addChildren(path);
	}

	/**
	 * @see java.lang.ClassLoader#findResource(java.lang.String)
	 */
	@Override
	public URL findResource(final String name) {
		return AccessController.doPrivileged(new PrivilegedAction<URL>() {
			public URL run() {
				return classpath.findResource(name, true);
			}
		}, acc);
	}

	/**
	 * @see java.lang.ClassLoader#findClass(java.lang.String)
	 */
	@Override
	protected Class<?> findClass(final String name)
			throws ClassNotFoundException {
		try {
			return AccessController.doPrivileged(
					new PrivilegedExceptionAction<Class>() {
						public Class run() throws Exception {
							Class klass = classpath.findClass(name);
							if (klass != null
									|| (klass = findClass(name, name.replace(
											'.', '/').concat(".class"),
											classpath)) != null) {
								return klass;
							}
							// String path = name.replace('.', '/').concat(".class");
							// Classpath cp = classpath.findClasspath(path, false);
							// if (cp != null) {
							// try {
							// return defineClass(name, cp.getResource(path, false),
					// cp);
							// } catch (IOException e) {
							// throw new ClassNotFoundException(name, e);
							// }
							// }
							throw new ClassNotFoundException(name);
						}
					}, acc);
		} catch (java.security.PrivilegedActionException pae) {
			throw (ClassNotFoundException) pae.getException();
		}
	}

	private Class findClass(String name, String pathname, Classpath cp)
			throws ClassNotFoundException {
		Resource src = cp.getResource(pathname, false);
		if (src != null) {
			try {
				return defineClass(name, src, cp);
			} catch (IOException e) {
				throw new ClassNotFoundException(name, e);
			}
		}
		Class klass = null;
		for (Classpath path : cp.children) {
			if ((klass = findClass(name, pathname, path)) != null) {
				break;
			}
		}
		return klass;
	}

	// /////////////////////////////////////////////////////////
	// // A stolen part from java.net.URLClassLoader
	// /////////////////////////////////////////////////////////

	/*
	 * Defines a Class using the class bytes obtained from the specified
	 * Resource. The resulting Class must be resolved before it can be used.
	 */
	private Class defineClass(String name, Resource res, Classpath path)
			throws IOException {
		int i = name.lastIndexOf('.');
		URL url = res.getCodeSourceURL();
		if (i != -1) {
			String pkgname = name.substring(0, i);
			Package pkg = getPackage(pkgname);
			Manifest man = res.getManifest();
			if (pkg != null) {
				if (pkg.isSealed()) {
					if (!pkg.isSealed(url)) {
						throw new SecurityException(
								"sealing violation: package " + pkgname
										+ " is sealed");
					}

				} else {
					if ((man != null) && isSealed(pkgname, man)) {
						throw new SecurityException(
								"sealing violation: can't seal package "
										+ pkgname + ": already loaded");
					}
				}
			} else {
				if (man != null) {
					definePackage(pkgname, man, url);
				} else {
					definePackage(pkgname, null, null, null, null, null, null,
							null);
				}
			}
		}
		byte[] b = res.getBytes();
		java.security.cert.Certificate[] certs = res.getCertificates();
		CodeSource cs = new CodeSource(url, certs);
		Class klass = defineClass(name, b, 0, b.length, cs);
		path.linkClass(klass);
		return klass;
	}

	/**
	 * Defines a new package by name in this ClassLoader. The attributes
	 * contained in the specified Manifest will be used to obtain package
	 * version and sealing information. For sealed packages, the additional URL
	 * specifies the code source URL from which the package was loaded.
	 * 
	 * @param name
	 *            the package name
	 * @param man
	 *            the Manifest containing package version and sealing
	 *            information
	 * @param url
	 *            the code source url for the package, or null if none
	 * @exception IllegalArgumentException
	 *                if the package name duplicates an existing package either
	 *                in this class loader or one of its ancestors
	 * @return the newly defined Package object
	 */
	protected Package definePackage(String name, Manifest man, URL url)
			throws IllegalArgumentException {
		String path = name.replace('.', '/').concat("/");
		String specTitle = null, specVersion = null, specVendor = null;
		String implTitle = null, implVersion = null, implVendor = null;
		String sealed = null;
		URL sealBase = null;
		Attributes attr = man.getAttributes(path);
		if (attr != null) {
			specTitle = attr.getValue(Name.SPECIFICATION_TITLE);
			specVersion = attr.getValue(Name.SPECIFICATION_VERSION);
			specVendor = attr.getValue(Name.SPECIFICATION_VENDOR);
			implTitle = attr.getValue(Name.IMPLEMENTATION_TITLE);
			implVersion = attr.getValue(Name.IMPLEMENTATION_VERSION);
			implVendor = attr.getValue(Name.IMPLEMENTATION_VENDOR);
			sealed = attr.getValue(Name.SEALED);
		}
		attr = man.getMainAttributes();
		if (attr != null) {
			if (specTitle == null) {
				specTitle = attr.getValue(Name.SPECIFICATION_TITLE);
			}
			if (specVersion == null) {
				specVersion = attr.getValue(Name.SPECIFICATION_VERSION);
			}
			if (specVendor == null) {
				specVendor = attr.getValue(Name.SPECIFICATION_VENDOR);
			}
			if (implTitle == null) {
				implTitle = attr.getValue(Name.IMPLEMENTATION_TITLE);
			}
			if (implVersion == null) {
				implVersion = attr.getValue(Name.IMPLEMENTATION_VERSION);
			}
			if (implVendor == null) {
				implVendor = attr.getValue(Name.IMPLEMENTATION_VENDOR);
			}
			if (sealed == null) {
				sealed = attr.getValue(Name.SEALED);
			}
		}
		if ("true".equalsIgnoreCase(sealed)) {
			sealBase = url;
		}
		return definePackage(name, specTitle, specVersion, specVendor,
				implTitle, implVersion, implVendor, sealBase);
	}

	/*
	 * Returns true if the specified package name is sealed according to the
	 * given manifest.
	 */
	private boolean isSealed(String name, Manifest man) {
		String path = name.replace('.', '/').concat("/");
		Attributes attr = man.getAttributes(path);
		String sealed = null;
		if (attr != null) {
			sealed = attr.getValue(Name.SEALED);
		}
		if (sealed == null) {
			if ((attr = man.getMainAttributes()) != null) {
				sealed = attr.getValue(Name.SEALED);
			}
		}
		return "true".equalsIgnoreCase(sealed);
	}

}
