/*
   XGBitmapImageRep.m

   NSBitmapImageRep for GNUstep GUI X/GPS Backend

   Copyright (C) 1996-1999 Free Software Foundation, Inc.

   Author:  Adam Fedor <fedor@colorado.edu>
   Author:  Scott Christley <scottc@net-community.com>
   Date: Feb 1996
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998
   Author:  Richard Frith-Macdonald <richard@brainstorm.co.uk>
   Date: Mar 1999

   This file is part of the GNUstep GUI X/GPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "XGDrawingEngine.h"
#include "gnustep/xgps/XGGState.h"
#include <Foundation/NSDebug.h>
#include <Foundation/NSUserDefaults.h>

/* Composite source image (pixmap) onto a destination image with alpha.
   Only works for op=Sover now */
int
_pixmap_combine_alpha(RContext *context, RXImage *source_im, 
		     RXImage *source_alpha, RXImage *dest_im,
		     RXImage *dest_alpha, float origin_x, float origin_y,
		     NSRect drect, NSCompositingOperation op,
		     XGDrawMechanism drawMechanism)
{
  unsigned long  pixel;
  unsigned short oldAlpha = 0;
  unsigned long  oldPixel = 0;
  unsigned char  oldAr = 0;
  unsigned char	 oldAg = 0;
  unsigned char	 oldAb = 0;
  unsigned char	 oldBr = 0;
  unsigned char	 oldBg = 0;
  unsigned char	 oldBb = 0;

  if (drawMechanism == XGDM_FAST15 || drawMechanism == XGDM_FAST16)
    {
      unsigned  rshift, gshift;
      unsigned	row;
      rshift = 11; gshift = 6;
      if (drawMechanism == XGDM_FAST15)
      	{
	  rshift = 10;
	  gshift = 5;
	}

      for (row = 0; row < NSHeight(drect); row++)
	{
	  unsigned	col;

	  for (col = 0; col < NSWidth(drect); col++)
	    {
	      int r, g, b, alpha;
	      pixel = XGetPixel(source_im->image, 
				col+origin_x, row+origin_y);
	      r = (pixel >> rshift) & 0x1f;
	      g = (pixel >> 5)  & 0x1f;
	      b = pixel & 0x1f;
	      pixel = 255;
	      if (source_alpha)
		pixel = XGetPixel(source_alpha->image,
				  col+origin_x, row+origin_y);
	      alpha = pixel >> 3;
	      if (alpha == 0)
		continue;		// background unchanged.
	      if (alpha != 31)
		{
		  unsigned short	ialpha = 31 - alpha;

		  /*
		   * Get the background pixel and convert to RGB.
		   */
		  pixel = XGetPixel(dest_im->image, col, row);
		  if (pixel != oldPixel)
		    {
		      oldPixel = pixel;
		      oldBb = pixel & 0x1f;
		      pixel >>= gshift;
		      oldBg = pixel & 0x1f;
		      pixel >>= 5;
		      oldBr = pixel;
		      oldAlpha = 0;
		    }
		  if (alpha != oldAlpha)
		    {
		      oldAlpha = alpha;
		      oldAr = ialpha * oldBr;
		      oldAg = ialpha * oldBg;
		      oldAb = ialpha * oldBb;
		    }
		      
		  // mix in alpha to produce RGB out
		  r = (oldAr + (r*alpha))/31;
		  g = (oldAg + (g*alpha))/31;
		  b = (oldAb + (b*alpha))/31;
		}
	      pixel = (((r << 5) + g) << gshift) + b;
	      XPutPixel(dest_im->image, col, row, pixel);
	      if (dest_alpha)
		XPutPixel(dest_alpha->image, col, row, (alpha << 3));
	    }
	}
    }
  else if (drawMechanism == XGDM_FAST32)
    {
      unsigned		row;

      for (row = 0; row < NSHeight(drect); row++)
	{
	  unsigned	col;

	  for (col = 0; col < NSWidth(drect); col++)
	    {
	      int r, g, b, alpha;
	      pixel = XGetPixel(source_im->image, 
				col+origin_x, row+origin_y);
	      r = (pixel >> 16) & 0xff;
	      g = (pixel >> 8)  & 0xff;
	      b = pixel & 0xff;
	      pixel = 255;
	      if (source_alpha)
		pixel = XGetPixel(source_alpha->image,
				  col+origin_x, row+origin_y);
	      alpha = pixel & 0xff;
	      if (alpha == 0)
		continue;		// background unchanged.

	      if (alpha != 255)
		{
		  unsigned short	ialpha = 255 - alpha;

		  pixel = XGetPixel(dest_im->image, col, row);
		  if (pixel != oldPixel)
		    {
		      oldPixel = pixel;
		      oldBr = (pixel & 0xff0000) >> 16;
		      oldBg = (pixel & 0xff00) >> 8;
		      oldBb = pixel & 0xff;
		      oldAlpha = 0;
		    }
		  if (alpha != oldAlpha)
		    {
		      oldAlpha = alpha;
		      oldAr = ialpha * oldBr;
		      oldAg = ialpha * oldBg;
		      oldAb = ialpha * oldBb;
		    }
		      
		  // mix in alpha to produce RGB out
		  r = (oldAr + (r*alpha))/255;
		  g = (oldAg + (g*alpha))/255;
		  b = (oldAb + (b*alpha))/255;
		}
	      pixel = (((r << 8) + g) << 8) + b;
	      XPutPixel(dest_im->image, col, row, pixel);
	      if (dest_alpha)
		XPutPixel(dest_alpha->image, col, row, alpha);
	    }
	}
    }
  else if (drawMechanism == XGDM_FAST32_BGR)
    {
      unsigned          row;
      
      for (row = 0; row < NSHeight(drect); row++)
	{
	  unsigned      col;
	  
	  for (col = 0; col < NSWidth(drect); col++)
	    {
	      int r, g, b, alpha;
	      pixel = XGetPixel(source_im->image, 
				col+origin_x, row+origin_y);
	      b = (pixel >> 16) & 0xff;
	      r = (pixel >> 8)  & 0xff;
	      g = pixel & 0xff;
	      pixel = 255;
	      if (source_alpha)
		pixel = XGetPixel(source_alpha->image,
				  col+origin_x, row+origin_y);
	      alpha = pixel & 0xff;
	      if (alpha == 0)
		continue;		// background unchanged.

	      if (alpha != 255)
		{
		  unsigned short    ialpha = 255 - alpha;
		  
		  pixel = XGetPixel(dest_im->image, col, row);
		  if (pixel != oldPixel)
		    {
		      oldPixel = pixel;
		      oldBb = (pixel & 0xff0000) >> 16;
		      oldBg = (pixel & 0xff00) >> 8;
		      oldBr = pixel & 0xff;
		      oldAlpha = 0;
		    }
		  if (alpha != oldAlpha)
		    {
		      oldAlpha = alpha;
		      oldAr = ialpha * oldBr;
		      oldAg = ialpha * oldBg;
		      oldAb = ialpha * oldBb;
		    }
		  
		  // mix in alpha to produce RGB out
		  r = (oldAr + (r*alpha))/255;
		  g = (oldAg + (g*alpha))/255;
		  b = (oldAb + (b*alpha))/255;
		}
	      pixel = (((b << 8) + g) << 8) + r;
	      XPutPixel(dest_im->image, col, row, pixel);
	      if (dest_alpha)
		XPutPixel(dest_alpha->image, col, row, alpha);
	    }
	}
    }
  else
    {
      XColor		c2;
      unsigned		row;

      /*
       * This block of code should be totally portable as it uses the
       * 'official' X mechanism for converting from pixel values to
       * RGB color values - on the downside, it's very slow.
       */
      pixel = (unsigned long)-1;	// Never valid?
      c2.pixel = pixel;

      for (row = 0; row < NSHeight(drect); row++)
	{
	  unsigned	col;

	  for (col = 0; col < NSWidth(drect); col++)
	    {
	      int r, g, b, alpha;
	      XColor pcolor, acolor, c0, c1;
	      pcolor.pixel = XGetPixel(source_im->image, 
				       col+origin_x, row+origin_y);
	      XQueryColor(context->dpy, context->cmap, &pcolor);
	      r = pcolor.red >> 8;
	      g = pcolor.green >> 8;
	      b = pcolor.blue >> 8;
	      alpha = 255;
	      if (source_alpha)
		{
		  acolor.pixel = XGetPixel(source_alpha->image,
				    col+origin_x, row+origin_y);
		  XQueryColor(context->dpy, context->cmap, &acolor);
		  alpha = acolor.red >> 8;
		}
	      if (alpha == 0)
		continue;		// background unchanged.

	      if (alpha == 255)
		{
		  c1 = pcolor;
		}
	      else
		{
		  unsigned short	ialpha = 255 - alpha;
		      
		  /*
		   * RGB color components in X are 16-bit values -
		   * but our bitmap contains 8-bit values so we must
		   * adjust our values up to 16-bit, which we can do
		   * by increasing their alpha component by 256.
		   */
		  alpha <<= 8;
		      
		  /*
		   * Get the background pixel and convert to RGB if
		   * we haven't already done the conversion earlier.
		   */
		  c0.pixel = XGetPixel(dest_im->image, col, row);
		  if (c0.pixel != oldPixel)
		    {
		      oldPixel = c0.pixel;
		      XQueryColor(context->dpy, context->cmap, &c0);
		    }
		      
		  // mix in alpha to produce RGB out
		  c1.red = ((c0.red*ialpha) + (r*alpha))/255;
		  c1.green = ((c0.green*ialpha) + (g*alpha))/255;
		  c1.blue = ((c0.blue*ialpha) + (b*alpha))/255;
		}

	      /*
	       * Convert the X RGB value to a colormap entry if we
	       * don't already have one.  Then set the pixel.
	       * NB.  XAllocColor() will not necessarily return a
	       * color with exactly the rgb components we gave it, so
	       * we must record those components beforehand.
	       */
	      if (c2.pixel == (unsigned long)-1
		  || c2.red != c1.red
		  || c2.green != c1.green
		  || c2.blue != c1.blue)
		{
		  c2.red = c1.red;
		  c2.green = c1.green;
		  c2.blue = c1.blue;
		  XAllocColor(context->dpy, context->cmap, &c1);
		  c2.pixel = c1.pixel;
		}
	      XPutPixel(dest_im->image, col, row, c1.pixel);
	      if (dest_alpha)
		XPutPixel(dest_alpha->image, col, row, acolor.pixel);
	    }
	}
    }
  
  return 0;
}

/* Composite source image bitmap onto a destination image with alpha.
   Only works for op=Sover now */
int
_bitmap_combine_alpha(RContext *context, 
		      const unsigned char *image_data,
		      float width, float height, int numColors,
		      BOOL usesAlpha, RXImage *dest_im, RXImage *dest_alpha,
		      NSRect drect, NSCompositingOperation op,
		      XGDrawMechanism drawMechanism)
{
  XColor	c0;
  XColor	c1;

  if (numColors == 3 || (numColors == 4 && usesAlpha)
      || numColors == 1 || (numColors == 2 && usesAlpha))
    {
      BOOL colorimage;
      unsigned long    	pixel;
      unsigned short	oldAlpha = 0;
      unsigned long	oldPixel = 0;
      unsigned char    	oldAr = 0;
      unsigned char    	oldAg = 0;
      unsigned char    	oldAb = 0;
      unsigned char    	oldBr = 0;
      unsigned char    	oldBg = 0;
      unsigned char    	oldBb = 0;

      colorimage = NO;
      if (numColors == 3 || (numColors == 4 && usesAlpha))
	colorimage = YES;

      if (drawMechanism == XGDM_FAST15 || drawMechanism == XGDM_FAST16)
        {
          unsigned  rshift, gshift;
          unsigned	row;
          rshift = 11; gshift = 6;
          if (drawMechanism == XGDM_FAST15)
      	    {
	      rshift = 10;
	      gshift = 5;
	    }

	  for (row = 0; row < height; row++)
	    {
	      unsigned	col;

	      for (col = 0; col < width; col++)
		{
		  unsigned short alpha;
		  unsigned char	r, g, b;
		  if (colorimage)
		    {
		      r = *image_data++;
		      g = *image_data++;
		      b = *image_data++;
		    }
		  else 
		    r = g = b = *image_data++;
		  if (row >= NSHeight(drect) || col >= NSWidth(drect))
		    {
		      if (usesAlpha)
			image_data++;
		      continue;
		    }

		  /*
		   * Convert 8-bit components down to the 5-bit values
		   * that the display system can actually handle.
		   */
		  r >>= 3;
		  g >>= 3;
		  b >>= 3;
		  if (usesAlpha)
		    {
		      alpha = (unsigned short)*image_data++;
		      if (dest_alpha)
			XPutPixel(dest_alpha->image, col, row, alpha);
		      if (alpha == 0)
			continue;		// background unchanged.

		      alpha >>= 3;		// Convert to 5-bit.

		      if (alpha != 31)
			{
			  unsigned short	ialpha = 31 - alpha;

			  /*
			   * Get the background pixel and convert to RGB.
			   */
			  pixel = XGetPixel(dest_im->image, col, row);
			  if (pixel != oldPixel)
			    {
			      oldPixel = pixel;
			      oldBb = pixel & 0x1f;
			      pixel >>= gshift;
			      oldBg = pixel & 0x1f;
			      pixel >>= 5;
			      oldBr = pixel;
			      oldAlpha = 0;
			    }
			  if (alpha != oldAlpha)
			    {
			      oldAlpha = alpha;
			      oldAr = ialpha * oldBr;
			      oldAg = ialpha * oldBg;
			      oldAb = ialpha * oldBb;
			    }

			  // mix in alpha to produce RGB out
			  r = (oldAr + (r*alpha))/31;
			  g = (oldAg + (g*alpha))/31;
			  b = (oldAb + (b*alpha))/31;
			}
		    }
		  pixel = (((r << 5) + g) << gshift) + b;
		  XPutPixel(dest_im->image, col, row, pixel);
		}
	    }
	}
      else if (drawMechanism == XGDM_FAST32)
	{
	  unsigned		row;

	  for (row = 0; row < height; row++)
	    {
	      unsigned	col;

	      for (col = 0; col < width; col++)
		{
		  unsigned short	alpha;
		  unsigned char	r, g, b;
		  if (colorimage)
		    {
		      r = *image_data++;
		      g = *image_data++;
		      b = *image_data++;
		    }
		  else 
		    r = g = b = *image_data++;
		  if (row >= NSHeight(drect) || col >= NSWidth(drect))
		    {
		      if (usesAlpha)
			image_data++;
		      continue;
		    }

		  if (usesAlpha)
		    {

		      alpha = (unsigned short)*image_data++;
		      if (dest_alpha)
			XPutPixel(dest_alpha->image, col, row, alpha);
		      if (alpha == 0)
			continue;		// background unchanged.

		      if (alpha != 255)
			{
			  unsigned short	ialpha = 255 - alpha;

			  pixel = XGetPixel(dest_im->image, col, row);
			  if (pixel != oldPixel)
			    {
			      oldPixel = pixel;
			      oldBr = (pixel & 0xff0000) >> 16;
			      oldBg = (pixel & 0xff00) >> 8;
			      oldBb = pixel & 0xff;
			      oldAlpha = 0;
			    }
			  if (alpha != oldAlpha)
			    {
			      oldAlpha = alpha;
			      oldAr = ialpha * oldBr;
			      oldAg = ialpha * oldBg;
			      oldAb = ialpha * oldBb;
			    }

			  // mix in alpha to produce RGB out
			  r = (oldAr + (r*alpha))/255;
			  g = (oldAg + (g*alpha))/255;
			  b = (oldAb + (b*alpha))/255;
			}
		    }
		  pixel = (((r << 8) + g) << 8) + b;
		  XPutPixel(dest_im->image, col, row, pixel);
		}
	    }
	}
      else if (drawMechanism == XGDM_FAST32_BGR)
	{
	  unsigned          row;
 
	  for (row = 0; row < height; row++)
	    {
	      unsigned      col;
 
	      for (col = 0; col < width; col++)
		{
		  unsigned short	alpha;
		  unsigned char	r, g, b;
		  if (colorimage)
		    {
		      r = *image_data++;
		      g = *image_data++;
		      b = *image_data++;
		    }
		  else 
		    r = g = b = *image_data++;
		  if (row >= NSHeight(drect) || col >= NSWidth(drect))
		    {
		      if (usesAlpha)
			image_data++;
		      continue;
		    }
 
		  if (usesAlpha)
		    {
 		      alpha = (unsigned short)*image_data++;
		      if (dest_alpha)
			XPutPixel(dest_alpha->image, col, row, alpha);
		      if (alpha == 0)
			continue;           // background unchanged.
 
		      if (alpha != 255)
			{
			  unsigned short    ialpha = 255 - alpha;
 
			  pixel = XGetPixel(dest_im->image, col, row);
			  if (pixel != oldPixel)
			    {
			      oldPixel = pixel;
			      oldBb = (pixel & 0xff0000) >> 16;
			      oldBg = (pixel & 0xff00) >> 8;
			      oldBr = pixel & 0xff;
			      oldAlpha = 0;
			    }
			  if (alpha != oldAlpha)
			    {
			      oldAlpha = alpha;
			      oldAr = ialpha * oldBr;
			      oldAg = ialpha * oldBg;
			      oldAb = ialpha * oldBb;
			    }
 
			  // mix in alpha to produce RGB out
			  r = (oldAr + (r*alpha))/255;
			  g = (oldAg + (g*alpha))/255;
			  b = (oldAb + (b*alpha))/255;
			}
		    }
		  pixel = (((b << 8) + g) << 8) + r;
		  XPutPixel(dest_im->image, col, row, pixel);
		}
	    }
	}
      else
	{
	  XColor		c2, a2;
	  unsigned		row, oldAlpha = 65537;

	  /*
	   * This block of code should be totally portable as it uses the
	   * 'official' X mechanism for converting from pixel values to
	   * RGB color values - on the downside, it's very slow.
	   */
	  pixel = (unsigned long)-1;	// Never valid?
	  c2.pixel = pixel;

	  for (row = 0; row < height; row++)
	    {
	      unsigned	col;

	      for (col = 0; col < width; col++)
		{
		  unsigned short alpha;
		  unsigned char	r, g, b;
		  if (colorimage)
		    {
		      r = *image_data++;
		      g = *image_data++;
		      b = *image_data++;
		    }
		  else 
		    r = g = b = *image_data++;
		  if (row >= NSHeight(drect) || col >= NSWidth(drect))
		    {
		      if (usesAlpha)
			image_data++;
		      continue;
		    }

		  if (usesAlpha)
		    {
		      alpha = (unsigned short)*image_data++;
		      if (alpha != oldAlpha)
			{
			  oldAlpha = alpha;
			  a2.red = alpha << 8;
			  a2.green = alpha << 8;
			  a2.blue = alpha << 8;
			  XAllocColor(context->dpy, context->cmap, &a2);
			}
		      if (dest_alpha)
			XPutPixel(dest_alpha->image, col, row, a2.pixel);
		      if (alpha == 0)
			continue;		// background unchanged.

		      if (alpha == 255)
			{
			  c1.red = r << 8;
			  c1.green = g << 8;
			  c1.blue = b << 8;
			}
		      else
			{
			  unsigned short	ialpha = 255 - alpha;

			  /*
			   * RGB color components in X are 16-bit values -
			   * but our bitmap contains 8-bit values so we must
			   * adjust our values up to 16-bit, which we can do
			   * by increasing their alpha component by 256.
			   */
			  alpha <<= 8;

			  /*
			   * Get the background pixel and convert to RGB if
			   * we haven't already done the conversion earlier.
			   */
			  c0.pixel = XGetPixel(dest_im->image, col, row);
			  if (c0.pixel != pixel)
			    {
			      pixel = c0.pixel;
			      XQueryColor(context->dpy, context->cmap, &c0);
			    }

			  // mix in alpha to produce RGB out
			  c1.red = ((c0.red*ialpha) + (r*alpha))/255;
			  c1.green = ((c0.green*ialpha) + (g*alpha))/255;
			  c1.blue = ((c0.blue*ialpha) + (b*alpha))/255;
			}
		    }
		  else
		    {
		      /*
		       * Simply convert our 8-bit RGB values to X-style
		       * 16-bit values, then determine the X colormap
		       * entry and set the pixel.
		       */
		      c1.red = r << 8;
		      c1.green = g << 8;
		      c1.blue = b << 8;
		    }

		  /*
		   * Convert the X RGB value to a colormap entry if we
		   * don't already have one.  Then set the pixel.
		   * NB.  XAllocColor() will not necessarily return a
		   * color with exactly the rgb components we gave it, so
		   * we must record those components beforehand.
		   */
		  if (c2.pixel == (unsigned long)-1
		      || c2.red != c1.red
		      || c2.green != c1.green
		      || c2.blue != c1.blue)
		    {
		      c2.red = c1.red;
		      c2.green = c1.green;
		      c2.blue = c1.blue;
		      XAllocColor(context->dpy, context->cmap, &c1);
		      c2.pixel = c1.pixel;
		    }
		  XPutPixel(dest_im->image, col, row, c1.pixel);
		}
	    }
	}
    }
  else
    {
      NSLog(@"Backend cannot handle images with %d colors", numColors);
    }
  return 0;
}

void NSDrawBitmap(NSRect rect,
                  int pixelsWide,
                  int pixelsHigh,
                  int bitsPerSample,
                  int samplesPerPixel,
                  int bitsPerPixel,
                  int bytesPerRow,
                  BOOL isPlanar,
                  BOOL hasAlpha,
                  NSString *colorSpaceName,
                  const unsigned char *const data[5])
{
  XGContext *context = GSCurrentContext();
  NSMutableDictionary *dict;

  DPSmoveto(context, NSMinX(rect), NSMinY(rect));

  /* This does essentially what the DPS...image operators do, so
     as to avoid an extra method call */
  dict = [NSMutableDictionary dictionary];
  [dict setObject: [NSNumber numberWithInt: pixelsWide] forKey: @"Width"];
  [dict setObject: [NSNumber numberWithInt: pixelsHigh] forKey: @"Height"];
  [dict setObject: [NSNumber numberWithInt: bitsPerSample] 
	forKey: @"BitsPerComponent"];
  [dict setObject: [NSNumber numberWithInt: samplesPerPixel] 
	forKey: @"SamplesPerPixel"];
  /* Fixme - handle multiple data sources */
  [dict setObject: [NSValue valueWithPointer: data] forKey: @"DataSource"];
  [dict setObject: [NSNumber numberWithBool: isPlanar] 
			     forKey: @"MultipleDataSources"];

  [dict setObject: [NSNumber numberWithBool: hasAlpha] forKey: @"Alpha"];
  [[context xrCurrentGState] DPSimage: dict];
}

