/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/*
 * Name          : pips/src/pipsReadAttDef.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 * Synopsis      : 
 *
 *   pipsReadAttDef (file, varName, attName,
 *                    attType, attNum, attValueArray)
 *
 *   FILE *file          IS  File containing attribute definitions
 *   char *varName       IS  Returned variable name
 *   char *attName       IS  Returned attribute name
 *   nc_type *attType    IS  Returned nc_type of attribute
 *   int  *attNum        IS  Number of values in 'attValueArray'
 *   char *attValueArray IS  Returned attribute values
 * 
 * Description   : 
 *
 *   Parses the contents of an open text file and
 *   returns names and values for ONE attribute definition.
 *   'varName' is the affected variable or NC_GLOBAL,
 *   'attValueArray' contains the new values for the attribute
 *   'attName' of variable 'varName'. If the ':' in an attribute
 *   definition is followed by a ';', then 'attNum' = 0,
 *   meaning, that this attribute has to be deleted.
 */

#include "pips.h"   /* PIPS system header file  */

#ifdef __STDC__

int pipsReadAttDef (FILE *file, char *varName, char *attName,
                     nc_type *attType, int *attNum, char *attValueArray)

#else

pipsReadAttDef (file, varName, attName, attType, attNum, attValueArray)
    FILE *file;
    char *varName;
    char *attName;
    nc_type *attType;
    int *attNum;
    char *attValueArray;

#endif
{
  int result;                    /* Number of items read by scanf            */
  char token [3];                /* buffer to read and identify a token      */
  int moreAttValues = 1;         /* is true as long as ',' is behind a value */
  char *attValue;                /* Where to put next attr. value            */
  double d;                      /* holds scanned double value               */
  float f;                       /* holds scanned float value                */
  long l;                        /* holds scanned long value                 */
  short s;                       /* holds scanned short value                */


/******************************************************************************
 *  I N I T   V A R I A B L E S
 ******************************************************************************
 */
  varName [0] = 0;               /* Default is "" == NC_GLOBAL               */
  attValue = attValueArray;      /* Store first value at beginning of array  */
  *attNum = 0;                   /* No attribute values stored yet           */
  *attType = NC_UNSPECIFIED;     /* First att. value will determine type     */

/******************************************************************************
 *   ' : '   O R   ' : '   V A R N A M E 
 ******************************************************************************
 */
  result = fscanf (file, " %1[:]", token);
  switch (result) {

/*
 * End of file NOT within an attribute definition
 */
    case EOF :
      return (EOF);

/*
 * No ':' found -> expect variable name now
 */
    case 0 :
      result = fscanf (file, " %[^: ]", varName);
      switch (result) {

/*
 * Syntax error: End of file instead of variable name found
 */
        case EOF :
          pipsPrintError (-1,PipsPrematureEof,NULL,PipsDontExit);
          return (PipsPrematureEof);

/*
 * Syntax error: neither ':' nor variable name found
 */
        case 0 :
          /* Ignore all chars to end of line */
          fscanf (file, "%*[^\n]");      
          pipsPrintError (-1,PipsVarOrColonExp,NULL,PipsDontExit);
          return (PipsVarOrColonExp);

/*
 * variable name found, must be followed by a ':'
 */
        case 1 :
          result = fscanf (file, " %1[:]", token);
          switch (result) {

/*
 * Syntax error: End of file instead of ':' found
 */
            case EOF :
              pipsPrintError (-1,PipsPrematureEof,NULL,PipsDontExit);
              return (PipsPrematureEof);

/*
 * Syntax error: No ':' after variable name found
 */
            case 0 :
              /* Ignore all chars to end of line */
              fscanf (file, "%*[^\n]");      
              pipsPrintError (-1,PipsColonExpected,NULL,PipsDontExit);
              return (PipsColonExpected);
          } /* end switch */
      } /* end switch */
  } /* end switch */


/******************************************************************************
 *   ' A T T R I B U T E N A M E  = '
 ******************************************************************************
 * Read attribute name (everything up to a '=')
 * it must follow the colon on the same line
 */
  if ((result = fscanf (file, "%1[\n]", token)) != 1)
    result = fscanf (file, " %[^= ]", attName);
  else
    result = 0;
  switch (result) {

/*
 * Syntax error: End of file instead of attribute name found
 */
    case EOF :
      pipsPrintError (-1, PipsPrematureEof, NULL, PipsDontExit);
      return (PipsPrematureEof);

/*
 * Syntax error: No attribute name found
 */
    case 0 :
      /* Ignore all chars to end of line */
      fscanf (file, "%*[^\n]");      
      pipsPrintError (-1, PipsAttNameExp, NULL, PipsDontExit);
      return (PipsAttNameExp);
  } /* end switch */

/*
 * Read '='
 */
  result = fscanf (file, " %1[=]", token);
  switch (result) {

/*
 * Syntax error: End of file instead of '=' found
 */
    case EOF :
      pipsPrintError (-1, PipsPrematureEof, NULL, PipsDontExit);
      return (PipsPrematureEof);

/*
 * Syntax error: '=' after attribute name missing
 */
    case 0 :
      /* Ignore all chars to end of line */
      fscanf (file, "%*[^\n]");      
      pipsPrintError (-1, PipsEqualSignExp, NULL, PipsDontExit);
      return (PipsEqualSignExp);
  } /* end switch */

/*
 * Finding a ';' now means 'empty attribute value'
 * which means, attribute has to be deleted
 */
  result = fscanf (file, " %1[;]", token);
  switch (result) {

/*
 * Syntax error: End of file instead of ';' found
 */
    case EOF :
      pipsPrintError (-1, PipsPrematureEof, NULL, PipsDontExit);
      return (PipsPrematureEof);

/*
 * Tell caller to delete current attribute
 */
    case 1 :
      /* Ignore all chars to end of line */
      fscanf (file, "%*[^\n]");
      *attNum = 0;
      return (PipsOk);
  } /* end switch */


/******************************************************************************
 *  ' V A L U E 1 , '   ' V A L U E 2 , '  . . .
 ******************************************************************************
 * Read a comma separated and semikolon terminated list of attribute values
 */
  do {

/*
 * Read a single attribute value, first try numerical, then character
 * Distinguish between:  0  0123  0x123  0.123  and any other number
 */
    if (fscanf (file, " %1[0]", token) == 1)          /* It starts with '0'  */
      if (fscanf (file, "%1[xX]", token) == 1) {      /* It starts with '0x' */
        result = fscanf (file, "%hx", &s);
        d = s;
      } else
        if (fscanf (file, "%1[.]", token) == 1)       /* It starts with '0.' */
          result = fscanf (file, " %lf", &d);
        else {
          if ((result = fscanf (file, "%ho", &s)) == 0) {  /* It's just a '0' */
            result = 1;       
            s = 0;
          }
          d = s;
        }
    else        /* It's another number or a character string */
      result = fscanf (file, " %lf", &d);

/*
 * Preset token[0] according to the estimated type of 'd'
 * and convert values to the variables s(hort), l(ong), f(loat)
 */
    s = (short) d;
    f = (float) d;
    l = (long)  d;
    if (d - ((long) d) <= DBL_MIN)    /* number has no fraction */
      token[0] = 'l';
    else
      token[0] = 'd';

    switch (result) {

/*
 * Syntax error: End of file before end of definition found
 */
      case EOF :
        pipsPrintError (-1, PipsPrematureEof,NULL,PipsDontExit);
        return (PipsPrematureEof);

/*
 * If numerical value found, then read optional suffix
 */
      case 1 :
        result = fscanf (file, " %1[fFdDlLsS]", token);

/*
 * Syntax error: End of file before end of definition found
 */
        if (result == EOF) {
          pipsPrintError (-1,PipsPrematureEof,NULL,PipsDontExit);
          return (PipsPrematureEof);
        } /* end if */

/*
 * Suffix found: Set type, if not set yet
 */
        if (*attType == NC_UNSPECIFIED)
          switch (token[0]) {
            case 'f' : 
            case 'F' : *attType = NC_FLOAT;
                       break;
            case 'd' :
            case 'D' : *attType = NC_DOUBLE;
                       break;
            case 'l' :
            case 'L' : *attType = NC_LONG;
                       break;
            case 's' :
            case 'S' : *attType = NC_SHORT;
          } /* end switch */

/*
 * Move scanned value to attribute value array (type dependent)
 */
        switch (*attType) {
          case NC_FLOAT :
                     memcpy (attValue, &f, sizeof (float));
                     attValue += sizeof (float);
                     (*attNum)++;
                     break;
          case NC_DOUBLE :
                     memcpy (attValue, &d, sizeof (double));
                     attValue += sizeof (double);
                     (*attNum)++;
                     break;
          case NC_LONG :
                     memcpy (attValue, &l, sizeof (long));
                     attValue += sizeof (long);
                     (*attNum)++;
                     break;
          case NC_SHORT :
                     memcpy (attValue, &s, sizeof (short));
                     attValue += sizeof (short);
                     (*attNum)++;
          default : ;
        } /* end switch */
        break;  /* case 1 : result ... */

/*
 * A numerical value has not been found:
 * expect "string constant" or 'characters'
 */
      case 0 :
        result = fscanf (file, " %1[\"] %[^\n\"] %1[\"]",
                         token, attValue, token);
        if (result == 0)
          result = fscanf (file, " %1['] %[^\n'] %1[']",
                           token, attValue, token);
        switch (result) {

/*
 * Syntax error: End of file before end of definition found
 */
          case EOF :
            pipsPrintError (-1,PipsPrematureEof,NULL,PipsDontExit);
            return (PipsPrematureEof);

/*
 * Syntax error: Attribute value was not recognized
 */
           case 0 :
           case 1 :
           case 2 :
             /* Ignore all chars to end of line */
             fscanf (file, "%*[^\n]");      
             pipsPrintError (-1,PipsAttValueExp,NULL,PipsDontExit);
             return (PipsAttValueExp);
           case 3 :
             *attType = NC_CHAR;
/*
 * Move pointer to end of string and increase item counter
 */
             (*attNum)+= strlen (attValue);
             attValue += strlen (attValue);
         } /* end switch */
    } /* end switch */

/*
 * Check for ',' as separator of multiple attribute values
 */
    result = fscanf (file, " %1[,]", token);
    switch (result) {

/*
 * Syntax error: End of file instead of ',' found
 */
    case EOF : pipsPrintError (-1, PipsPrematureEof, NULL, PipsDontExit);
               return (PipsPrematureEof);

/*
 * No comma found -> that should have been the last attribute value
 */
    case 0   : moreAttValues = 0;
    } /* end switch */

} while (moreAttValues);

/******************************************************************************
 *   R E A D   ' ; '   A S   D E F I N I T I O N   D E L I M I T E R 
 ******************************************************************************
 */
  result = fscanf (file, " %1[;]", token);
  switch (result) {

/*
 * Syntax error: End of file instead of ';' found
 */
  case EOF : pipsPrintError (-1, PipsPrematureEof, NULL, PipsDontExit);
             return (PipsPrematureEof);

/*
 * Syntax error: Definition is not delimited by a ';'
 */
  case 0   : /* Ignore all chars to end of line */
             fscanf (file, "%*[^\n]");      
             pipsPrintError (-1, PipsSemikolonExp, NULL, PipsDontExit);
             return (PipsSemikolonExp);
  } /* end switch */

/*
 * Ignore everything up to the next (newline)
 */
  fscanf (file, "%*[^\n]");

/*
 * 'attNum' has to be increased by 1 (for \0 character), if type is NC_CHAR
 */
  (*attNum) += (*attType == NC_CHAR);

  return (PipsOk);
} /* end pipsReadAttDef */


/* end pipsReadAttDef.c */
