/* glpchol/chol_numb.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <float.h>
#include <math.h>
#include <stddef.h>
#include "glpchol.h"

/*----------------------------------------------------------------------
-- chol_numb - compute Cholesky decomposition A = U'*U; numeric phase
--
-- *Synopsis*
--
-- #include "glpchol.h"
-- int chol_numb(MAT *A, MAT *U, void *head[], double work[]);
--
-- *Description*
--
-- The chol_numb routine implements numeric phase of the Cholesky
-- decomposition A = U'*U, where A is a given sparse symmetric positive
-- definite matrix, U is resultant upper triangular matrix, assuming
-- that the pattern of the matrix U (i.e. distribution of its non-zero
-- elements) is known. (Symbolic phase of this operation can be
-- performed by means of the chol_symb routine.)
--
-- So far as the matrix A is symmetric, on entry it should contain only
-- upper triangular part. This matrix remains unchanged on exit.
--
-- The resultant matrix U should not coincide with the matrix A. Order
-- of U should be equal to order of A. On entry distribution of non-zero
-- elements of U should be correct, however, numeric values of elements
-- are not essential. On exit the matrix U has the same pattern (some
-- its elements may be equal to zero because of numeric cancellation,
-- i.e. this matrix may contain explicit zeros).
--
-- Auxiliary arrays head and work should contain at least 1+n elements,
-- where n is order of the matrices A and U. If some of these parameters
-- is NULL, the chol_numb routine automatically allocates and frees the
-- corresponding working array(s).
--
-- *Returns*
--
-- The routine returns number of non-positive or very small diagonal
-- elements of the matrix U, which were replaced by huge positive number
-- (see the method description below). Zero return code means that the
-- input matrix is not very ill-conditioned.
--
-- *Method*
--
-- The chol_numb routine computes the matrix U in row-wise manner (see
-- also the description of the chol_symb routine). Before computing k-th
-- row the matrix U is the following:
--
--    1       k         n
-- 1  + + + + x x x x x x
--    . + + + x x x x x x
--    . . + + x x x x x x
--    . . . + x x x x x x
-- k  . . . . * * * * * *
--    . . . . . * * * * *
--    . . . . . . * * * *
--    . . . . . . . * * *
--    . . . . . . . . * *
-- n  . . . . . . . . . *
--
-- where '+' and 'x' denotes rows, which have been computed; '*' denotes
-- rows, which have been not processed yet.
--
-- It is well known that in order to numerically compute k-th row of the
-- matrix U it is necessary to copy to that row k-th row of the original
-- matrix A and subtract from it each i-th row of the matrix U (where
-- i = 1, 2, ..., k-1), multiplied by gaussian multiplier u[i,k] (if, of
-- course, u[i,k] is non-zero). Since U is upper triangular, we should
-- skip those elements of each i-th row, which are placed to the right
-- of the main diagonal.
--
-- In order to avoid scanning skipped elements of i-th row each time
-- when the row is processed, the routine uses ordered representation
-- of the matrix U, i.e. all elements of each row of U are ordered in
-- increasing their column indices (the routine automatically performs
-- ordering, so on entry the matrix U may be unordered). On k-th step
-- of the main loop the location head[i] (i = 1, 2, ..., k-1) points to
-- the sublist of elements of i-th row, whose column indices are not
-- less than k, because only these elements are needed in subsequent
-- operations (elements that are removed from sublists are marked on
-- the figure above by '+'; active elements that belong to sublists yet
-- are marked by '*'). Immediately after i-th row has been subtracted
-- from k-th row (i.e. u[i,k] != 0) the routine shifts the pointer
-- head[i] to the next row element, that automatically removes u[i,k]
-- from the sublist. This technique allows significantly improving the
-- performance.
--
-- If the input matrix A (which is assumed to be positive definite) is
-- close to semidefinite and therefore ill-conditioned, it may happen
-- that on some elimination step diagonal element u[k,k] is non-positive
-- or very small. So far as the routine doesn't use pivoting, it can't
-- permute rows and columns of U in order to choose more appropriate
-- diagonal element. Therefore the routine uses a technique proposed in
-- the paper:
--
-- S.J.Wright. The Cholesky factorization in interior-point and barrier
-- methods. Preprint MCS-P600-0596, Mathematics and Computer Science
-- Division, Argonne National Laboratory, Argonne, Ill., May 1996.
--
-- The routine just replaces non-positive or very small pivot element
-- u[k,k] by huge positive number. Such replacement involves the
-- off-diagonal elements in the k-th column of the matrix U to be close
-- to zero and causes the k-th conponent of the solution vector also to
-- be close to zero. Of course, this technique works only in the case
-- when the system A*x = b is consistent. */

int chol_numb(MAT *A, MAT *U, void *_head[], double _work[])
{     ELEM *e, *ee, **head = (ELEM **)_head;
      int n = A->m, i, j, k, ret = 0;
      double big, ukk, *work = _work;
      if (A == U)
         fault("chol_numb: invalid specification of resultant matrix");
      if (!(U->m == n && U->n == n && A->n == n))
         fault("chol_numb: inconsistent dimension");
      /* allocate and clear auxiliary arrays */
      if (_head == NULL) head = ucalloc(1+n, sizeof(ELEM *));
      for (i = 1; i <= n; i++) head[i] = NULL;
      if (_work == NULL) work = ucalloc(1+n, sizeof(double));
      for (j = 1; j <= n; j++) work[j] = 0.0;
      /* element lists of the matrix U should be ordered by increasing
         column indices */
      sort_mat(U);
      /* determine maximal absolute value of diagonal elements of the
         matrix A */
      big = DBL_EPSILON;
      for (i = 1; i <= n; i++)
      {  for (e = A->row[i]; e != NULL; e = e->row)
         {  if (e->i == e->j)
            {  if (big < fabs(e->val)) big = fabs(e->val);
            }
         }
      }
      /* main loop */
      for (k = 1; k <= n; k++)
      {  /* compute k-th row of the matrix U */
         /* work := k-th row of the matrix A */
         for (e = A->row[k]; e != NULL; e = e->row) work[e->j] = e->val;
         /* non-zeros in k-th column of U (except diagonal element)
            determine rows of U, which should be subtracted from k-th
            row */
         for (e = U->col[k]; e != NULL; e = e->col)
         {  i = e->i;
            assert(i <= k);
            if (i == k) continue; /* skip diagonal element u[k,k] */
            /* u[i,k] != 0, therefore it should be the first element in
               the sublist of i-th row */
            assert(head[i] != NULL && head[i]->j == k);
            /* work := work - f * (i-th row of U), where f = u[i,k] is
               gaussian multiplier */
            for (ee = head[i]; ee != NULL; ee = ee->row)
               work[ee->j] -= e->val * ee->val;
            /* remove u[i,k] from the sublist of i-th row */
            head[i] = head[i]->row;
         }
         /* determine pivot element u[k,k] */
         ukk = work[k];
         /* if the matrix A is close to semidefinite and therefore
            ill-conditioned, it may happen that u[k,k] is non-positive
            or very small due to round-off errors; in this case such
            pivot element is just replaced by huge positive number */
         if (ukk < (DBL_EPSILON * DBL_EPSILON) * big)
         {  ret++;
            ukk = work[k] = 0.1 * DBL_MAX;
         }
         /* now the array work contains elements of k-th row of the
            matrix U; however, k-th row should be normalized by dividing
            all its elements on sqrt(u[k,k]) */
         ukk = sqrt(ukk);
         for (e = U->row[k]; e != NULL; e = e->row)
            e->val = work[e->j] / ukk, work[e->j] = 0.0;
         /* if the pattern of U is correct, all elements of the array
            work now should be equal to zero */
         /* initialize the pointer to the sublist of k-th row (each row
            list starts from the diagonal element u[k,k], therefore this
            element should be skipped before) */
         assert(U->row[k] != NULL && U->row[k]->j == k);
         head[k] = U->row[k]->row;
      }
      /* free auxiliary arrays */
      if (_head == NULL) ufree(head);
      if (_work == NULL) ufree(work);
      /* return to the calling program */
      return ret;
}

/* eof */
