/* glpmat/clear_line.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- clear_line - clear row or column of sparse matrix.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *clear_line(MAT *A, int k);
--
-- *Description*
--
-- The clear_line routine performs one of the following:
--
-- If k > 0, the routine clears k-th row of the sparse matrix A.
--
-- If k < 0, the routine clears k-th column of the sparse matrix A.
--
-- If k = 0, the routine clears entire matrix A.
--
-- Should note that in any case this operation requires the time O(nz),
-- where nz is number of non-zeros of the matrix A.
--
-- *Returns*
--
-- The clear_line routine returns a pointer to the matrix A. */

MAT *clear_line(MAT *A, int k)
{     ELEM *e, *en; int i, j;
      if (k > 0)
      {  /* clear k-th row of the matrix */
         i = +k;
         if (!(1 <= i && i <= A->m))
            fault("clear_line: row number out of range");
         for (e = A->row[i]; e != NULL; e = en)
         {  en = e->row;
            if (A->col[e->j] == e)
               A->col[e->j] = e->col;
            else
            {  ELEM *f;
               for (f = A->col[e->j]; f != NULL; f = f->col)
                  if (f->col == e) break;
               assert(f != NULL);
               f->col = e->col;
            }
            free_atom(A->pool, e);
         }
         A->row[i] = NULL;
      }
      else if (k < 0)
      {  /* clear k-th column of the matrix */
         j = -k;
         if (!(1 <= j && j <= A->n))
            fault("clear_line: column number out of range");
         for (e = A->col[j]; e != NULL; e = en)
         {  en = e->col;
            if (A->row[e->i] == e)
               A->row[e->i] = e->row;
            else
            {  ELEM *f;
               for (f = A->row[e->i]; f != NULL; f = f->row)
                  if (f->row == e) break;
               assert(f != NULL);
               f->row = e->row;
            }
            free_atom(A->pool, e);
         }
         A->col[j] = NULL;
      }
      else /* k == 0 */
      {  /* clear entire matrix */
         clear_mat(A);
      }
      return A;
}

/* eof */
