/* glpmat/mprd_symb.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- mprd_symb - multiply matrices (C := A * B); symbolic phase
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *mprd_symb(MAT *C, MAT *A, MAT *B, char work[]);
--
-- *Description*
--
-- The mprd_symb routine computes the product C = A * B symbolically,
-- i.e. it only determines distribution of non-zeros of the matrix C not
-- computing their numeric values.
--
-- Input matrices A and B remain unchanged on exit. Number of columns of
-- the matrix A should be equal to number of rows of the matrix B.
-- Numeric values of elements of these matrices are ignored (i.e. they
-- all are considered as non-zeros).
--
-- The resultant matrix C should not coincide neither with matrix A nor
-- with matrix B. Number of rows of C should be equal number of rows of
-- A, number of columns of C should be equal number of columns of B. On
-- entry the contents of the matrix C is ignored.
--
-- The auxiliary array work should contain at least 1+n elements, where
-- n is number of columns of the matrix C. If this parameter is NULL,
-- the mprd_symb routine automatically allocates and frees the working
-- array.
--
-- *Returns*
--
-- The mprd_symb routine returns a pointer to the matrix C. */

MAT *mprd_symb(MAT *C, MAT *A, MAT *B, char _work[])
{     ELEM *e, *ee;
      int i, j, k;
      char *work = _work;
      if (C == A || C == B)
         fault("mprd_symb: invalid specification of resultant matrix");
      if (!(C->m == A->m && A->n == B->m && B->n == C->n))
         fault("mprd_symb: inconsistent dimension; product undefined");
      clear_mat(C);
      if (_work == NULL) work = ucalloc(1+C->n, sizeof(char));
      for (j = 1; j <= C->n; j++) work[j] = 0;
      for (i = 1; i <= C->m; i++)
      {  /* compute i-th row of C */
         for (e = A->row[i]; e != NULL; e = e->row)
         {  k = e->j;
            for (ee = B->row[k]; ee != NULL; ee = ee->row)
            {  j = ee->j;
               /* a[i,k] != 0 and b[k,j] != 0 therefore c[i,j] != 0 */
               if (work[j] == 0) new_elem(C, i, j, 1.0), work[j] = 1;
            }
         }
         /* clear working array */
         for (e = C->row[i]; e != NULL; e = e->row) work[e->j] = 0;
      }
      if (_work == NULL) ufree(work);
      return C;
}

/* eof */
