/* glprsm/rsm_exact_dzeta.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glprsm.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- rsm_exact_dzeta - compute exact weight (dual steepest edge).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double rsm_exact_dzeta(RSM *rsm, int i);
--
-- *Description*
--
-- The rsm_exact_dzeta routine computes exact value of the weight at
-- the basis variable xB[i] for the current basis using directly the
-- formula:
--
--    dzeta[i] = 1 + SUM alfa[i,j]^2, i = 1, ..., m,
--                    j
--
-- where alfa[i,j] is the element of the current simplex table placed
-- in i-th row and j-th column.
--
-- This operation is extremely inefficient, therefore it may be used
-- only for debugging purposes.
--
-- *Returns*
--
-- The rsm_exact_dzeta routine returns the computed value of the weight
-- dzeta[i]. */

double rsm_exact_dzeta(RSM *rsm, int i)
{     ELEM *e;
      int m = rsm->m, n = rsm->n, j, k;
      double *bi, *ai, t;
      assert(1 <= i && i <= m);
      /* compute i-th row of the inverse inv(B) */
      bi = ucalloc(1+m, sizeof(double));
      bi[i] = 1.0;
      rsm_btran(rsm, bi);
      /* compute i-th row of the simplex table */
      ai = ucalloc(1+n, sizeof(double));
      for (j = 1; j <= n; j++)
      {  k = rsm->indn[j]; /* x[k] = xN[j] */
         ai[j] = 0.0;
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            ai[j] -= e->val * bi[e->i];
      }
      /* compute dzeta[i] */
      t = 1.0;
      for (k = 1; k <= n; k++) t += ai[k] * ai[k];
      ufree(bi), ufree(ai);
      return t;
}

/* eof */
