/* glpimg.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPIMG_H
#define _GLPIMG_H

#define create_img            _glp_create_img
#define delete_img            _glp_delete_img
#define get_imgval            _glp_get_imgval
#define get_pixel             _glp_get_pixel
#define line_to               _glp_line_to
#define move_to               _glp_move_to
#define reset_rgb             _glp_reset_rgb
#define save_as_bmp           _glp_save_as_bmp
#define save_as_tiff          _glp_save_as_tiff
#define save_img              _glp_save_img
#define set_clipreg           _glp_set_clipreg
#define set_color             _glp_set_color
#define set_imgval            _glp_set_imgval
#define set_logorg            _glp_set_logorg
#define set_pixel             _glp_set_pixel
#define set_viewport          _glp_set_viewport

typedef struct IMG IMG;
typedef struct RGB RGB;

struct IMG
{     /* raster image */
      int type;
      /* image type: */
#define IMG_2     0 /* 2 colors (1 bit per pixel) */
#define IMG_4     1 /* 4 colors (2 bits per pixel) */
#define IMG_16    2 /* 16 colors (4 bits per pixel) */
#define IMG_256   3 /* 256 colors (8 bits per pixel) */
      RGB *rgb; /* RGB rgb[2, 4, 16, or 256]; */
      /* color map table */
      int xres, yres;
      /* horizontal and vertical resolution (in pixels per meter) */
      int xsize, ysize;
      /* width and height of the image (in pixels) */
      int len;
      /* length of each horizontal scan line (in bytes) */
      unsigned char *scan; /* unsigned char scan[ysize*len]; */
      /* array of scan lines */
      int color;
      /* current color index */
      int x0, y0;
      /* absolute coordinates of the base point (it may be placed out
         of the image or the clipping region) */
      int x1, y1, x2, y2;
      /* absolute coordinates of the rectangular clipping region */
      int x, y;
      /* absolute coordinates of the current point (it may be placed
         out of the image or the clipping region) */
};

struct RGB
{     /* color map entry */
      unsigned char red;
      /* red intensity (0-255) */
      unsigned char green;
      /* green intensity (0-255) */
      unsigned char blue;
      /* blue intensity (0-255) */
};

/* default color indices: */
#define IMG_BLACK          0x00
#define IMG_BLUE           0x01
#define IMG_GREEN          0x02
#define IMG_CYAN           0x03
#define IMG_RED            0x04
#define IMG_MAGENTA        0x05
#define IMG_BROWN          0x06
#define IMG_LIGHTGRAY      0x07
#define IMG_DARKGRAY       0x08
#define IMG_LIGHTBLUE      0x09
#define IMG_LIGHTGREEN     0x0A
#define IMG_LIGHTCYAN      0x0B
#define IMG_LIGHTRED       0x0C
#define IMG_LIGHTMAGENTA   0x0D
#define IMG_YELLOW         0x0E
#define IMG_WHITE          0x0F

/*** base routines ***/

extern IMG *create_img(int type, int xsize, int ysize);
/* create raster image */

extern void reset_rgb(IMG *img);
/* initialize color map table by default spectrum */

extern int get_imgval(IMG *img, int x, int y);
/* get current value of pixel */

extern void set_imgval(IMG *img, int x, int y, int val);
/* set pixel to given value */

extern void delete_img(IMG *img);
/* delete raster image */

extern int save_as_bmp(IMG *img, char *fname);
/* write raster image using Windows Bitmap format */

extern int save_as_tiff(IMG *img, char *fname);
/* write raster image using TIFF format */

extern int (*save_img)(IMG *img, char *fname);
/* write raster image using default format */

/*** high-level routines ***/

extern void set_color(IMG *img, int color);
/* change current color index */

extern void set_logorg(IMG *img, int x, int y);
/* change base point */

extern void set_clipreg(IMG *img, int x1, int y1, int x2, int y2);
/* change clipping region */

extern void set_viewport(IMG *img, int x1, int y1, int x2, int y2);
/* change view port */

extern int get_pixel(IMG *img, int x, int y);
/* get color index of pixel */

extern void set_pixel(IMG *img, int x, int y);
/* set pixel to current color index */

extern void move_to(IMG *img, int x, int y);
/* change current point */

extern void line_to(IMG *img, int x, int y);
/* generate straightforward segment */

#endif

/* eof */
