/* glplpx8b.c */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glplang.h"
#include "glplib.h"
#include "glplpx.h"
#include "glpmat.h"

/*----------------------------------------------------------------------
-- lpx_read_lpm - read LP/MIP model written in GLPK/L.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- LPX *lpx_read_lpm(char *infile, char *outfile);
--
-- *Description*
--
-- The routine lpx_read_lpm reads LP/MIP model written in the modeling
-- language GLPK/L from a text file whose name is the character string
-- infile.
--
-- If the parameter outfile is not NULL, the routine *writes* generated
-- LP/MIP problem to a text file whose name is outfile.
--
-- For detailed description of GLPK/L modeling language see the package
-- documentation.
--
-- *Returns*
--
-- If no error occurred, the routine returns a pointer to the created
-- problem object. Otherwise the routine returns NULL. */

LPX *lpx_read_lpm(char *infile, char *outfile)
{     LPX *lp = NULL;
      struct prob *prob = NULL;
      MAT *A = NULL;
      int m, n, i, j;
      /* initialize the language processor environment */
      if (initialize(infile) != 0) goto fail;
      /* set error handling */
      pdb->flag = 1;
      if (setjmp(pdb->jump)) goto fail;
      /* parse model description */
      load_model();
      /* write generated LP/MIP problem to the specified file in plain
         text format */
      if (outfile != NULL)
      {  int ret;
         ret = gener_lp(outfile);
         if (ret != 0) goto fail;
      }
      /* create data structure for generating LP/MIP */
      prob = create_prob();
      m = prob->m;
      n = prob->n;
      /* create problem object */
      lp = lpx_create_prob();
      lpx_set_prob_name(lp, pdb->model_name);
      /* create columns that correspond to model variables */
      if (n > 0) lpx_add_cols(lp, n);
      for (j = 1; j <= n; j++)
      {  VAR *var = prob->memb[m+j]->link;
         char *name = gener_name(prob, m+j);
         int typx;
         lpx_set_col_name(lp, j, name);
         if (var->kind)
         {  lpx_set_class(lp, LPX_MIP);
            lpx_set_col_kind(lp, j, LPX_IV);
         }
         switch (var->type)
         {  case 'F': typx = LPX_FR; break;
            case 'L': typx = LPX_LO; break;
            case 'U': typx = LPX_UP; break;
            case 'D': typx = LPX_DB; break;
            case 'S': typx = LPX_FX; break;
            default: insist(var->type != var->type);
         }
         lpx_set_col_bnds(lp, j, typx, var->lb, var->ub);
      }
      /* create rows that correspond to model constraints; create the
         constraint matrix */
      if (m > 0) lpx_add_rows(lp, m);
      A = create_mat(m, n);
      for (i = 1; i <= m; i++)
      {  CONS *cons = prob->memb[i]->link;
         char *name = gener_name(prob, i);
         struct elem *form, *e;
         int typx;
         lpx_set_row_name(lp, i, name);
         if (prob->obj_row == i) lpx_set_obj_name(lp, name);
         form = build_form(prob, i);
         if (form == NULL) goto fail;
         for (e = form; e != NULL; e = e->next)
         {  if (e->j == 0)
            {  if (cons->type == 'F')
               {  print("lpx_read_lpm: free row `%s' has constant term",
                     name);
                  goto fail;
               }
               cons->lb -= e->val, cons->ub -= e->val;
            }
            else if (e->val != 0.0)
               new_elem(A, i, e->j, e->val);
            /* set coefficient of the objective function */
            if (prob->obj_row == i)
               lpx_set_col_coef(lp, e->j, e->val);
         }
         erase_form(prob, form);
         switch (cons->type)
         {  case 'F': typx = LPX_FR; break;
            case 'L': typx = LPX_LO; break;
            case 'U': typx = LPX_UP; break;
            case 'D': typx = LPX_DB; break;
            case 'S': typx = LPX_FX; break;
            default: insist(cons->type != cons->type);
         }
         lpx_set_row_bnds(lp, i, typx, cons->lb, cons->ub);
      }
      /* load the constraint matrix into the problem object */
      {  int nz = count_nz(A, 0);
         int *rn = ucalloc(1+nz, sizeof(int));
         int *cn = ucalloc(1+nz, sizeof(int));
         double *a = ucalloc(1+nz, sizeof(double));
         int k = 0;
         ELEM *e;
         for (i = 1; i <= m; i++)
         {  for (e = A->row[i]; e != NULL; e = e->row)
            {  k++;
               insist(k <= nz);
               rn[k] = e->i, cn[k] = e->j, a[k] = e->val;
            }
         }
         insist(k == nz);
         lpx_load_mat3(lp, nz, rn, cn, a);
         ufree(rn);
         ufree(cn);
         ufree(a);
      }
      delete_mat(A);
      /* set optimization direction */
      switch (prob->obj_dir)
      {  case '-': lpx_set_obj_dir(lp, LPX_MIN); break;
         case '+': lpx_set_obj_dir(lp, LPX_MAX); break;
         default: insist(prob->obj_dir != prob->obj_dir);
      }
      /* free auxiliary data structure */
      delete_prob(prob);
      /* terminate the language processor environment */
      terminate();
      /* model has been read successfully */
      return lp;
fail: /* the operation failed */
      print("lpx_read_lpm: processing terminated due to errors");
      if (lp != NULL) lpx_delete_prob(lp);
      if (prob != NULL) delete_prob(prob);
      if (pdb != NULL) terminate();
      if (A != NULL) delete_mat(A);
      return NULL;
}

/* eof */
