<?PHP
/*
	Helper functions for WPMU
	4th March 2005
	
	There are 2 types of helper functions; those that use hooks (do_action, apply_filters hooks) and those that support WPMU the Administration plugin.
	
	Where possible - existing hooks have been used, but occasionally - extra hooks have been implemented in the WP Core.
*/

/**
 * Retrieves a list of client blogs
 *
 * @return array
 */
function wpmu_getBlogs(){
	$blogs = array();
	
	$link = opendir(ABSPATH . 'wp-content/blogs/');
	while(false !== ($dir = readdir($link))){
		if(is_dir(ABSPATH . 'wp-content/blogs/' . $dir)) {
			switch($dir){
				case '.':
				case '..':
				case 'DEFAULT': 
				break;
				case 'main':  
					  array_unshift($blogs, $dir);
				break;
				
				default:	
					$blogs[] = $dir;
				break;
			}
		}
	}
	
	closedir($link);
	
	return $blogs;
}


// Cache clearing
/**
 * Clears the smarty cache when post/comments occur
 *
 */
function wpmu_clear_cache() {
	global $wpsmarty;
	$wpsmarty->clear_all_cache();
	$wpsmarty->clear_compiled_tpl();
}
add_action('comment_post', 'wpmu_clear_cache');
add_action('delete_post', 'wpmu_clear_cache');
add_action('delete_comment', 'wpmu_clear_cache');
add_action('private_to_published', 'wpmu_clear_cache');
add_action('publish_phone', 'wpmu_clear_cache');
add_action('publish_post', 'wpmu_clear_cache');
add_action('trackback_post', 'wpmu_clear_cache');
add_action('wp_set_comment_status', 'wpmu_clear_cache');

/**
 *	Cleans the passed path.
 *	Removes the wpmu & the blog name from the path
 *	This should only be used for display filters
 *
 * @param unknown $path
 * @return unknown
 */
function wpmu_tidypath($path) {
	global $wpblog;
	
	$path = str_replace('/wpmu/','/',$path);
	$path = str_replace("/{$wpblog}/",'/',$path);
	
	return $path;
}
add_filter('wpmu_display_get_page_string','wpmu_tidypath',10);
add_filter('wpmu_display_get_permalink','wpmu_tidypath',10);

/**
 * Wrapper for querying the databse
 *
 * @param string $query
 * @return object
 */
function wpmu_adminDatabaseQuery($query) {
	/* @var $wpdb wpdb */
	global $wpdb;
	
	$result = $wpdb->get_row($query);
	
	return $result;
}

/**
 * Set an option in the Master Blog 'main' options table
 *
 * @param string $optionName
 * @param mixed $optionValue
 */
function wpmu_adminOptionSet($optionName, $optionValue) {
	wpmu_blogOption_set('main',$optionName, $optionValue);
}

/**
 * Retrieve a setting from the Master Blog Tables
 *
 * @param string $optionName
 * @param mixed $default
 * @return mixed
 */
function wpmu_adminOption_get($optionName, $default=null) {
	return wpmu_blogOption_get('main',$optionName, $default);
}

/**
 * Retrives a setting from a specific client blog
 *
 * @param string $blogName
 * @param string $optionName
 * @param mixed $default
 * @return mixed
 */
function wpmu_blogOption_get($blogName, $optionName, $default=null) {
 	global $wpdb;
 	
	// check if options table exists
	if( defined('WP_INSTALLING') )
	    return null;

	$query = wpmu_adminDatabaseQuery("SELECT * FROM ".$wpdb->options." WHERE option_name = '{$optionName}'");
	if (!isset($query) || is_null($query)) {
		$result = $default;
	} else {
		$result = $query->option_value;
		if (!(unserialize($result)===false)) {
			$result = unserialize($result);
		}
	}
		
	return $result;
}


/**
 * Sets an option for a specific client blog
 *
 * @param string $blogName
 * @param string $optionName
 * @param mixed $optionValue
 */
function wpmu_blogOption_set($blogName, $optionName, $optionValue) {
	
 	global $wpdb;
 	
 	$isExisting = wpmu_blogOption_get($blogName,$optionName,null);
 	
 	if (is_array($optionValue) || is_object($optionValue)) {
 		$optionValue = serialize($optionValue);
 	}
 	
	if (is_null($isExisting)) {
		wpmu_adminDatabaseQuery("INSERT INTO ".$wpdb->options." (option_name, option_value) values ('{$optionName}', '{$optionValue}')");
	} else {
		wpmu_adminDatabaseQuery("UPDATE ".$wpdb->options." set option_value = '{$optionValue}' where option_name = '{$optionName}'");
	}
}

/*
  Determines if the available space defined by the admin has been exceeded by the user
*/
/**
 * Returns how much space is available (also shows a picture) for the current client blog, retrieving the value from the master blog 'main' option table
 *
 * @param string $action
 * @return string
 */
function wpmu_checkAvailableSpace($action) {
	// Using the action.
	// Set the action to 'not-writable' to block the upload
	global $wpblog;
	
	// Default space allowed is 10 MB 
	$spaceAllowed = wpmu_adminOption_get("wpmu_space_allocated", 10485760 );
	
	$dirName = ABSPATH."wp-content/blogs/{$wpblog}/images/";
	
  	$dir  = dir($dirName);
   	$size = 0;

	while($file = $dir->read()) {
		if ($file != '.' && $file != '..') {
			if (is_dir($file)) {
	           $size += dirsize($dirName . '/' . $file);
	       } else {
	           $size += filesize($dirName . '/' . $file);
	       }
	   }
	}
	$dir->close();
	
	?>	
	<table align="center" width="20%" cellpadding="0" cellspacing="0">
	<tr>
	<td>Space Available (<?php print $spaceAllowed-$size ?><i>bytes)</i></td>
	</tr>
	<tr>
	<td bgcolor="<?php echo ((($size/$spaceAllowed)*100)<70)?"Green":"Red"; ?>">&nbsp;</td><td bgcolor="Black" width="<?php echo (($size/$spaceAllowed)*100); ?>%"></td>
	</tr>
	</table>
	<?
	
	if (($spaceAllowed-$size)>0) {
		return $action;
	} else {
		// No space left
		return 'not-writable';	
	}
}
add_filter('fileupload_init','wpmu_checkAvailableSpace');


/**
 *  Retrieves a list of all plugins - and flags those that have been enabled
 *
 * @param array $args ByRef Array of plugins
 */
function wpmu_allBlogPlugins_get($args) {
	
	$allPlugins = get_plugins();
	
	$enabledPlugins = wpmu_adminOption_get('wpmu_global_blog_plugins',array());
	$enabledPlugins = unserialize($enabledPlugins);
	if (!is_array($enabledPlugins)) {
		$enabledPlugins = array();
	}
	
	foreach ($enabledPlugins as $thisPlugin) {
		if (isset($allPlugins[$thisPlugin])) {
			$allPlugins[$thisPlugin]['enabled']='1';
		}		
	}
	
	$args[] = array(
					'caption'=>'Plugins Enabled',
					'name'=>'wpmu_global_blog_plugins',
					'value'=>$allPlugins,
					'type'=>'checkbox'
					);	
}
add_action('wpmu_options_admin_get','wpmu_allBlogPlugins_get');

/**
 * Sets global plugins for all blogs
 *
 * @param array $args The plugins to configure
 */
function wpmu_allBlogPlugins_set($args) {
	
	// obtain the list of Admin Plugins to work with
	$activePlugins = $args['wpmu_global_blog_plugins'];
	
	// Process the array - turn it into something that is similar to the active_plugins array
	// and work out if the plugin has an installation phase
	$adminPlugins = array();
	$pluginsThatNeedInstalling = array();
	foreach ($activePlugins as $filename => $value) {
		$adminPlugins[] = $filename;
		// Check if the plugin has an install stage - by looking for AutoInstall in the plugin header
		if (wpmu_checkPluginForInstallStages($filename)) {
			$pluginsThatNeedInstalling[$filename]=true;
		} else {
			//
		}
	}

	// Store the setting first in the main blog settings for wpmu
	wpmu_adminOptionSet('wpmu_global_blog_plugins',serialize($adminPlugins));
	
	// Now rotate through the client blogs and apply the setting
	$allBlogs = wpmu_getBlogs();
	foreach ($allBlogs as $thisBlog) {
		// Exclude the 'main' blog
		if ($allBlogs!='main') {		
			$currentClientPlugings = wpmu_blogOption_get($thisBlog,'active_plugins',array());
					
			if (!(unserialize($currentClientPlugings)===false)) {
				$currentClientPlugings = unserialize($currentClientPlugings);
			}
			
			// Deal with the case when we don't get back a serialized array
			if (!is_array($currentClientPlugings)) {
				$currentClientPlugings = array();
			}
						
			// Work through the currently setup plugins for this blog
			foreach ($adminPlugins as $thisAdminPlugin) {
					// If the client blog doesn't already has the plugin
					if (!in_array($thisAdminPlugin, $currentClientPlugings)) {
						$currentClientPlugings[] = $thisAdminPlugin;
						if (isset($pluginsThatNeedInstalling[$thisAdminPlugin])) {
							$clientPluginsThatNeedInstalling[] = $thisAdminPlugin;
						} else {
							//
						}
					} else {
						//
					}
			}
			
			// Store the new plugins for this blog
			wpmu_blogOption_set($thisBlog,'active_plugins',serialize($currentClientPlugings));
			if (count($clientPluginsThatNeedInstalling)>0) {
				wpmu_blogOption_set($thisBlog,'wpmu_plugins_pending',serialize($clientPluginsThatNeedInstalling));
			}		
		}
	}
}
add_action('wpmu_options_admin_set','wpmu_allBlogPlugins_set');

/**
 * Determines if a plugin contains auto-install instructions
 *
 * @param string $plugin_file
 * @return string|false Returns the install instructions or false
 */
function wpmu_checkPluginForInstallStages($plugin_file) {
	
	if (file_exists(ABSPATH.'wp-content/plugins/'.$plugin_file)) {
		$plugin_file = ABSPATH.'wp-content/plugins/'.$plugin_file;
	} else {
		die("{$plugin_file}");
	}		
	
	$plugin_data = implode('', file($plugin_file));
	preg_match("|AutoInstall:(.*)|i", $plugin_data, $pluginAutoInstall);
	
	if (is_array($pluginAutoInstall)) {
		$pluginAutoInstall = $pluginAutoInstall[0];
	} else {
		$pluginAutoInstall = "";		
	}
		
	if (isset($pluginAutoInstall) && !is_null($pluginAutoInstall)) {
		$pluginAutoInstall = str_replace('AutoInstall:','',$pluginAutoInstall);
		$pluginAutoInstall = str_replace('<code>','',$pluginAutoInstall);
		$pluginAutoInstall = str_replace('</code>','',$pluginAutoInstall);
		return $pluginAutoInstall;
	}
	return false;
}

/**
 * Evaluates an expression
 *
 * @param unknown $string
 * @return unknown
 */
function wpmu_eval($string) {
	
	$result = eval($string);
		
	if ($result===false || is_null($result)) {
		return false;
	}
	return true;
}

/**
 * Auto installs plugins
 *
 */
function wpmu_autoInstallPlugins($pendingPlugins) {
	global $wpblog;
			
	foreach ($pendingPlugins as $key => $filename) {
		$installOptions = wpmu_checkPluginForInstallStages($filename);
		if ($installOptions) {
			// Include the file
			require_once(ABSPATH.'wp-content/plugins/'.$filename);
			$result = wpmu_eval($installOptions);
		} else {
			$result = true;
		}
		if ($result==true) {
			// Everything worked ok
			unset($pendingPlugins[$key]);
		} else {
			// There was a problem - should be logged
		}
	}
	wpmu_blogOption_set($wpblog,'wpmu_plugins_pending',serialize($pendingPlugins));
}

/*
 Determine if there are any plugins that need installing or configuring
 This will enable not just the auto install/configure of plugins - but updates (to plugins or plugin data) as well
*/
$pendingPlugins = wpmu_blogOption_get($wpblog,'wpmu_plugins_pending', null);

if (is_null($pendingPlugins)) {
	// If null is returned - then this is a brand new blog
	$adminPlugins = wpmu_adminOption_get('wpmu_global_blog_plugins',null);
	if (!is_null($adminPlugins)) {
		// We've something to do
		wpmu_autoInstallPlugins($adminPlugins);
	}
} else {
	if (count($pendingPlugins)>0) {
		wpmu_autoInstallPlugins($pendingPlugins);
	}
}



?>
