/* hpojip-test -- HP OfficeJet image-processing test application. */

/* Copyright (C) 2001-2002 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */


#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include "hpojip.h"


static IP_IMAGE_TRAITS defTraits = {
	640,		/* pixels per row */
	24,		/* bits per pixel */
	3,		/* components per pixel */
	300<<16,	/* horiz dpi, 16.16 */
	300<<16,	/* vert dpi, 16.16 */
	-1,		/* num rows (-1 = unknown) */
	1,		/* num pages */
	1		/* page num */
};

#define PRINTF(args...) fprintf(stderr,args)

#define NEW_XFORM(e) \
	do { \
		if (nXforms>=IP_MAX_XFORMS) { \
			PRINTF("Exceeded maximum number of xforms (%d)!\n\n", \
				IP_MAX_XFORMS); \
			goto syntaxError; \
		} \
		pXform=xforms+(nXforms++); \
		pXform->eXform=(e); \
	} while(0)

int parseFixedPoint(char *s,int fractionBits) {
	int whole=0,fraction=0;

	if (s[0]=='0' && (s[1]=='x' || s[1]=='X')) return strtol(s,0,0);

	sscanf(s,"%d.%d",&whole,&fraction);
	return ((whole<<fractionBits)|fraction);
}

#define LEN_BUFFER 4096

int main(int argc, char *argv[]) {
	char *inFilename=0,*outFilename=0;
	IP_HANDLE hJob=0;
	IP_XFORM_SPEC xforms[IP_MAX_XFORMS],*pXform=0;
	BYTE inBuffer[LEN_BUFFER],outBuffer[LEN_BUFFER];
	WORD wResult;
	DWORD dwInputAvail, dwInputUsed, dwInputNextPos=0;
	DWORD dwOutputUsed,dwOutputThisPos;
	int retcode=1,nXforms=0,inFile=-1,outFile=-1,i,totalBytes=0;
	int mirrorIn=0,mirrorOut=0;

	memset(xforms,0,sizeof(xforms));

    while (42) {
	argc--; argv++; if (argc<=0) break;

	if (**argv!='-') {
		if (!inFilename) {
			inFilename=*argv;
		} else if (!outFilename) {
			outFilename=*argv;
		} else {

			goto syntaxError;
		}

	} else if ((*argv)[1]=='-') {
		if (!pXform) {
			PRINTF("Got --<x> before -<x>!\n\n");
			goto syntaxError;
		}
		switch (pXform->eXform) {
		   case X_FAX_ENCODE:		goto x_fax_encode;
		   case X_FAX_DECODE:		goto x_fax_decode;
		   case X_PCX_ENCODE:		goto x_pcx_encode;
		   case X_PCX_DECODE:		goto x_pcx_decode;
		   case X_JPG_ENCODE:		goto x_jpg_encode;
		   case X_JPG_DECODE:		goto x_jpg_decode;
		   case X_JPG_FIX:		goto x_jpg_fix;
		   case X_TIF_ENCODE:		goto x_tif_encode;
		   case X_TIF_DECODE:		goto x_tif_decode;
		   case X_PNM_ENCODE:		goto x_pnm_encode;
		   case X_PNM_DECODE:		goto x_pnm_decode;
		   case X_SCALE:		goto x_scale;
		   case X_GRAY_2_BI:		goto x_gray_2_bi;
		   case X_BI_2_GRAY:		goto x_bi_2_gray;
		   case X_CNV_COLOR_SPACE:	goto x_cnv_color_space;
		   case X_Y_EXTRACT:		goto x_y_extract;
		   case X_THUMB:		goto x_thumb;
		   case X_TABLE:		goto x_table;
		   case X_CROP:			goto x_crop;
		   case X_TONEMAP:		goto x_tonemap;
		   case X_SATURATION:		goto x_saturation;
		   case X_ROTATE:		goto x_rotate;
		   case X_PAD:			goto x_pad;
		   case X_FAKE_MONO:		goto x_fake_mono;
		   case X_GRAYOUT:		goto x_grayout;
		   case X_CHANGE_BPP:		goto x_change_bpp;
		   case X_MATRIX:		goto x_matrix;
		   case X_CONVOLVE:		goto x_convolve;
		   case X_INVERT:		goto x_invert;
		   case X_SKEL:			goto x_skel;
		}

		goto syntaxError;

	} else if (!strcmp(*argv,"-defwidth")) {
		argc--; argv++; if (argc<=0) goto syntaxError;
		defTraits.iPixelsPerRow=strtol(*argv,0,0);

	} else if (!strcmp(*argv,"-defbpp")) {
		argc--; argv++; if (argc<=0) goto syntaxError;
		defTraits.iBitsPerPixel=strtol(*argv,0,0);

	} else if (!strcmp(*argv,"-defcpp")) {
		argc--; argv++; if (argc<=0) goto syntaxError;
		defTraits.iComponentsPerPixel=strtol(*argv,0,0);

	} else if (!strcmp(*argv,"-defhdpi")) {
		argc--; argv++; if (argc<=0) goto syntaxError;
		defTraits.lHorizDPI=parseFixedPoint(*argv,16);

	} else if (!strcmp(*argv,"-defvdpi")) {
		argc--; argv++; if (argc<=0) goto syntaxError;
		defTraits.lVertDPI=parseFixedPoint(*argv,16);

	} else if (!strcmp(*argv,"-defheight")) {
		argc--; argv++; if (argc<=0) goto syntaxError;
		defTraits.lNumRows=strtol(*argv,0,0);

	} else if (!strcmp(*argv,"-defpages")) {
		argc--; argv++; if (argc<=0) goto syntaxError;
		defTraits.iNumPages=strtol(*argv,0,0);

	} else if (!strcmp(*argv,"-defpagenum")) {
		argc--; argv++; if (argc<=0) goto syntaxError;
		defTraits.iPageNum=strtol(*argv,0,0);

	} else if (!strcmp(*argv,"-mirrorin")) {
		mirrorIn=1;

	} else if (!strcmp(*argv,"-mirrorout")) {
		mirrorOut=1;


	} else if (!strcmp(*argv,"-encfax")) {
		NEW_XFORM(X_FAX_ENCODE);
		pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MH;
		continue;

x_fax_encode:
		if (!strcmp(*argv,"--mh")) {
			pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MH;

		} else if (!strcmp(*argv,"--mr")) {
			pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MR;

		} else if (!strcmp(*argv,"--mmr")) {
			pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MMR;

		} else if (!strcmp(*argv,"--noeol")) {
			pXform->aXformInfo[IP_FAX_NO_EOLS].dword=1;

		} else if (!strcmp(*argv,"--minrowbits")) {
			pXform->aXformInfo[
				IP_FAX_MIN_ROW_LEN].dword=
				strtol(*argv,0,0);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-decfax")) {
		NEW_XFORM(X_FAX_DECODE);
		pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MH;
		continue;

x_fax_decode:
		if (!strcmp(*argv,"--mh")) {
			pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MH;

		} else if (!strcmp(*argv,"--mr")) {
			pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MR;

		} else if (!strcmp(*argv,"--mmr")) {
			pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MMR;

		} else if (!strcmp(*argv,"--noeol")) {
			pXform->aXformInfo[IP_FAX_NO_EOLS].dword=1;

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-encpcx")) {
		NEW_XFORM(X_PCX_ENCODE);
		continue;

x_pcx_encode:
		goto syntaxError;

	} else if (!strcmp(*argv,"-decpcx")) {
		NEW_XFORM(X_PCX_DECODE);
		continue;

x_pcx_decode:
		goto syntaxError;

	} else if (!strcmp(*argv,"-encjpg")) {
		NEW_XFORM(X_JPG_ENCODE);
		continue;

x_jpg_encode:
		if (!strcmp(*argv,"--qfac")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_JPG_ENCODE_QUALITY_FACTORS].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--sampfac")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_JPG_ENCODE_SAMPLE_FACTORS].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--subsampled")) {
			pXform->aXformInfo[
				IP_JPG_ENCODE_ALREADY_SUBSAMPLED].dword=1;

		} else if (!strcmp(*argv,"--oj500")) {
			pXform->aXformInfo[
				IP_JPG_ENCODE_FOR_DENALI].dword=1;

		} else if (!strcmp(*argv,"--dnl")) {
			pXform->aXformInfo[
				IP_JPG_ENCODE_OUTPUT_DNL].dword=1;

		} else if (!strcmp(*argv,"--colorfax")) {
			pXform->aXformInfo[
				IP_JPG_ENCODE_FOR_COLOR_FAX].dword=1;

		} else if (!strcmp(*argv,"--hdrlen")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_JPG_ENCODE_DUMMY_HEADER_LEN].dword=
				strtol(*argv,0,0);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-decjpg")) {
		NEW_XFORM(X_JPG_DECODE);
		continue;

x_jpg_decode:
		if (!strcmp(*argv,"--subsampled")) {
			pXform->aXformInfo[
				IP_JPG_DECODE_OUTPUT_SUBSAMPLED].dword=1;

		} else if (!strcmp(*argv,"--oj500")) {
			pXform->aXformInfo[
				IP_JPG_DECODE_FROM_DENALI].dword=1;

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-fixjpg")) {
		NEW_XFORM(X_JPG_FIX);
		continue;

x_jpg_fix:
		goto syntaxError;

	} else if (!strcmp(*argv,"-enctif")) {
		NEW_XFORM(X_TIF_ENCODE);
		continue;

x_tif_encode:
		if (!strcmp(*argv,"--outfile")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[IP_TIFF_FILE_PATH].pvoid=*argv;

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-dectif")) {
		NEW_XFORM(X_TIF_DECODE);
		continue;

x_tif_decode:
		goto syntaxError;

	} else if (!strcmp(*argv,"-encpnm")) {
		NEW_XFORM(X_PNM_ENCODE);
		continue;

x_pnm_encode:
		goto syntaxError;

	} else if (!strcmp(*argv,"-decpnm")) {
		NEW_XFORM(X_PNM_DECODE);
		continue;

x_pnm_decode:
		goto syntaxError;


	} else if (!strcmp(*argv,"-scale")) {
		NEW_XFORM(X_SCALE);
		pXform->aXformInfo[IP_SCALE_HORIZ_FACTOR].dword=0x01000000;
		pXform->aXformInfo[IP_SCALE_VERT_FACTOR].dword=0x01000000;
		continue;

x_scale:
		if (!strcmp(*argv,"--hfac")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_SCALE_HORIZ_FACTOR].dword=
				parseFixedPoint(*argv,24);

		} else if (!strcmp(*argv,"--vfac")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_SCALE_VERT_FACTOR].dword=
				parseFixedPoint(*argv,24);

		} else if (!strcmp(*argv,"--fast")) {
			pXform->aXformInfo[IP_SCALE_FAST].dword=1;

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-gray2bi")) {
		NEW_XFORM(X_GRAY_2_BI);
		continue;

x_gray_2_bi:
		if (!strcmp(*argv,"--threshold")) {
			pXform->aXformInfo[IP_GRAY_2_BI_THRESHOLD].dword=1;

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-bi2gray")) {
		NEW_XFORM(X_BI_2_GRAY);
		pXform->aXformInfo[IP_BI_2_GRAY_OUTPUT_BPP].dword=8;
		pXform->aXformInfo[IP_BI_2_GRAY_WHITE_PIXEL].dword=0xFFFFFF;
		pXform->aXformInfo[IP_BI_2_GRAY_BLACK_PIXEL].dword=0x000000;
		continue;

x_bi_2_gray:
		if (!strcmp(*argv,"--bpp")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_BI_2_GRAY_OUTPUT_BPP].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--white")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_BI_2_GRAY_WHITE_PIXEL].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--black")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_BI_2_GRAY_BLACK_PIXEL].dword=
				strtol(*argv,0,0);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-cnvcolspc")) {
		NEW_XFORM(X_CNV_COLOR_SPACE);
		pXform->aXformInfo[
			IP_CNV_COLOR_SPACE_WHICH_CNV].dword=IP_CNV_YCC_TO_SRGB;
		pXform->aXformInfo[
			IP_CNV_COLOR_SPACE_GAMMA].dword=0x00010000;
		continue;

x_cnv_color_space:
		if (!strcmp(*argv,"--ycc2cielab")) {
			pXform->aXformInfo[
				IP_CNV_COLOR_SPACE_WHICH_CNV].dword=
				IP_CNV_YCC_TO_CIELAB;

		} else if (!strcmp(*argv,"--cielab2ycc")) {
			pXform->aXformInfo[
				IP_CNV_COLOR_SPACE_WHICH_CNV].dword=
				IP_CNV_CIELAB_TO_YCC;

		} else if (!strcmp(*argv,"--ycc2srgb")) {
			pXform->aXformInfo[
				IP_CNV_COLOR_SPACE_WHICH_CNV].dword=
				IP_CNV_YCC_TO_SRGB;

		} else if (!strcmp(*argv,"--srgb2ycc")) {
			pXform->aXformInfo[
				IP_CNV_COLOR_SPACE_WHICH_CNV].dword=
				IP_CNV_SRGB_TO_YCC;

		} else if (!strcmp(*argv,"--lhs2srgb")) {
			pXform->aXformInfo[
				IP_CNV_COLOR_SPACE_WHICH_CNV].dword=
				IP_CNV_LHS_TO_SRGB;

		} else if (!strcmp(*argv,"--srgb2lhs")) {
			pXform->aXformInfo[
				IP_CNV_COLOR_SPACE_WHICH_CNV].dword=
				IP_CNV_SRGB_TO_LHS;

		} else if (!strcmp(*argv,"--bgrswap")) {
			pXform->aXformInfo[
				IP_CNV_COLOR_SPACE_WHICH_CNV].dword=
				IP_CNV_BGR_SWAP;

		} else if (!strcmp(*argv,"--gamma")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_CNV_COLOR_SPACE_GAMMA].dword=
				parseFixedPoint(*argv,16);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-yextract")) {
		NEW_XFORM(X_Y_EXTRACT);
		continue;

x_y_extract:
		if (!strcmp(*argv,"--ycc")) {
			pXform->aXformInfo[
				IP_Y_EXTRACT_COLOR_SPACE].dword=
				IP_Y_EXTRACT_LUM_CHROME;

		} else if (!strcmp(*argv,"--rgb")) {
			pXform->aXformInfo[
				IP_Y_EXTRACT_COLOR_SPACE].dword=
				IP_Y_EXTRACT_RGB;

		} else if (!strcmp(*argv,"--gbr")) {
			pXform->aXformInfo[
				IP_Y_EXTRACT_COLOR_SPACE].dword=
				IP_Y_EXTRACT_BGR;

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-thumb")) {
		NEW_XFORM(X_THUMB);
		argc--; argv++; if (argc<=0) goto syntaxError;
		pXform->aXformInfo[
			IP_SATURATION_FACTOR].dword=
			strtol(*argv,0,0);
		continue;

x_thumb:
		goto syntaxError;

	} else if (!strcmp(*argv,"-table")) {
		NEW_XFORM(X_TABLE);
		continue;

x_table:
		if (!strcmp(*argv,"--gamma")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_TABLE_WHICH].dword=
				IP_TABLE_GAMMA;
			pXform->aXformInfo[
				IP_TABLE_OPTION].dword=
				parseFixedPoint(*argv,16);

		} else if (!strcmp(*argv,"--threshold")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_TABLE_WHICH].dword=
				IP_TABLE_THRESHOLD;
			pXform->aXformInfo[
				IP_TABLE_OPTION].dword=
				strtol(*argv,0,0);

		} else {
			/* TODO */
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-crop")) {
		NEW_XFORM(X_CROP);
		continue;

x_crop:
		if (!strcmp(*argv,"--left")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_CROP_LEFT].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--right")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_CROP_RIGHT].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--top")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_CROP_TOP].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--height")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_CROP_MAXOUTROWS].dword=
				strtol(*argv,0,0);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-tonemap")) {
		NEW_XFORM(X_TONEMAP);
		goto syntaxError;
		/* TODO */
		continue;

x_tonemap:
		goto syntaxError;

	} else if (!strcmp(*argv,"-saturation")) {
		NEW_XFORM(X_SATURATION);
		argc--; argv++; if (argc<=0) goto syntaxError;
		pXform->aXformInfo[
			IP_SATURATION_FACTOR].dword=
			parseFixedPoint(*argv,8);
		continue;

x_saturation:
		goto syntaxError;

	} else if (!strcmp(*argv,"-rotate")) {
		NEW_XFORM(X_ROTATE);
		continue;

x_rotate:
		if (!strcmp(*argv,"--ul")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			i=strtol(*argv,0,0)<<16;
			argc--; argv++; if (argc<=0) goto syntaxError;
			i|=strtol(*argv,0,0);
			pXform->aXformInfo[IP_ROTATE_UPPER_LEFT].dword=i;

		} else if (!strcmp(*argv,"--ur")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			i=strtol(*argv,0,0)<<16;
			argc--; argv++; if (argc<=0) goto syntaxError;
			i|=strtol(*argv,0,0);
			pXform->aXformInfo[IP_ROTATE_UPPER_RIGHT].dword=i;

		} else if (!strcmp(*argv,"--ll")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			i=strtol(*argv,0,0)<<16;
			argc--; argv++; if (argc<=0) goto syntaxError;
			i|=strtol(*argv,0,0);
			pXform->aXformInfo[IP_ROTATE_LOWER_LEFT].dword=i;

		} else if (!strcmp(*argv,"--width")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[IP_ROTATE_OUTPUT_SIZE].dword=
				(pXform->aXformInfo[IP_ROTATE_OUTPUT_SIZE].dword
				&0xFFFF)|(strtol(*argv,0,0)<<16);

		} else if (!strcmp(*argv,"--height")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[IP_ROTATE_OUTPUT_SIZE].dword=
				(pXform->aXformInfo[IP_ROTATE_OUTPUT_SIZE].dword
				&0xFFFF0000)|(strtol(*argv,0,0));

		} else if (!strcmp(*argv,"--fast")) {
			pXform->aXformInfo[IP_ROTATE_FAST].dword=1;

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-pad")) {
		NEW_XFORM(X_PAD);
		continue;

x_pad:
		if (!strcmp(*argv,"--left")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_PAD_LEFT].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--right")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_PAD_RIGHT].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--top")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_PAD_TOP].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--bottom")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_PAD_BOTTOM].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--value")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_PAD_VALUE].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--minheight")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_PAD_MIN_HEIGHT].dword=
				strtol(*argv,0,0);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-fakemono")) {
		NEW_XFORM(X_FAKE_MONO);
		pXform->aXformInfo[IP_FAKE_MONO_BPP].dword=8;
		continue;

x_fake_mono:
		if (!strcmp(*argv,"--bpp")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_FAKE_MONO_BPP].dword=
				strtol(*argv,0,0);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-grayout")) {
		NEW_XFORM(X_GRAYOUT);
		continue;

x_grayout:
		if (!strcmp(*argv,"--left")) {
			pXform->aXformInfo[
				IP_GRAYOUT_LEFT].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--right")) {
			pXform->aXformInfo[
				IP_GRAYOUT_RIGHT].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--top")) {
			pXform->aXformInfo[
				IP_GRAYOUT_TOP].dword=
				strtol(*argv,0,0);

		} else if (!strcmp(*argv,"--bottom")) {
			pXform->aXformInfo[
				IP_GRAYOUT_BOTTOM].dword=
				strtol(*argv,0,0);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-chgbpp")) {
		NEW_XFORM(X_CHANGE_BPP);
		pXform->aXformInfo[IP_CHANGE_BPP_OUTPUT_BPP].dword=24;
		continue;

x_change_bpp:
		if (!strcmp(*argv,"--bpp")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_CHANGE_BPP_OUTPUT_BPP].dword=
				strtol(*argv,0,0);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-matrix")) {
		int *pMatrix=malloc(9*sizeof(int));
		NEW_XFORM(X_MATRIX);
		pXform->aXformInfo[0].pvoid=pMatrix;
		for (i=0;i<9;i++) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			*pMatrix++=parseFixedPoint(*argv,24);
		}
		continue;

x_matrix:
		goto syntaxError;

	} else if (!strcmp(*argv,"-convolve")) {
		NEW_XFORM(X_CONVOLVE);
		pXform->aXformInfo[IP_CONVOLVE_DIVISOR].dword=1;
		continue;

x_convolve:
		if (!strcmp(*argv,"--xymatrix")) {
			int *pConvMatrix;
			argc--; argv++; if (argc<=0) goto syntaxError;
			i=pXform->aXformInfo[
				IP_CONVOLVE_NCOLS].dword=
				strtol(*argv,0,0);
			argc--; argv++; if (argc<=0) goto syntaxError;
			i*=(pXform->aXformInfo[
				IP_CONVOLVE_NROWS].dword=
				strtol(*argv,0,0));
			pConvMatrix=malloc(i*sizeof(int));
			pXform->aXformInfo[
				IP_CONVOLVE_MATRIX].pvoid=
				pConvMatrix;
			for (;i>0;i--,pConvMatrix++) {
				argc--; argv++; if (argc<=0) goto syntaxError;
				*pConvMatrix=strtol(*argv,0,0);
			}

		} else if (!strcmp(*argv,"--divisor")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pXform->aXformInfo[
				IP_CONVOLVE_DIVISOR].dword=
				strtol(*argv,0,0);

		} else {
			goto syntaxError;
		}

	} else if (!strcmp(*argv,"-invert")) {
		NEW_XFORM(X_INVERT);
		continue;

x_invert:
		goto syntaxError;

	} else if (!strcmp(*argv,"-skel")) {
		NEW_XFORM(X_SKEL);
		continue;

x_skel:
		goto syntaxError;

	} else {
syntaxError:
		if (argc>0) {
			PRINTF("Problem near parameter <%s>!\n\n",*argv);
		}
		PRINTF(
"Syntax: \n"
"    hpojip-test infile.ext outfile.ext [-<xform> [--<options>]... ]...\n"
"Input default settings:\n"
"    -defwidth <n>\n"
"    -defbpp <n>\n"
"    -defcpp <n>\n"
"    -defhdpi <16.16>\n"
"    -defvdpi <16.16>\n"
"    -defheight <n>\n"
"    -defpages <n>\n"
"    -defpagenum <n>\n"
"    -mirrorin\n"
"    -mirrorout\n"
"Transforms:\n"
"    -encfax --mh --mr --mmr --noeol --minrowbits <n>\n"
"    -decfax --mh --mr --mmr --noeol\n"
"    -encpcx\n"
"    -decpcx\n"
"    -encjpg --qfac <n> --sampfac <n> --subsampled --oj500 --dnl \\\n"
"            --colorfax --hdrlen <n>\n"
"    -decjpg --subsampled --oj500\n"
"    -fixjpg\n"
"    -enctif --outfile <filename.ext>\n"
"    -dectif\n"
"    -encpnm\n"
"    -decpnm\n"
"    -scale --hfac <8.24> --vfac <8.24> --fast\n"
"    -gray2bi --threshold\n"
"    -bi2gray --bpp <8|24> --white 0xRRGGBB --black 0xRRGGBB\n"
"    -cnvcolspc --ycc2cielab --cielab2ycc --ycc2srgb --srgb2ycc \\\n"
"        --lhs2srgb --srgb2lhs --bgrswap --gamma <16.16>\n"
"    -yextract --ycc --rgb --bgr\n"
"    -thumb <n>\n"
"    -table --gamma <16.16> --threshold <n>\n"
"    -crop --left <n> --right <n> --top <n> --height <n>\n"
"    -tonemap\n"
"    -saturation <24.8>\n"
"    -rotate --ul <x> <y> --ur <x> <y> --ll <x> <y> \\\n"
"            --width <n> --height <n> --fast\n"
"    -pad --left <n> --right <n> --top <n> --bottom <n> \\\n"
"            --value <n> --minheight <n>\n"
"    -fakemono --bpp <1|8>\n"
"    -grayout --left <n> --right <n> --top <n> --bottom <n>\n"
"    -chgbpp --bpp <1|8|16|24|48>\n"
"    -matrix <8.24> <8.24> ... (9 total)\n"
"    -convolve --xymatrix <rows> <cols> <n0,0> <n1,0> ... --divisor <n>\n"
"    -invert\n"
"    -skel\n"
			);
		goto abort;
	}
    }

	if (!inFilename) {
		PRINTF("Missing inFilename!\n\n");
		goto syntaxError;
	}
	if (!outFilename) {
		PRINTF("Missing outFilename!\n\n");
		goto syntaxError;
	}

	wResult=ipOpen(nXforms,xforms,0,&hJob);
	if (wResult!=IP_DONE) {
		PRINTF("ipOpen() returns 0x%4.4X!\n",wResult);
		goto abort;
	}

	wResult=ipSetDefaultInputTraits(hJob,&defTraits);
	if (wResult!=IP_DONE) {
		PRINTF("ipSetDefaultInputTraits() returns 0x%4.4X!\n",wResult);
		goto abort;
	}

	inFile=open(inFilename,O_RDONLY);
	if (inFile<0) {
		PRINTF("Error opening input file <%s>!\n",inFilename);
		goto abort;
	}
	outFile=open(outFilename,O_CREAT|O_RDWR|O_TRUNC,0600);
	if (outFile<0) {
		PRINTF("Error opening output file <%s>!\n",outFilename);
		goto abort;
	}

	do {
		if (lseek(inFile,dwInputNextPos,SEEK_SET)!=dwInputNextPos) {
			PRINTF("lseek(inPos=%d) failed!\n",dwInputNextPos);
			goto abort;
		}
		dwInputAvail=read(inFile,inBuffer,LEN_BUFFER);
		if (dwInputAvail<0) {
			PRINTF("read(inFile) failed!\n");
			goto abort;
		}
		if (mirrorIn) {
			ipMirrorBytes(inBuffer,dwInputAvail);
		}

#if 0
		PRINTF("\nBefore ipConvert: pos=%d, dwInputAvail=%d.\n",
			dwInputNextPos,dwInputAvail);
#endif
		wResult=ipConvert(
			hJob,
			dwInputAvail,
			dwInputAvail>0?inBuffer:0,
			&dwInputUsed,
			&dwInputNextPos,
			LEN_BUFFER,
			outBuffer,
			&dwOutputUsed,
			&dwOutputThisPos);
#if 0
		PRINTF("After ipConvert: dwInputUsed=%d, dwInputNextPos=%d, "
			"dwOutputUsed=%d, dwOutputThisPos=%d, "
			"wResult=0x%4.4X.\n\n",
			dwInputUsed,dwInputNextPos,dwOutputUsed,
			dwOutputThisPos,wResult);
#endif
		if (wResult&(IP_INPUT_ERROR|IP_FATAL_ERROR)) {
			PRINTF("ipConvert(inPos=%d) returns 0x%4.4X!\n",
				dwInputNextPos,wResult);
			goto abort;
		}

	    if (dwOutputUsed>0) {
		if (lseek(outFile,dwOutputThisPos,SEEK_SET)!=dwOutputThisPos) {
			PRINTF("lseek(outPos=%d) failed!\n",dwOutputThisPos);
			goto abort;
		}

		if (mirrorOut) {
			ipMirrorBytes(outBuffer,dwOutputUsed);
		}
		i=write(outFile,outBuffer,dwOutputUsed);
		if (i!=dwOutputUsed) {
			PRINTF("write(outPos=%d) returns %d!\n",
				dwOutputThisPos,i);
			goto abort;
		}
		totalBytes+=dwOutputUsed;
	    }
	} while (!(wResult&IP_DONE));

	retcode=0;
abort:
	if (hJob) {
		wResult=ipClose(hJob);
		hJob=0;
		if (wResult!=IP_DONE) {

		}
	}
	if (inFile>=0) close(inFile);
	if (outFile>=0) close(outFile);
	return retcode;
}
