/* ptal-photod -- PTAL mtools daemon for digital-camera memory card access */
 
/* Copyright (C) 2001-2002 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */
 
/* Original author: David Paschal */

#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <syslog.h>
#include "ptal.h"


/* TODO: Make a more generalized storage API in libptal. */

#define HPPSP_SERVICE_DISK_IO			"HP-CARD-ACCESS"
#define HPPSP_SOCKET_DISK_IO			17
#define HPPSP_SERVICE_STREAMING_SAVE		"HP-SAVE-IMAGES"
#define HPPSP_SOCKET_STREAMING_SAVE		18

#define HPPSP_REPLY_NAK				0x0101
#define HPPSP_REPLY_INVALID_COMMAND		0x0103	/* >=Version 2 only. */
#define HPPSP_REPLY_VERSION			0x2150	/* >=Version 2 only. */

#define HPPSP_REQUEST_READ_SECTORS		0x0010
#define HPPSP_REQUEST_READ_SECTOR_RANGE		0x0011	/* >=Version 2 only. */
#define HPPSP_REQUEST_WRITE_SECTORS		0x0020
#define HPPSP_REQUEST_WRITE_SECTOR_RANGE	0x0021	/* >=Version 2 only. */
#define HPPSP_REQUEST_RESET			0x0030
#define HPPSP_REQUEST_QUERY_STATUS		0x0031
#define HPPSP_REQUEST_QUERY_GEOMETRY		0x0032
#define HPPSP_REQUEST_NOTIFY_DIRECTORY_UPDATED	0x0033	/* No response. */

#define HPPSP_REPLY_ACK				0x0100
#define HPPSP_REPLY_SECTOR_DATA			0x0110
#define HPPSP_REPLY_GEOMETRY			0x0120
#define HPPSP_REPLY_STATUS			0x0131

#define HPPSP_REQUEST_CANCEL			HPPSP_REPLY_NAK
#define HPPSP_REQUEST_SAVE_STATUS_V1		0x1010	/* Version 1 only. */
#define HPPSP_REQUEST_FILE_HEADER		0x1020	/* Version 1 only. */
#define HPPSP_REQUEST_FILE_DATA_V1		0x1030	/* Version 1 only. */
#define HPPSP_REQUEST_SAVE_STATUS_V2		0x2010	/* >=Version 2 only. */
#define HPPSP_REQUEST_DIRECTORY_DATA		0x2040	/* >=Version 2 only. */
#define HPPSP_REQUEST_VERSION			0x2050	/* >=Version 2 only. */
#define HPPSP_REQUEST_FILES_SAVED		0x2060	/* >=Version 2 only. */
#define HPPSP_REQUEST_FILE_DATA_V2		0x2070	/* >=Version 2 only. */

#define HPPSP_REPLY_CANCEL			0x0102
#define HPPSP_REPLY_SAVE_STATUS			0x1110
#define HPPSP_REPLY_FILE_HEADER			0x1120
#define HPPSP_REPLY_FILE_DATA			0x1130
#define HPPSP_REPLY_END_OF_STREAM		0x1131

#define HPPSP_VERSION_UNKNOWN			(-1)
#define HPPSP_VERSION_1				0x0000
#define HPPSP_VERSION_2				0x0101



struct HppspGeneric {
	unsigned char command[2];
} __attribute__((packed));
#define HPPSP_LEN_GENERIC (sizeof(struct HppspGeneric))

struct HppspReplyInvalidCommand {
	unsigned char command[2];
	unsigned char offendingCommand[2];
} __attribute__((packed));
#define HPPSP_LEN_REPLY_INVALID_COMMAND \
	(sizeof(struct HppspReplyInvalidCommand))

struct HppspReplyVersion {
	unsigned char command[2];
	unsigned char version[2];
} __attribute__((packed));
#define HPPSP_LEN_REPLY_VERSION \
	(sizeof(struct HppspReplyVersion))

struct HppspSector {
	unsigned char sector[4];
} __attribute__((packed));
#define HPPSP_BYTES_PER_SECTOR	512
struct HppspData {
	unsigned char data[HPPSP_BYTES_PER_SECTOR];
} __attribute__((packed));
struct HppspSectorAndData {
	struct HppspSector sector[1];
	struct HppspData data;
} __attribute__((packed));

struct HppspRequestReadSectors {
	unsigned char command[2];
	unsigned char sectorCount[2];
	struct HppspSector sectors[1];		/* Sector list. */
} __attribute__((packed));
#define HPPSP_LEN_REQUEST_READ_SECTORS_ONE \
	(sizeof(struct HppspRequestReadSectors))
#define HPPSP_LEN_REQUEST_READ_SECTORS_NONE \
	(HPPSP_LEN_REQUEST_READ_SECTORS_ONE-sizeof(struct HppspSector))

struct HppspRequestReadSectorRange {
	unsigned char command[2];
	unsigned char sectorCount[2];
	struct HppspSector firstSector;
} __attribute__((packed));
#define HPPSP_LEN_REQUEST_READ_SECTOR_RANGE \
	(sizeof(struct HppspRequestReadSectorRange))

struct HppspRequestWriteSectors {
	unsigned char command[2];
	unsigned char sectorCount[2];
	unsigned char checksum[2];
	struct HppspSectorAndData sectorsAndData;
} __attribute__((packed));
#define HPPSP_LEN_REQUEST_WRITE_SECTORS_ONE \
	(sizeof(struct HppspRequestWriteSectors))
#define HPPSP_LEN_REQUEST_WRITE_SECTORS_NONE \
	(HPPSP_LEN_REQUEST_WRITE_SECTORS_ONE-sizeof(struct HppspSectorAndData))

struct HppspRequestWriteSectorRange {
	unsigned char command[2];
	unsigned char sectorCount[2];
	struct HppspSector firstSector;
	struct HppspData data[1];
} __attribute__((packed));
#define HPPSP_LEN_REQUEST_WRITE_SECTOR_RANGE_ONE \
	(sizeof (struct HppspRequestWriteSectorRange))
#define HPPSP_LEN_REQUEST_WRITE_SECTOR_RANGE_NONE \
	(HPPSP_LEN_REQUEST_WRITE_SECTOR_RANGE_ONE-sizeof(struct HppspData))

struct HppspReplySectorData {
	unsigned char command[2];
	unsigned char sectorCount[4];
	unsigned char version[2];
	struct HppspData data[1];
} __attribute__((packed));
#define HPPSP_LEN_REPLY_READ_SECTOR_ONE \
	(sizeof(struct HppspReplySectorData))
#define HPPSP_LEN_REPLY_READ_SECTOR_NONE \
	(HPPSP_LEN_REPLY_READ_SECTOR_ONE-sizeof(struct HppspData))

struct HppspReplyGeometry {
	unsigned char command[2];
	unsigned char bytesPerSector[4];	/* ==HPPSP_BYTES_PER_SECTOR */
	unsigned char reserved[2];
} __attribute__((packed));
#define HPPSP_LEN_REPLY_GEOMETRY (sizeof(struct HppspReplyGeometry))

struct HppspReplyStatus {
	unsigned char command[2];
	unsigned char activeDrive[2];
} __attribute__((packed));
#define HPPSP_LEN_REPLY_STATUS (sizeof(struct HppspReplyStatus))

#define HPPSP_ACTIVE_DRIVE_NONE			4
#define HPPSP_ACTIVE_DRIVE_COMPACT_FLASH	0
#define HPPSP_ACTIVE_DRIVE_SMART_MEDIA		5
#define HPPSP_ACTIVE_DRIVE_MEMORY_STICK		6


struct HppspRequestFile {
	unsigned char command[2];	/* File Header or Data. */
	unsigned char fileNumber[2];
} __attribute__((packed));
#define HPPSP_LEN_REQUEST_FILE (sizeof(struct HppspRequestFile))

struct HppspReplySaveStatus {
	unsigned char command[2];
	unsigned char fileCount[2];
	unsigned char estCombinedSize[4];
	unsigned char flags[4];
} __attribute__((packed));
#define HPPSP_LEN_REPLY_SAVE_STATUS (sizeof(struct HppspReplySaveStatus))

#define HPPSP_SAVE_STATUS_FLAG_SOURCE_MASK		0x000F
#define HPPSP_SAVE_STATUS_FLAG_SOURCE_COMPACT_FLASH	0x0001
#define HPPSP_SAVE_STATUS_FLAG_SOURCE_SSFDC		0x0002
#define HPPSP_SAVE_STATUS_FLAG_SOURCE_INFRARED		0x0003

struct HppspDirentry {
	unsigned char filename[8];
	unsigned char extension[3];
	unsigned char attributes;
	unsigned char reserved;
	unsigned char creationCentiSeconds;
	unsigned char creationTime[2];
	unsigned char creationDate[2];
	unsigned char accessDate[2];
	unsigned char clusterHigh[2];
	unsigned char modificationTime[2];
	unsigned char modificationDate[2];
	unsigned char clusterLow[2];
	unsigned char size[4];
} __attribute__((packed));

struct HppspReplyFileHeader {
	unsigned char command[2];
	unsigned char fileNumber[2];
	struct HppspDirentry direntry;
	unsigned char lenFilespec[2];
	unsigned char checksum[2];
	unsigned char data[1];		/* Filespec (path+name). */
} __attribute__((packed));
#define HPPSP_LEN_REPLY_FILE_HEADER_NODATA (sizeof(struct HppspReplyFileHeader)-1)

struct HppspReplyFileData {
	unsigned char command[2];
	unsigned char fileNumber[2];
	unsigned char sequenceNumber[2];
	unsigned char packetSize[4];
	unsigned char checksum[2];
	unsigned char data[1];		/* File data. */
} __attribute__((packed));
#define HPPSP_LEN_REPLY_FILE_DATA_NODATA (sizeof(struct HppspReplyFileData)-1)


#define _PTAL_LOG_ERROR(args...) \
	do { \
		PTAL_LOG_ERROR(args); \
		syslog(LOG_LPR|LOG_ERR,args); \
	} while(0)

#if 0
#define _PTAL_LOG_DEBUG(args...) \
	do { \
		PTAL_LOG_DEBUG(args); \
		syslog(LOG_LPR|LOG_INFO,args); \
	} while(0)
#else
#define _PTAL_LOG_DEBUG PTAL_LOG_DEBUG
#endif

#define FLOPPYD_DEFAULT_PORT		5703
#define FLOPPYD_PROTOCOL_VERSION_10	10
#define FLOPPYD_PROTOCOL_VERSION_11	11
#define MAX_CONNECTIONS			1
#define LEN_BUFFER			4096

enum FloppydOpcodes
{
	OP_READ,
	OP_WRITE,
	OP_SEEK,
	OP_FLUSH,
	OP_CLOSE,
	OP_IOCTL,
	OP_OPRO,
	OP_OPRW
};

enum AuthErrorsEnum {
	AUTH_SUCCESS,
	AUTH_PACKETOVERSIZE,
	AUTH_AUTHFAILED,
	AUTH_WRONGVERSION,
	AUTH_DEVLOCKED,
	AUTH_BADPACKET
};

char *ptalName=0;
int readOnly=0;

int readFromMtools(int fd,unsigned char *buffer,int len,char *desc) {
	int r=read(fd,(char *)buffer,len);
	if (r!=len) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Error reading %s, "
			"readlen=%d, expected=%d)!\n",
			ptalName,desc,r,len);
		return PTAL_ERROR;
	}
	PTAL_LOG_DEBUG("ptal-photod: Read %d bytes for %s from mtools.\n",
		desc);
	return r;
}

int readDwordFromMtools(int fd,int *i,char *desc) {
	unsigned char buffer[4];
	int ii;
	if (!i) i=&ii;

	if (readFromMtools(fd,buffer,4,desc)==PTAL_ERROR) return PTAL_ERROR;

	*i=BEND_GET_LONG(buffer);
	PTAL_LOG_DEBUG("ptal-photod: Got %s dword=%d from mtools.\n",desc,*i);
	return PTAL_OK;
}

int readConstDwordFromMtools(int fd,int iconst,char *desc) {
	int i;

	if (readDwordFromMtools(fd,&i,desc)==PTAL_ERROR) return PTAL_ERROR;
	if (i!=iconst) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Error reading %s: "
			"expected=%d, got=%d!\n",ptalName,desc,iconst,i);
		return PTAL_ERROR;
	}

	return PTAL_OK;
}

int readCountedBytesFromMtools(int fd,unsigned char *buffer,int maxlen,
    char *desc) {
	int r,len;

	r=readDwordFromMtools(fd,&len,desc);
	if (r==PTAL_ERROR) return r;
	if (len>maxlen) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Oversided %s, "
			"byte count=%d, maxlen=%d!\n",
			ptalName,desc,len,maxlen);
		return PTAL_ERROR;
	}

	return readFromMtools(fd,buffer,len,desc);
}

int writeDwordToMtools(int fd,int i,char *desc) {
	unsigned char buffer[4];
	int r;

	BEND_SET_LONG(buffer,i);
	r=write(fd,(char *)buffer,4);
	if (r!=4) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Error writing %s=%d "
			"(len=%d)!\n",ptalName,desc,i,r);
		return PTAL_ERROR;
	}

	PTAL_LOG_DEBUG("ptal-photod: Wrote %s=%d to mtools.\n",desc,i);
	return PTAL_OK;
}

int writeResultToMtools(int fd,int gotlen,int errcode,char *desc) {
	if (writeDwordToMtools(fd,8,desc)==PTAL_ERROR ||
	    writeDwordToMtools(fd,gotlen,desc)==PTAL_ERROR ||
	    writeDwordToMtools(fd,errcode,desc)==PTAL_ERROR) {
		return PTAL_ERROR;
	}
	return PTAL_OK;
}

/* Despite the performance loss, I think it's safer to read/write 1 sector
 * at a time, due to the possibility of getting a short sector data packet
 * followed by a NAK.  TODO: Revisit. */
void handleSession(int fd,ptalChannel_t chan) {
	int opcode,r,i,seekOffset,seekWhence;
	int countdown,countup,sectorNum,sectorOffset,sectorCopyCount;
	union {
		struct HppspGeneric generic;
		struct HppspReplyGeometry replyGeometry;
		struct HppspRequestReadSectors readSectors;
		struct HppspReplySectorData replySectorData;
		struct HppspRequestWriteSectors writeSectors;
		unsigned char buffer[LEN_BUFFER];
	} __attribute__((packed)) pkt;
	unsigned char *buffer=pkt.buffer;
	int mtoolsProtoVersion;
	int deviceProtoVersion=HPPSP_VERSION_UNKNOWN,bytesPerSector;
	int fileOffset=0,dirty=0;

	PTAL_LOG_DEBUG("ptal-photod(%s): handleSession() started on fd=%d.\n",
		ptalName,fd);

	/* Read protocol version. */
	if (readConstDwordFromMtools(fd,4,"protoVerLen")==PTAL_ERROR ||
	    readDwordFromMtools(fd,&mtoolsProtoVersion,"protoVer")==
	     PTAL_ERROR) {
		goto abort;
	}
	if (mtoolsProtoVersion!=FLOPPYD_PROTOCOL_VERSION_10) {
		if (mtoolsProtoVersion!=FLOPPYD_PROTOCOL_VERSION_11) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Unknown mtools "
				"protocol version=%d!\n",
				ptalName,mtoolsProtoVersion);
		}
		/* Send reply indicating wrong protocol version. */
		writeDwordToMtools(fd,4,"authWrongVerLen");
		writeDwordToMtools(fd,AUTH_WRONGVERSION,"authWrongVer");
		goto abort;
	}

	/* Send reply indicating compatible protocol version. */
	if (writeDwordToMtools(fd,4,"authRightVerLen")==PTAL_ERROR ||
	    writeDwordToMtools(fd,AUTH_SUCCESS,"authRightVer")==PTAL_ERROR) {
		goto abort;
	}

	/* Read (and ignore) Xauthority information. */
	r=read(fd,(char *)buffer,LEN_BUFFER);
	if (r<=0) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Error reading Xauthority "
			"information!\n",ptalName);
		goto abort;
	}

	/* Send affirmative reply to Xauthority information. */
	if (writeDwordToMtools(fd,4,"authSuccessLen")==PTAL_ERROR ||
	    writeDwordToMtools(fd,AUTH_SUCCESS,"authSuccess")==PTAL_ERROR) {
		goto abort;
	}

	/* Try to open the Disk I/O channel. */
	if (ptalChannelOpen(chan)==PTAL_ERROR) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Error opening the "
			"peripheral's photo-card access channel!\n",
			ptalName);
		goto abort;
	}

	/* TODO: Clean up device I/O.  Read multiple sectors at a time. */

	/* Ask for device geometry. */
	BEND_SET_SHORT(pkt.generic.command,HPPSP_REQUEST_QUERY_GEOMETRY);
	r=ptalChannelWrite(chan,(unsigned char *)&pkt,HPPSP_LEN_GENERIC);
	if (r!=HPPSP_LEN_GENERIC) {
		_PTAL_LOG_ERROR("ptal-photod(%s): "
			"ptalChannelWrite(queryGeometry) returns %d!\n",
			ptalName,r);
		goto abort;
	}
	/* Read, validate, and interpret geometry reply. */
	r=ptalChannelRead(chan,(unsigned char *)&pkt,HPPSP_LEN_REPLY_GEOMETRY);
	if (r<HPPSP_LEN_GENERIC) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Error reading geometry reply "
			"packet (len=%d)!\n",ptalName,r);
		goto abort;
	}
	opcode=BEND_GET_SHORT(pkt.generic.command);
	if (opcode==HPPSP_REPLY_NAK) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Unable to read geometry "
			"(NAK)! \nPerhaps no photo card is inserted?\n",
			ptalName);
		goto abort;
	}
	if (opcode!=HPPSP_REPLY_GEOMETRY ||
	    r!=HPPSP_LEN_REPLY_GEOMETRY) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Geometry query "
			"returned len=%d, opcode=0x%4.4X!\n",
			ptalName,r,opcode);
		for (i=0;i<r;i++) {
			_PTAL_LOG_ERROR("ptal-photod(%s): "
				"pkt[0x%2.2X]=0x%2.2X.\n",ptalName,
				i,((unsigned char *)(&pkt))[i]);
		}
		goto abort;
	}
	bytesPerSector=BEND_GET_LONG(pkt.replyGeometry.bytesPerSector);
	_PTAL_LOG_DEBUG("ptal-photod(%s): bytesPerSector=%d.\n",
		ptalName,bytesPerSector);
	/* TODO: Remove this limitation. */
	if (bytesPerSector!=HPPSP_BYTES_PER_SECTOR) {
		_PTAL_LOG_ERROR("ptal-photod(%s): Sorry, bytesPerSector=%d "
			"is not currently supported.\n",
			ptalName,bytesPerSector);
		goto abort;
	}

    /* Read and process commands from mtools. */
    while (42) {
	/* Read the opcode. */
	if (readCountedBytesFromMtools(fd,buffer,1,"opcode")==PTAL_ERROR) {
		goto abort;
	}
	opcode=buffer[0];
	PTAL_LOG_DEBUG("ptal-photod: Got opcode=%d from mtools.\n",opcode);

	/* Handle the opcode. */
	switch (opcode) {
	   case OP_READ:
		/* Read the read length. */
		if (readConstDwordFromMtools(fd,4,"readLenLen")==PTAL_ERROR ||
		    readDwordFromMtools(fd,&countdown,"readLen")==PTAL_ERROR) {
			goto abort;
		}
		countup=0;

		/* Write (prematurely) successful result to mtools. */
		if (writeResultToMtools(fd,countdown,0,"readSuccess")==
		     PTAL_ERROR ||
		    writeDwordToMtools(fd,countdown,"readCount")==
		     PTAL_ERROR) {
			goto abort;
		}

	    /* Read sectors one at a time until we're done. */
	    while (countdown) {
		/* Figure out how much of this sector mtools really wants. */
		sectorNum=fileOffset/bytesPerSector;
		sectorOffset=fileOffset%bytesPerSector;
		sectorCopyCount=bytesPerSector-sectorOffset;
		if (sectorCopyCount>countdown) sectorCopyCount=countdown;

		/* Ask to read the sector. */
		PTAL_LOG_DEBUG("ptal-photod: Reading sector=%d.\n",sectorNum);
		BEND_SET_SHORT(pkt.readSectors.command,
			HPPSP_REQUEST_READ_SECTORS);
		BEND_SET_SHORT(pkt.readSectors.sectorCount,1);
		BEND_SET_LONG(pkt.readSectors.sectors[0].sector,sectorNum);
		r=ptalChannelWrite(chan,(unsigned char *)&pkt,
			HPPSP_LEN_REQUEST_READ_SECTORS_ONE);
		if (r!=HPPSP_LEN_REQUEST_READ_SECTORS_ONE) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Error writing read "
				"request packet (len=%d)!\n",ptalName,r);
			goto abort;
		}

		/* Read and validate the sector data. */
		/* TODO: Read based on bytesPerSector! */
		r=ptalChannelRead(chan,(unsigned char *)&pkt,sizeof(pkt));
		if (r<HPPSP_LEN_GENERIC) {
sdpReadError:
			_PTAL_LOG_ERROR("ptal-photod(%s): Error reading "
				"sector data packet (len=%d)!\n",ptalName,r);
			goto abort;
		}
		opcode=BEND_GET_SHORT(pkt.generic.command);
		if (opcode==HPPSP_REPLY_NAK) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Unable to read "
				"sector data packet (NAK)! \n"
				"Perhaps the photo card was removed?\n",
				ptalName);
			goto abort;
		}
		if (opcode!=HPPSP_REPLY_SECTOR_DATA) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Received unexpected "
				"opcode=0x%4.4X, len=%d instead of sector "
				"data packet!\n",ptalName,opcode,r);
			for (i=0;i<r;i++) {
				_PTAL_LOG_ERROR("ptal-photod(%s): "
					"pkt[0x%2.2X]=0x%2.2X.\n",ptalName,
					i,((unsigned char *)(&pkt))[i]);
			}
			goto abort;
		}
		if (r!=HPPSP_LEN_REPLY_READ_SECTOR_ONE) {
			goto sdpReadError;
		}
		r=BEND_GET_LONG(pkt.replySectorData.sectorCount);
		if (r && r!=1) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Unexpected "
				"sectorCount=%d!\n",ptalName,r);
		}
		r=BEND_GET_SHORT(pkt.replySectorData.version);
		if (r!=deviceProtoVersion) {
			_PTAL_LOG_DEBUG("ptal-photod(%s): Device protocol "
				"version is 0x%4.4X.\n",ptalName,r);
			deviceProtoVersion=r;
		}

		/* Send sector data to mtools. */
		PTAL_LOG_DEBUG("ptal-photod: Sending %d bytes to mtools "
			"from sector=%d, offset=%d.\n",
			sectorCopyCount,sectorNum,sectorOffset);
		r=write(fd,(char *)&pkt.replySectorData.data[0].
			data[sectorOffset],sectorCopyCount);
		if (r!=sectorCopyCount) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Error writing sector "
				"data to mtools (len=%d)!\n",ptalName,r);
			goto abort;
		}

		/* Keep track of progress. */
		fileOffset+=sectorCopyCount;
		countup+=sectorCopyCount;
		countdown-=sectorCopyCount;
	    }

		break;

	   case OP_WRITE:
		/* Read the write length. */
		if (readDwordFromMtools(fd,&countdown,"writeLen")==PTAL_ERROR) {
			goto abort;
		}
		countup=0;

		/* Simulate write-protected media. */
		if (readOnly) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Unable to write to "
				"read-only file system!\n",ptalName);
			writeResultToMtools(fd,PTAL_ERROR,EROFS,
				"writeProtected");
			goto abort;
		}

	    /* Write sectors one at a time until we're done. */
	    while (countdown) {
		/* Figure out how much of this sector mtools really wants
		 * to write. */
		sectorNum=fileOffset/bytesPerSector;
		sectorOffset=fileOffset%bytesPerSector;
		sectorCopyCount=bytesPerSector-sectorOffset;
		if (sectorCopyCount>countdown) sectorCopyCount=countdown;

	    /* If only a partial sector will be written, then read in the
	     * old sector before overlaying the new data. */
	    if (sectorOffset || sectorCopyCount<bytesPerSector) {
		/* Ask to read the sector. */
		PTAL_LOG_DEBUG("ptal-photod: _Reading sector=%d.\n",sectorNum);
		BEND_SET_SHORT(pkt.readSectors.command,
			HPPSP_REQUEST_READ_SECTORS);
		BEND_SET_SHORT(pkt.readSectors.sectorCount,1);
		BEND_SET_LONG(pkt.readSectors.sectors[0].sector,sectorNum);
		r=ptalChannelWrite(chan,(unsigned char *)&pkt,
			HPPSP_LEN_REQUEST_READ_SECTORS_ONE);
		if (r!=HPPSP_LEN_REQUEST_READ_SECTORS_ONE) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Error writing _read "
				"request packet (len=%d)!\n",ptalName,r);
			goto abort;
		}

		/* Read and validate the sector data header. */
		r=ptalChannelRead(chan,(unsigned char *)&pkt,
			HPPSP_LEN_REPLY_READ_SECTOR_NONE);
		if (r<HPPSP_LEN_GENERIC) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Error reading "
				"_sector data packet (len=%d)!\n",ptalName,r);
			goto abort;
		}
		opcode=BEND_GET_SHORT(pkt.generic.command);
		if (opcode==HPPSP_REPLY_NAK) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Unable to read "
				"_sector data packet (NAK)! \n"
				"Perhaps the photo card was removed?\n",
				ptalName);
			goto abort;
		}
		if (opcode!=HPPSP_REPLY_SECTOR_DATA) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Received unexpected "
				"opcode=0x%4.4X, len=%d instead of _sector "
				"data packet!\n",ptalName,opcode,r);
			for (i=0;i<r;i++) {
				_PTAL_LOG_ERROR("ptal-photod(%s): "
					"pkt[0x%2.2X]=0x%2.2X.\n",ptalName,
					i,((unsigned char *)(&pkt))[i]);
			}
			goto abort;
		}
		r=BEND_GET_LONG(pkt.replySectorData.sectorCount);
		if (r && r!=1) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Unexpected "
				"_sectorCount=%d!\n",ptalName,r);
		}
		r=BEND_GET_SHORT(pkt.replySectorData.version);
		if (r!=deviceProtoVersion) {
			_PTAL_LOG_DEBUG("ptal-photod(%s): _Device protocol "
				"version is 0x%4.4X.\n",ptalName,r);
			deviceProtoVersion=r;
		}
		/* Read the sector data into the right place where we'll
		 * want to write it back out. */
		r=ptalChannelRead(chan,
			pkt.writeSectors.sectorsAndData.data.data,
			bytesPerSector);
		if (r!=bytesPerSector) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Error reading "
				"remainder of sector data packet (len=%d)!\n",
				ptalName,r);
			goto abort;
		}
	    }

		/* Read the new data from mtools. */
		PTAL_LOG_DEBUG("ptal-photod: Copying %d bytes from mtools "
			"to sector=%d, offset=%d.\n",
			sectorCopyCount,sectorNum,sectorOffset);
		if (readFromMtools(fd,&pkt.writeSectors.sectorsAndData.data.data
		     [sectorOffset],sectorCopyCount,"sectorData")==PTAL_ERROR) {
			goto abort;
		}

		/* Ask to write the sector. */
		PTAL_LOG_DEBUG("ptal-photod: Writing sector=%d.\n",sectorNum);
		BEND_SET_SHORT(pkt.writeSectors.command,
			HPPSP_REQUEST_WRITE_SECTORS);
		BEND_SET_SHORT(pkt.writeSectors.sectorCount,1);
		BEND_SET_SHORT(pkt.writeSectors.checksum,0);
		BEND_SET_LONG(pkt.writeSectors.sectorsAndData.sector[0].sector,
			sectorNum);
		/* TODO: Write based on bytesPerSector! */
		r=ptalChannelWrite(chan,(unsigned char *)&pkt,
			HPPSP_LEN_REQUEST_WRITE_SECTORS_ONE);
		if (r!=HPPSP_LEN_REQUEST_WRITE_SECTORS_ONE) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Error writing write "
				"request packet (len=%d)!\n",ptalName,r);
			goto abort;
		}

		/* Read and validate the response (ACK or NAK). */
		r=ptalChannelRead(chan,(unsigned char *)&pkt,
			HPPSP_LEN_GENERIC);
		if (r<HPPSP_LEN_GENERIC) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Error reading "
				"write-sectors reply (len=%d)!\n",ptalName,r);
			goto abort;
		}
		opcode=BEND_GET_SHORT(pkt.generic.command);
		if (opcode==HPPSP_REPLY_NAK) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Unable to write "
				"sector=%d (NAK)! \n"
				"Perhaps the photo card was removed?\n",
				ptalName,sectorNum);
			goto abort;
		}
		if (opcode!=HPPSP_REPLY_ACK) {
			_PTAL_LOG_ERROR("ptal-photod(%s): Received unexpected "
				"opcode=0x%4.4X instead of ACK!\n",
				ptalName,opcode);
			goto abort;
		}
		/* Remember the successful write, so we can tell the device
		 * to re-read the directory when we're done. */
		dirty=1;

		/* Keep track of progress. */
		fileOffset+=sectorCopyCount;
		countup+=sectorCopyCount;
		countdown-=sectorCopyCount;
	    }

		/* Write successful result to mtools. */
		if (writeResultToMtools(fd,countup,0,"writeSuccess")==
		     PTAL_ERROR) {
			goto abort;
		}
		break;

	   case OP_SEEK:
		/* Read parameters from mtools. */
		if (readConstDwordFromMtools(fd,8,"seekLen")==
		     PTAL_ERROR ||
		    readDwordFromMtools(fd,&seekOffset,"seekOffset")==
		     PTAL_ERROR ||
		    readDwordFromMtools(fd,&seekWhence,"seekWhence")==
		     PTAL_ERROR) {
			goto abort;
		}

		/* Do the right kind of seek. */
		switch (seekWhence) {
		   case SEEK_CUR:
			seekOffset+=fileOffset;
		   case SEEK_SET:
			if (seekOffset<0) {
				_PTAL_LOG_ERROR("ptal-photod(%s): "
					"fileOffset=%d seeks to %d!\n",
					ptalName,fileOffset,seekOffset);
				if (writeResultToMtools(fd,PTAL_ERROR,EINVAL,
				     "seekBadOffset")==PTAL_ERROR) {
					goto abort;
				}
				break;
			}
			fileOffset=seekOffset;
			break;

		   case SEEK_END:
		   default:
			_PTAL_LOG_ERROR("ptal-photod(%s): "
				"Invalid seekWhence=%d\n!",
				ptalName,seekWhence);
			writeResultToMtools(fd,PTAL_ERROR,EINVAL,
				"seekBadWhence");
			goto abort;
		}

		if (writeResultToMtools(fd,fileOffset,0,"seekSuccess")==
		     PTAL_ERROR) {
			goto abort;
		}
		break;

	   case OP_CLOSE:
		PTAL_LOG_DEBUG("ptal-photod: Got mtools close command.\n");
		goto abort;

	   case OP_FLUSH:
		if (readCountedBytesFromMtools(fd,buffer,LEN_BUFFER,
		     "flushArg")==PTAL_ERROR) {
			goto abort;
		}

		/* This opcode doesn't do anything. */

		if (writeResultToMtools(fd,0,0,"flushSuccess")==
		     PTAL_ERROR) {
			goto abort;
		}
		break;

	   case OP_IOCTL:
	   default:
		_PTAL_LOG_ERROR("ptal-photod(%s): Invalid mtools opcode=%d\n!",
			ptalName,opcode);
		goto abort;
	}
    }

abort:
	if (dirty) {
		/* Tell the device to re-read the directory. */
		PTAL_LOG_DEBUG("ptal-photod(%s): Issuing directory-updated "
			"command.\n",ptalName);
		BEND_SET_SHORT(pkt.generic.command,
			HPPSP_REQUEST_NOTIFY_DIRECTORY_UPDATED);
		r=ptalChannelWrite(chan,(unsigned char *)&pkt,
			HPPSP_LEN_GENERIC);
		if (r!=HPPSP_LEN_GENERIC) {
			_PTAL_LOG_ERROR("ptal-photod(%s): "
				"ptalChannelWrite(notifyUpdate) returns %d!\n",
				ptalName,r);
		}
	}
	ptalChannelClose(chan);
	close(fd);
	PTAL_LOG_DEBUG("ptal-photod(%s): handleSession() done.\n",ptalName);
}

void signalHandler(int signum) {
	signal(signum,signalHandler);
	PTAL_LOG_DEBUG("ptal-photod(%s): Caught signal %d!\n",
		ptalName,signum);
}

int main(int argc,char **argv) {
	struct sockaddr_in addrServer,addrClient;
	int basePort=FLOPPYD_DEFAULT_PORT,portOffset=0,tcpPort=-1;
	int maxAltPorts=0,noFork=0;
	int fdSocket,fdSession,retcode=255,lenSockAddr=sizeof(addrServer),r;
	ptalDevice_t dev;
	ptalChannel_t chan;

	bzero((char *)&addrServer,lenSockAddr);
	addrServer.sin_addr.s_addr=htonl(0x7F000001);	/* 127.0.0.1 */

	/* Set up syslog. */
	openlog("ptal-photod",LOG_NDELAY,LOG_LPR);

	ptalInit();

	while (42) {
		argc--; argv++; if (argc<=0) break;

		if (**argv!='-') {
			if (ptalName) goto syntaxError;
			ptalName=*argv;

		} else if (!strcmp(*argv,"-bindto")) {
			struct hostent *phe;
			argc--; argv++; if (argc<=0) goto syntaxError;
			if ((phe=gethostbyname(*argv))!=0) {
				bcopy(phe->h_addr,
					(char *)(&addrServer.sin_addr.s_addr),
					phe->h_length);
			} else if ((addrServer.sin_addr.s_addr=
			 		inet_addr(*argv))==-1) {
				_PTAL_LOG_ERROR("ptal-photod(%s): Error "
					"looking up TCP/IP address \"%s\"!\n",
					ptalName,*argv);
				goto abort;
			}

		} else if (!strcmp(*argv,"-bindtoall")) {
			addrServer.sin_addr.s_addr=INADDR_ANY;

		} else if (!strcmp(*argv,"-baseport")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			basePort=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-portoffset")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			portOffset=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-tcpport")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			tcpPort=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-maxaltports")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			maxAltPorts=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-readonly")) {
			readOnly=1;

		} else if (!strcmp(*argv,"-readwrite")) {
			readOnly=0;

		} else if (!strcmp(*argv,"-nofork")) {
			noFork=1;

		} else {
syntaxError:
			PTAL_LOG_ERROR(
"Syntax: ptal-photod <devname> [<options>...]\n"
"Where <devname> may be one of:\n"
				);
			ptalDeviceEnumerate(0,
				ptalDeviceEnumeratePrintCallback,0);
			PTAL_LOG_ERROR(
"Supported <options>:\n"
"  -bindto <ipaddr> -- binds to specified TCP/IP address (default=127.0.0.1)\n"
"  -bindtoall       -- binds to all TCP/IP interfaces\n"
"  -baseport <n>    -- sets TCP/IP base port (default=%d)\n"
"  -portoffset <n>  -- sets TCP/IP port offset (default=0)\n"
"  -tcpport <n>     -- binds to TCP/IP port <n> (default=baseport+portoffset)\n"
"  -maxaltports <n> -- tries <n> successive ports if tcpport is already in use\n"
"  -readonly        -- disables writes\n"
"  -readwrite       -- enables writes (default)\n"
"  -nofork          -- runs in foreground (default is background daemon)\n"
				,FLOPPYD_DEFAULT_PORT);
			goto abort;
		}
	}

	if (!ptalName) goto syntaxError;
	dev=ptalDeviceOpen(ptalName);
	if (!dev) {
		_PTAL_LOG_ERROR("ptal-photod(%s): ptalDeviceOpen failed!\n",
			ptalName);
		goto abort;
	}
	chan=ptalChannelAllocate(dev);
	if (!chan) {
		_PTAL_LOG_ERROR("ptal-photod(%s): ptalChannelAllocate "
			"failed!\n",ptalName);
		goto abort;
	}
	ptalChannelSetRemoteService(chan,PTAL_STYPE_GENERIC,
		HPPSP_SOCKET_DISK_IO,HPPSP_SERVICE_DISK_IO);

	fdSocket=socket(PF_INET,SOCK_STREAM,0);
	if (fdSocket<0) {
		_PTAL_LOG_ERROR("ptal-photod(%s): socket() failed, "
			"errno=%d!\n",ptalName,errno);
		goto abort;
	}

	if (tcpPort<0) tcpPort=basePort+portOffset;
	addrServer.sin_family=AF_INET;
	while (42) {
		addrServer.sin_port=htons(tcpPort);
		if (bind(fdSocket,(struct sockaddr *)&addrServer,
		     lenSockAddr)>=0) {
			break;
		}
		r=errno;
		if ( /* r==EADDRINUSE && */ maxAltPorts>0) {
			tcpPort++;
			maxAltPorts--;
		} else {
			_PTAL_LOG_ERROR("ptal-photod(%s): bind(tcpPort=%d) "
				"failed, errno=%d!\n",ptalName,tcpPort,r);
			if (r==EADDRINUSE) {
				_PTAL_LOG_ERROR("ptal-photod(%s): "
					"tcpPort=%d is already in use!\n",
					ptalName,tcpPort);
			}
			goto abort;
		}
	}

	if (listen(fdSocket,MAX_CONNECTIONS)<0) {
		_PTAL_LOG_ERROR("ptal-photod(%s): listen() failed, errno=%d!\n",
			ptalName,errno);
		goto abort;
	}

	if (!noFork) {
		r=fork();
		if (r<0) {
			_PTAL_LOG_ERROR("ptal-photod(%s): fork() failed, "
				"errno=%d!\n",ptalName,errno);
			goto abort;
		}
		if (r) goto done;
	}

	signal(SIGPIPE,signalHandler);

	syslog(LOG_LPR|LOG_NOTICE,
		"ptal-photod(%s) successfully initialized, "
		"listening on port %d.\n",ptalName,tcpPort);

	while (42) {
		lenSockAddr=sizeof(addrClient);
		fdSession=accept(fdSocket,
			(struct sockaddr *)&addrClient,&lenSockAddr);
		if (fdSession<0) {
			_PTAL_LOG_ERROR("ptal-photod(%s): accept() failed, "
				"errno=%d!\n",ptalName,errno);
			goto abort;
		}
		handleSession(fdSession,chan);
	}

done:
	retcode=0;
abort:
	return retcode;
}
