/* libsane-hpoj -- SANE backend for hpoj-supported multi-function peripherals */

/* Copyright (C) 2001-2003 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */
 
/* Original author: David Paschal */


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "hpoj.h"
#include "hpoj-tables.h"

static SANE_Device **hpojDeviceList=0;
static int hpojDeviceListCount=0;

static int hpojPmlSelectCallback(ptalChannel_t chan,void *cbd);


static unsigned long hpojDivideAndShift(int line,
    unsigned long numerator1,unsigned long numerator2,
    unsigned long denominator,int shift) {
	unsigned long result,remainder,shiftLoss=0;
	unsigned long long ll=numerator1;
	ll*=numerator2;
	if (shift>0) ll<<=shift;
	remainder=ll%denominator;
	ll/=denominator;
	if (shift<0) {
		shiftLoss=ll&((1<<(-shift))-1);
		ll>>=(-shift);
	}
	result=ll;
	if (result!=ll) {
		PTAL_LOG_ERROR("hpojDivideAndShift(line=%d,"
			"num1=%lu,num2=%lu,denom=%lu,shift=%d): "
			"result=%lu truncated from %llu!\n",
			line,numerator1,numerator2,denominator,shift,
			result,ll);
#if 0
	} else if (remainder || shiftLoss) {
		PTAL_LOG_ERROR("hpojDivideAndShift(line=%d,"
			"num1=%lu,num2=%lu,denom=%lu,shift=%d): "
			"result=%lu, remainder=%lu, shiftLoss=%lu!\n",
			line,numerator1,numerator2,denominator,shift,
			result,remainder,shiftLoss);
#endif
#if 0
	} else {
		PTAL_LOG_DEBUG("hpojDivideAndShift(line=%d,"
			"num1=%lu,num2=%lu,denom=%lu,shift=%d): "
			"result=%lu=0x%8.8lX.\n",
			line,numerator1,numerator2,denominator,shift,
			result,result);
#endif
	}
	return result;
}


static SANE_Status hpojSclSendCommand(hpojScanner_t hpoj,int cmd,int param) {
	static struct timeval startTimeout={SCL_SEND_COMMAND_START_TIMEOUT,0};
	static struct timeval continueTimeout={SCL_SEND_COMMAND_CONTINUE_TIMEOUT,0};
	char buffer[LEN_SCL_BUFFER];
	int datalen;
	char punc=SCL_CMD_PUNC(cmd);
	char letter1=SCL_CMD_LETTER1(cmd),letter2=SCL_CMD_LETTER2(cmd);

	PTAL_LOG_DEBUG("hpoj:%s: hpojSclSendCommand(cmd=%d, param=%d, "
		"punc=<%c>, letter1=<%c>, letter2=<%c>)\n",
		hpoj->saneDevice.name,cmd,param,punc,letter1,letter2);
	if (cmd==SCL_CMD_INQUIRE_PRESENT_VALUE ||
	    cmd==SCL_CMD_INQUIRE_MINIMUM_VALUE ||
	    cmd==SCL_CMD_INQUIRE_MAXIMUM_VALUE) {
		PTAL_LOG_DEBUG("hpoj:%s: Inquiring about cmd=%d, "
			"punc=<%c>, letter1=<%c>, letter2=<%c>.\n",
			hpoj->saneDevice.name,param,SCL_CMD_PUNC(param),
			SCL_CMD_LETTER1(param),SCL_CMD_LETTER2(param));
	}

	if (cmd==SCL_CMD_RESET) {
		datalen=snprintf(buffer,LEN_SCL_BUFFER,"\x1B%c",
			letter2);
	} else {
	    if (cmd==SCL_CMD_CLEAR_ERROR_STACK) {
		datalen=snprintf(buffer,LEN_SCL_BUFFER,"\x1B%c%c%c",
			punc,letter1,letter2);
	    } else {
		datalen=snprintf(buffer,LEN_SCL_BUFFER,"\x1B%c%c%d%c",
			punc,letter1,param,letter2);
	    }
	    ptalChannelFlush(hpoj->chan,&startTimeout,&continueTimeout);
	}
	PTAL_LOG_DEBUG("hpoj:%s: Sending SCL command <<ESC>%s>>\n",
		hpoj->saneDevice.name,buffer+1);

	if (ptalChannelWrite(hpoj->chan,buffer,datalen)!=datalen) {

		return SANE_STATUS_IO_ERROR;
	}

	if (cmd==SCL_CMD_RESET) {
		ptalChannelFlush(hpoj->chan,&startTimeout,&continueTimeout);
	}

	return SANE_STATUS_GOOD;
}

static SANE_Status hpojSclInquire(hpojScanner_t hpoj,int cmd,int param,
    int *pValue,char *buffer,int maxlen) {
	SANE_Status retcode;
	static struct timeval startTimeout={SCL_INQUIRE_START_TIMEOUT,0};
	static struct timeval continueTimeout={SCL_INQUIRE_CONTINUE_TIMEOUT,0};
	int lenResponse,len,value;
	char _response[LEN_SCL_BUFFER+1],*response=_response;
	char expected[LEN_SCL_BUFFER],expectedChar;

	if (!pValue) pValue=&value;
	if (buffer && maxlen>0) memset(buffer,0,maxlen);
	memset(_response,0,LEN_SCL_BUFFER+1);

	/* Send inquiry command. */
	if ((retcode=hpojSclSendCommand(hpoj,cmd,param))!=SANE_STATUS_GOOD) {
		return retcode;
	}

	/* Figure out what format of response we expect. */
	expectedChar=SCL_CMD_LETTER2(cmd)-'A'+'a'-1;
	if (expectedChar=='q') expectedChar--;
	len=snprintf(expected,LEN_SCL_BUFFER,"\x1B%c%c%d%c",
		SCL_CMD_PUNC(cmd),SCL_CMD_LETTER1(cmd),
		param,expectedChar);

	/* Read the response. */
	lenResponse=ptalSclChannelRead(hpoj->chan,response,LEN_SCL_BUFFER,
		&startTimeout,&continueTimeout,1);
	PTAL_LOG_DEBUG("hpoj:%s: Got response (len=%d) <<ESC>%s>.\n",
		hpoj->saneDevice.name,lenResponse,response+1);

	/* Validate the first part of the response. */
	if (lenResponse<=len || memcmp(response,expected,len)) {
		PTAL_LOG_WARN("hpoj:%s: hpojSclInquire(cmd=%d,param=%d) "
			"didn't get expected response of <<ESC>%s>!\n",
			hpoj->saneDevice.name,cmd,param,expected+1);
		return SANE_STATUS_IO_ERROR;
	}
	response+=len;
	lenResponse-=len;

	/* Null response? */
	if (response[0]=='N') {
		PTAL_LOG_DEBUG("hpoj:%s: Got null response.\n",
			hpoj->saneDevice.name);
		return SANE_STATUS_UNSUPPORTED;
	}

	/* Parse integer part of non-null response.
	 * If this is a binary-data response, then this value is the
	 * length of the binary-data portion. */
	if (sscanf(response,"%d%n",pValue,&len)!=1) {
		PTAL_LOG_WARN("hpoj:%s: hpojSclInquire(cmd=%d,param=%d) "
			"didn't find integer!\n",
			hpoj->saneDevice.name,cmd,param);
		return SANE_STATUS_IO_ERROR;
	}

	/* Integer response? */
	if (response[len]=='V') {
		return SANE_STATUS_GOOD;
	}

	/* Binary-data response? */
	if (response[len]!='W') {
		PTAL_LOG_WARN("hpoj:%s: hpojSclInquire(cmd=%d,param=%d): "
			"Unexpected character '%c'!\n",
			hpoj->saneDevice.name,cmd,param,response[len]);
		return SANE_STATUS_IO_ERROR;
	}
	response+=len+1;
	lenResponse-=len+1;

	/* Make sure we got the right length of binary data. */
	if (lenResponse<0 || lenResponse!=*pValue || lenResponse>maxlen) {
		PTAL_LOG_WARN("hpoj:%s: hpojSclInquire(cmd=%d,param=%d) "
			"unexpected binary data lenResponse=%d, *pValue=%d, "
			"and/or maxlen=%d!\n",
			hpoj->saneDevice.name,cmd,param,
			lenResponse,*pValue,maxlen);
		return SANE_STATUS_IO_ERROR;
	}

	/* Copy binary data into user's buffer. */
	if (buffer) {
		maxlen=*pValue;
		memcpy(buffer,response,maxlen);
	}

	return SANE_STATUS_GOOD;
}

static SANE_Status hpojScannerToSaneError(hpojScanner_t hpoj) {
	SANE_Status retcode;

    if (hpoj->scannerType==SCANNER_TYPE_SCL) {
	int sclError;

	retcode=hpojSclInquire(hpoj,SCL_CMD_INQUIRE_DEVICE_PARAMETER,
		SCL_INQ_CURRENT_ERROR,&sclError,0,0);

	if (retcode==SANE_STATUS_UNSUPPORTED) {
		/* PTAL_LOG_DEBUG("hpoj:%s: hpojScannerToSaneError: "
			"no sclError.\n",hpoj->saneDevice.name); */
		retcode=SANE_STATUS_GOOD;

	} else if (retcode==SANE_STATUS_GOOD) {
		PTAL_LOG_DEBUG("hpoj:%s: hpojScannerToSaneError: "
			"sclError=%d.\n",hpoj->saneDevice.name,sclError);

		switch (sclError) {
		   case SCL_ERROR_UNRECOGNIZED_COMMAND:
		   case SCL_ERROR_PARAMETER_ERROR:
			retcode=SANE_STATUS_UNSUPPORTED;
			break;

		   case SCL_ERROR_NO_MEMORY:
			retcode=SANE_STATUS_NO_MEM;
			break;

		   case SCL_ERROR_CANCELLED:
			retcode=SANE_STATUS_CANCELLED;
			break;

		   case SCL_ERROR_PEN_DOOR_OPEN:
			retcode=SANE_STATUS_COVER_OPEN;
			break;

		   case SCL_ERROR_SCANNER_HEAD_LOCKED:
		   case SCL_ERROR_ADF_PAPER_JAM:
		   case SCL_ERROR_HOME_POSITION_MISSING:
		   case SCL_ERROR_ORIGINAL_ON_GLASS:
			retcode=SANE_STATUS_JAMMED;
			break;

		   case SCL_ERROR_PAPER_NOT_LOADED:
			retcode=SANE_STATUS_NO_DOCS;
			break;

		   default:
			retcode=SANE_STATUS_IO_ERROR;
			break;
		}
	}

    } else /* if (hpoj->scannerType==SCANNER_TYPE_PML) */ {
	int pmlError,type;

	if (ptalPmlRequestGet(hpoj->pml.objUploadError,0)==PTAL_ERROR) {
		retcode=SANE_STATUS_GOOD;

	} else if (ptalPmlGetIntegerValue(hpoj->pml.objUploadError,&type,
	    &pmlError)==PTAL_ERROR) {
		PTAL_LOG_WARN("hpoj:%s: hpojScannerToSaneError: "
			"ptalPmlGetIntegerValue failed, type=%d!\n",
			hpoj->saneDevice.name,type);
		retcode=SANE_STATUS_IO_ERROR;

	} else {
		PTAL_LOG_DEBUG("hpoj:%s: hpojScannerToSaneError: "
			"pmlError=%d.\n",hpoj->saneDevice.name,pmlError);

		switch (pmlError) {
		   case PML_UPLOAD_ERROR_SCANNER_JAM:
			retcode=SANE_STATUS_JAMMED;
			break;

		   case PML_UPLOAD_ERROR_MLC_CHANNEL_CLOSED:
		   case PML_UPLOAD_ERROR_STOPPED_BY_HOST:
		   case PML_UPLOAD_ERROR_STOP_KEY_PRESSED:
			retcode=SANE_STATUS_CANCELLED;
			break;

		   case PML_UPLOAD_ERROR_NO_DOC_IN_ADF:
		   case PML_UPLOAD_ERROR_DOC_LOADED:
			retcode=SANE_STATUS_NO_DOCS;
			break;

		   case PML_UPLOAD_ERROR_COVER_OPEN:
			retcode=SANE_STATUS_COVER_OPEN;
			break;

		   case PML_UPLOAD_ERROR_DEVICE_BUSY:
			retcode=SANE_STATUS_DEVICE_BUSY;
			break;

		   default:
			retcode=SANE_STATUS_IO_ERROR;
			break;
		}
	}
    }

	PTAL_LOG_DEBUG("hpoj:%s: hpojScannerToSaneError returns %d.\n",
		hpoj->saneDevice.name,retcode);
	return retcode;
}

static SANE_Status hpojSclSendCommandCheckError(hpojScanner_t hpoj,
    int cmd,int param) {
	SANE_Status retcode;

	hpojSclSendCommand(hpoj,SCL_CMD_CLEAR_ERROR_STACK,0);

	retcode=hpojSclSendCommand(hpoj,cmd,param);
	if (retcode==SANE_STATUS_GOOD &&
	    ((cmd!=SCL_CMD_CHANGE_DOCUMENT && cmd!=SCL_CMD_UNLOAD_DOCUMENT) ||
	     hpoj->beforeScan)) {
		retcode=hpojScannerToSaneError(hpoj);
	}

	return retcode;
}

static SANE_Status hpojScannerToSaneStatus(hpojScanner_t hpoj) {
	SANE_Status retcode;

    if (hpoj->scannerType==SCANNER_TYPE_SCL) {
	int sclStatus;

	retcode=hpojSclInquire(hpoj,SCL_CMD_INQUIRE_DEVICE_PARAMETER,
		SCL_INQ_ADF_FEED_STATUS,&sclStatus,0,0);

	if (retcode==SANE_STATUS_UNSUPPORTED) {
		retcode=SANE_STATUS_GOOD;

	} else if (retcode==SANE_STATUS_GOOD) {
		PTAL_LOG_DEBUG("hpoj:%s: hpojScannerToSaneStatus: "
			"sclStatus=%d.\n",hpoj->saneDevice.name,sclStatus);

		switch (sclStatus) {
		   case SCL_ADF_FEED_STATUS_OK:
			retcode=SANE_STATUS_GOOD;
			break;

		   case SCL_ADF_FEED_STATUS_BUSY:
			/* retcode=SANE_STATUS_DEVICE_BUSY; */
			retcode=SANE_STATUS_GOOD;
			break;

		   case SCL_ADF_FEED_STATUS_PAPER_JAM:
		   case SCL_ADF_FEED_STATUS_ORIGINAL_ON_GLASS:
			retcode=SANE_STATUS_JAMMED;
			break;

		   case SCL_ADF_FEED_STATUS_PORTRAIT_FEED:
			retcode=SANE_STATUS_UNSUPPORTED;
			break;

		   default:
			retcode=SANE_STATUS_IO_ERROR;
			break;
		}
	}

    } else /* if (hpoj->scannerType==SCANNER_TYPE_PML) */ {
	int pmlStatus,type;

	if (ptalPmlRequestGet(hpoj->pml.objScannerStatus,0)==PTAL_ERROR) {
		retcode=SANE_STATUS_GOOD;

	} else if (ptalPmlGetIntegerValue(hpoj->pml.objScannerStatus,&type,
	     &pmlStatus)==PTAL_ERROR) {
		PTAL_LOG_WARN("hpoj:%s: hpojScannerToSaneStatus: "
			"ptalPmlGetIntegerValue failed, type=%d!\n",
			hpoj->saneDevice.name,type);
		retcode=SANE_STATUS_IO_ERROR;

	} else {
		PTAL_LOG_DEBUG("hpoj:%s: hpojScannerToSaneStatus: "
			"pmlStatus=0x%2.2X.\n",
			hpoj->saneDevice.name,pmlStatus);

		if (pmlStatus&PML_SCANNER_STATUS_FEEDER_JAM) {
			retcode=SANE_STATUS_JAMMED;
		} else if (pmlStatus&PML_SCANNER_STATUS_FEEDER_OPEN) {
			retcode=SANE_STATUS_COVER_OPEN;
		} else if (pmlStatus&PML_SCANNER_STATUS_FEEDER_EMPTY) {
			if (hpoj->currentAdfMode!=ADF_MODE_ADF &&
			    hpoj->beforeScan) {
				retcode=SANE_STATUS_GOOD;
			} else {
				retcode=SANE_STATUS_NO_DOCS;
			}
		} else if (pmlStatus&PML_SCANNER_STATUS_INVALID_MEDIA_SIZE) {
			retcode=SANE_STATUS_INVAL;
		} else if (pmlStatus) {
			retcode=SANE_STATUS_IO_ERROR;
		} else {
			retcode=SANE_STATUS_GOOD;
		}
	}
    }

	PTAL_LOG_DEBUG("hpoj:%s: hpojScannerToSaneStatus returns %d.\n",
		hpoj->saneDevice.name,retcode);
	return retcode;
}

static int hpojScannerIsUninterruptible(hpojScanner_t hpoj,int *pUploadState) {
	int uploadState;
	if (!pUploadState) pUploadState=&uploadState;

	return (hpoj->scannerType==SCANNER_TYPE_PML &&
		hpoj->pml.scanDone &&
		ptalPmlRequestGet(hpoj->pml.objUploadState,0)!=PTAL_ERROR &&
		ptalPmlGetIntegerValue(hpoj->pml.objUploadState,0,
			pUploadState)!=PTAL_ERROR &&
		(*pUploadState==PML_UPLOAD_STATE_START ||
		 *pUploadState==PML_UPLOAD_STATE_ACTIVE ||
		 *pUploadState==PML_UPLOAD_STATE_NEWPAGE));
}

static SANE_Status hpojResetScanner(hpojScanner_t hpoj) {
	SANE_Status retcode;

	PTAL_LOG_DEBUG("hpoj:%s: hpojResetScanner\n",hpoj->saneDevice.name);

	if (hpoj->scannerType==SCANNER_TYPE_SCL) {
		retcode=hpojSclSendCommand(hpoj,SCL_CMD_RESET,0);
		if (retcode!=SANE_STATUS_GOOD) return retcode;

	} else /* if (hpoj->scannerType==SCANNER_TYPE_PML) */ {
		if (!hpojScannerIsUninterruptible(hpoj,0)) {
			ptalPmlSetIntegerValue(hpoj->pml.objUploadState,
				PTAL_PML_TYPE_ENUMERATION,
				PML_UPLOAD_STATE_IDLE);
			if (ptalPmlRequestSetRetry(hpoj->pml.objUploadState,
			     0,0)==PTAL_ERROR) {

				return SANE_STATUS_IO_ERROR;
			}
		}

		/* Clear upload error for the sake of the LaserJet 1100A. */
		ptalPmlSetIntegerValue(hpoj->pml.objUploadError,
			PTAL_PML_TYPE_SIGNED_INTEGER,0);
		ptalPmlRequestSet(hpoj->pml.objUploadError);	/* No retry. */
	}

	return SANE_STATUS_GOOD;
}

static SANE_Status hpojPmlAllocateObjects(hpojScanner_t hpoj) {
	if (hpoj->scannerType==SCANNER_TYPE_PML &&
	    !hpoj->pml.objScanToken) {
		int len;

		hpoj->pml.objScannerStatus=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x2\x1");
		hpoj->pml.objResolutionRange=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x2\x3");
		hpoj->pml.objUploadTimeout=
			ptalPmlAllocateID(hpoj->dev,"\x1\x1\x1\x12");
		hpoj->pml.objContrast=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\x1");
		hpoj->pml.objResolution=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\x2");
		hpoj->pml.objPixelDataType=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\x3");
		hpoj->pml.objCompression=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\x4");
		hpoj->pml.objCompressionFactor=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\x5");
		hpoj->pml.objUploadError=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\x6");
		hpoj->pml.objUploadState=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\xC");
		hpoj->pml.objAbcThresholds=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\xE");
		hpoj->pml.objSharpeningCoefficient=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\xF");
		hpoj->pml.objNeutralClipThresholds=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\x1F");
		hpoj->pml.objToneMap=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\x20");
		hpoj->pml.objCopierReduction=
			ptalPmlAllocateID(hpoj->dev,"\x1\x5\x1\x4");
		hpoj->pml.objScanToken=
			ptalPmlAllocateID(hpoj->dev,"\x1\x1\x1\x19");
		hpoj->pml.objModularHardware=
			ptalPmlAllocateID(hpoj->dev,"\x1\x2\x2\x1\x4B");

		if (ptalPmlRequestGet(hpoj->pml.objScanToken,0)!=PTAL_ERROR &&
		    (len=ptalPmlGetValue(hpoj->pml.objScanToken,0,
		      hpoj->pml.scanToken,PTAL_PML_MAX_VALUE_LEN))>0) {
			int i;
			hpoj->pml.lenScanToken=len;
			PTAL_LOG_DEBUG("hpoj:%s: lenScanToken=%d.\n",
				hpoj->saneDevice.name,hpoj->pml.lenScanToken);
			for (i=0;i<len;i++) {
				hpoj->pml.scanToken[i]=0;
				hpoj->pml.zeroScanToken[i]=0;
			}
			gettimeofday((struct timeval *)hpoj->pml.scanToken,0);
			i=sizeof(struct timeval);
			*((pid_t *)(hpoj->pml.scanToken+i))=getpid();
			i+=sizeof(pid_t);
			*((pid_t *)(hpoj->pml.scanToken+i))=getppid();

			if (getenv("SANE_HPOJ_RESET_SCAN_TOKEN")) {
				ptalPmlSetValue(hpoj->pml.objScanToken,
					PTAL_PML_TYPE_BINARY,
					hpoj->pml.zeroScanToken,
					hpoj->pml.lenScanToken);
				ptalPmlRequestSetRetry(
					hpoj->pml.objScanToken,0,0);
			}
		}
	}

	return SANE_STATUS_GOOD;
}

static void hpojConnClose(hpojScanner_t hpoj) {
	PTAL_LOG_DEBUG("hpoj:%s: hpojConnClose\n",hpoj->saneDevice.name);

	ptalChannelClose(hpoj->chan);
	if (hpoj->pml.scanTokenIsSet) {
		ptalPmlSetValue(hpoj->pml.objScanToken,
			PTAL_PML_TYPE_BINARY,
			hpoj->pml.zeroScanToken,hpoj->pml.lenScanToken);
		ptalPmlRequestSetRetry(hpoj->pml.objScanToken,0,0);
		hpoj->pml.scanTokenIsSet=0;
	}
	ptalPmlClose(hpoj->dev);
	ptalChannelClose(hpoj->chanReserve);
}

static SANE_Status hpojConnOpen(hpojScanner_t hpoj) {
	SANE_Status retcode;

	PTAL_LOG_DEBUG("hpoj:%s: hpojConnOpen\n",hpoj->saneDevice.name);

	if (hpoj->scannerType==SCANNER_TYPE_SCL) {
		if (ptalChannelOpenOrReopen(hpoj->chan)==PTAL_ERROR) {
			/* retcode=SANE_STATUS_IO_ERROR; */
			retcode=SANE_STATUS_DEVICE_BUSY;
			goto abort;
		}

	} else /* if (hpoj->scannerType==SCANNER_TYPE_PML) */ {
		if (ptalPmlOpen(hpoj->dev)==PTAL_ERROR) {
			retcode=SANE_STATUS_IO_ERROR;
			goto abort;
		}

		hpojPmlAllocateObjects(hpoj);

		if (!hpoj->pml.openFirst) {
			if (ptalChannelOpen(hpoj->chanReserve)==PTAL_ERROR) {
				retcode=SANE_STATUS_DEVICE_BUSY;
				goto abort;
			}
		} else if (hpoj->pml.lenScanToken) {
			ptalPmlSetValue(hpoj->pml.objScanToken,
				PTAL_PML_TYPE_BINARY,
				hpoj->pml.scanToken,hpoj->pml.lenScanToken);
			if (ptalPmlRequestSetRetry(hpoj->pml.objScanToken,0,0)==
			    PTAL_ERROR) {
				retcode=SANE_STATUS_DEVICE_BUSY;
				goto abort;
			}
			hpoj->pml.scanTokenIsSet=1;
		} else {
			/* Don't call ptalChannelOpenOrReopen() in this
			 * case, because ptalChannelIsStale() consumes
			 * a character. */
			if (ptalChannelOpen(hpoj->chan)==PTAL_ERROR) {
				/* retcode=SANE_STATUS_IO_ERROR; */
				retcode=SANE_STATUS_DEVICE_BUSY;
				goto abort;
			}
		}
	}

	retcode=hpojResetScanner(hpoj);
abort:
	if (retcode!=SANE_STATUS_GOOD) {
		hpojConnClose(hpoj);
	}
	return retcode;
}

static SANE_Status hpojConnPrepareScan(hpojScanner_t hpoj) {
	SANE_Status retcode;

	PTAL_LOG_DEBUG("hpoj:%s: hpojConnPrepareScan\n",hpoj->saneDevice.name);

	retcode=hpojConnOpen(hpoj);
	if (retcode!=SANE_STATUS_GOOD) return retcode;

    if (hpoj->scannerType==SCANNER_TYPE_SCL) {
	int i;

	/* Reserve scanner and make sure it got reserved. */
	hpojSclSendCommand(hpoj,SCL_CMD_SET_DEVICE_LOCK,1);
	hpojSclSendCommand(hpoj,SCL_CMD_SET_DEVICE_LOCK_TIMEOUT,
		SCL_DEVICE_LOCK_TIMEOUT);
	for (i=0;;i++) {
		char buffer[LEN_SCL_BUFFER];
		int len,j;
		struct timeval tv1,tv2;
		gettimeofday(&tv1,0);
		if (hpojSclInquire(hpoj,SCL_CMD_INQUIRE_DEVICE_PARAMETER,
		     SCL_INQ_SESSION_ID,&len,buffer,LEN_SCL_BUFFER)!=
		    SANE_STATUS_GOOD) {
			break;
		}
		gettimeofday(&tv2,0);
		for (j=0;j<len && buffer[j]=='0';j++);
		if (j<len) break;
		if (i>=SCL_PREPARE_SCAN_DEVICE_LOCK_MAX_RETRIES) {
			return SANE_STATUS_DEVICE_BUSY;
		}
		PTAL_LOG_WARN("hpoj:%s: hpojConnPrepareScan: "
			"Waiting for device lock.\n",hpoj->saneDevice.name);

		if (((unsigned)(tv2.tv_sec-tv1.tv_sec))<=
		    SCL_PREPARE_SCAN_DEVICE_LOCK_DELAY) {
			sleep(SCL_PREPARE_SCAN_DEVICE_LOCK_DELAY);
#if 0
		} else {
			PTAL_LOG_DEBUG("hpoj:%s: hpojConnPrepareScan: "
				"Not sleeping because inquiry took too long.\n",
				hpoj->saneDevice.name);
#endif
		}
	}
    }

	return SANE_STATUS_GOOD;
}

static SANE_Status hpojConnBeginScan(hpojScanner_t hpoj,int alreadyStarted) {
	PTAL_LOG_DEBUG("hpoj:%s: hpojConnBeginScan\n",hpoj->saneDevice.name);

	/* Open scan channel if we need to do it after starting the scan. */
	if (hpoj->scannerType==SCANNER_TYPE_PML) {
		if (!hpoj->pml.openFirst) {
			if (alreadyStarted) {
openScanChannel:
				if (ptalChannelOpen(hpoj->chan)==PTAL_ERROR) {
					return SANE_STATUS_IO_ERROR;
				}
			}
		} else if (hpoj->pml.lenScanToken) {
			if (!alreadyStarted) {
				goto openScanChannel;
			}
		} else {
			/* For this case we already opened the scan channel
			 * in hpojConnOpen(). */
		}
	}

	return SANE_STATUS_GOOD;
}

static void hpojConnEndScan(hpojScanner_t hpoj) {
	PTAL_LOG_DEBUG("hpoj:%s: hpojConnEndScan\n",hpoj->saneDevice.name);

	hpojResetScanner(hpoj);
	hpojConnClose(hpoj);
}

static void hpojResetScannerIfInNewPageState(hpojScanner_t hpoj) {
	int uploadState;

	if (hpojScannerIsUninterruptible(hpoj,&uploadState) &&
	    uploadState==PML_UPLOAD_STATE_NEWPAGE &&
	    !hpoj->pml.dontResetBeforeNextNonBatchPage) {
		hpoj->pml.scanDone=0;
		hpojConnEndScan(hpoj);
	}
}

static void hpojNumListClear(int *list) {
	memset(list,0,sizeof(int)*MAX_LIST_SIZE);
}

static int hpojNumListIsInList(int *list,int n) {
	int i;
	for (i=1;i<MAX_LIST_SIZE;i++) {
		if (list[i]==n) return 1;
	}
	return 0;
}

static int hpojNumListAdd(int *list,int n) {
	if (hpojNumListIsInList(list,n)) return 1;
	if (list[0]>=(MAX_LIST_SIZE-1)) return 0;
	list[0]++;
	list[list[0]]=n;
	return 1;
}

static int hpojNumListGetCount(int *list) {
	return list[0];
}

static int hpojNumListGetFirst(int *list) {
	int n=list[0];
	if (n>0) n=list[1];
	return n;
}

static void hpojStrListClear(SANE_String_Const *list) {
	memset(list,0,sizeof(char *)*MAX_LIST_SIZE);
}

static int hpojStrListIsInList(SANE_String_Const *list,char *s) {
	while (*list) {
		if (!strcasecmp(*list,s)) return 1;
		list++;
	}
	return 0;
}

static int hpojStrListAdd(SANE_String_Const *list,char *s) {
	int i;
	for (i=0;i<MAX_LIST_SIZE-1;i++) {
		if (!list[i]) {
			list[i]=s;
			return 1;
		}
		if (!strcasecmp(list[i],s)) return 1;
	}
	return 0;
}

#if 0
static int hpojStrListGetCount(SANE_String_Const *list) {
	int i;
	for (i=0;list[i];i++);
	return i;
}
#endif

static SANE_Status hpojSetDefaultValue(hpojScanner_t hpoj,int option) {
	PTAL_LOG_DEBUG("hpoj:%s: hpojSetDefaultValue(option=%d)\n",
		hpoj->saneDevice.name,option);

	switch (option) {
	   case OPTION_SCAN_MODE:
		if (hpoj->supportsScanMode[SCAN_MODE_COLOR]) {
			hpoj->currentScanMode=SCAN_MODE_COLOR;
		} else if (hpoj->supportsScanMode[SCAN_MODE_GRAYSCALE]) {
			hpoj->currentScanMode=SCAN_MODE_GRAYSCALE;
		} else /* if (hpoj->supportsScanMode[SCAN_MODE_LINEART]) */ {
			hpoj->currentScanMode=SCAN_MODE_LINEART;
		}
		break;

	   case OPTION_SCAN_RESOLUTION:
		if (hpoj->option[OPTION_SCAN_RESOLUTION].constraint_type==
		     SANE_CONSTRAINT_WORD_LIST) {
			hpoj->currentResolution=hpojNumListGetFirst(
				(SANE_Int *)hpoj->option[
				OPTION_SCAN_RESOLUTION].constraint.word_list);
		} else {
			hpoj->currentResolution=hpoj->resolutionRange.min;
		}
		break;

	   case OPTION_CONTRAST:
		hpoj->currentContrast=hpoj->defaultContrast;
		break;

	   case OPTION_COMPRESSION:
	     {
		int supportedCompression=hpoj->supportsScanMode[
			hpoj->currentScanMode];
		int defaultCompression=hpoj->defaultCompression[
			hpoj->currentScanMode];

		if (supportedCompression&defaultCompression) {
			hpoj->currentCompression=defaultCompression;
		} else if (supportedCompression&COMPRESSION_NONE) {
			hpoj->currentCompression=COMPRESSION_NONE;
		} else if (supportedCompression&COMPRESSION_MH) {
			hpoj->currentCompression=COMPRESSION_MH;
		} else if (supportedCompression&COMPRESSION_MR) {
			hpoj->currentCompression=COMPRESSION_MR;
		} else if (supportedCompression&COMPRESSION_MMR) {
			hpoj->currentCompression=COMPRESSION_MMR;
		} else if (supportedCompression&COMPRESSION_JPEG) {
			hpoj->currentCompression=COMPRESSION_JPEG;
		} else {
			hpoj->currentCompression=COMPRESSION_NONE;
		}
	     }
		break;

	   case OPTION_JPEG_COMPRESSION_FACTOR:
		hpoj->currentJpegCompressionFactor=
			hpoj->defaultJpegCompressionFactor;
		break;

	   case OPTION_BATCH_SCAN:
		hpoj->currentBatchScan=SANE_FALSE;
		break;

	   case OPTION_ADF_MODE:
		if (hpoj->supportedAdfModes&ADF_MODE_AUTO) {
			if (hpoj->scannerType==SCANNER_TYPE_PML &&
			    !hpoj->pml.flatbedCapability &&
			    hpoj->supportedAdfModes&ADF_MODE_ADF) {
				goto defaultToAdf;
			}
			hpoj->currentAdfMode=ADF_MODE_AUTO;
		} else if (hpoj->supportedAdfModes&ADF_MODE_FLATBED) {
			hpoj->currentAdfMode=ADF_MODE_FLATBED;
		} else if (hpoj->supportedAdfModes&ADF_MODE_ADF) {
defaultToAdf:
			hpoj->currentAdfMode=ADF_MODE_ADF;
		} else {
			hpoj->currentAdfMode=ADF_MODE_AUTO;
		}
		break;

	   case OPTION_DUPLEX:
		hpoj->currentDuplex=SANE_FALSE;
		break;

	   case OPTION_LENGTH_MEASUREMENT:
		hpoj->currentLengthMeasurement=LENGTH_MEASUREMENT_PADDED;
		break;

	   case OPTION_TL_X:
		hpoj->currentTlx=hpoj->tlxRange.min;
		break;

	   case OPTION_TL_Y:
		hpoj->currentTly=hpoj->tlyRange.min;
		break;

	   case OPTION_BR_X:
		hpoj->currentBrx=hpoj->brxRange.max;
		break;

	   case OPTION_BR_Y:
		hpoj->currentBry=hpoj->bryRange.max;
		break;

	   default:
		return SANE_STATUS_INVAL;
	}

	return SANE_STATUS_GOOD;
}

static int hpojUpdateDescriptors(hpojScanner_t hpoj,int option) {
	int initValues=(option==OPTION_FIRST);
	int reload=0;

	PTAL_LOG_DEBUG("hpoj:%s: hpojUpdateDescriptors(option=%d)\n",
		hpoj->saneDevice.name,option);

	/* OPTION_SCAN_MODE: */
	if (initValues) {
		hpojStrListClear(hpoj->scanModeList);
		if (hpoj->supportsScanMode[SCAN_MODE_LINEART]) {
			hpojStrListAdd(hpoj->scanModeList,
				STR_SCAN_MODE_LINEART);
		}
		if (hpoj->supportsScanMode[SCAN_MODE_GRAYSCALE]) {
			hpojStrListAdd(hpoj->scanModeList,
				STR_SCAN_MODE_GRAYSCALE);
		}
		if (hpoj->supportsScanMode[SCAN_MODE_COLOR]) {
			hpojStrListAdd(hpoj->scanModeList,
				STR_SCAN_MODE_COLOR);
		}
		hpojSetDefaultValue(hpoj,OPTION_SCAN_MODE);
		reload|=SANE_INFO_RELOAD_OPTIONS;
		reload|=SANE_INFO_RELOAD_PARAMS;
	} else if (option==OPTION_SCAN_MODE) {
		reload|=SANE_INFO_RELOAD_PARAMS;
	}

	/* OPTION_SCAN_RESOLUTION: */
	if (hpoj->option[OPTION_SCAN_RESOLUTION].constraint_type==
	     SANE_CONSTRAINT_WORD_LIST) {
		SANE_Int **pList=(SANE_Int **)&hpoj->option[
			OPTION_SCAN_RESOLUTION].constraint.word_list;

		if (hpoj->currentScanMode==SCAN_MODE_LINEART) {
			if (*pList!=hpoj->lineartResolutionList) {
				*pList=hpoj->lineartResolutionList;
				reload|=SANE_INFO_RELOAD_OPTIONS;
			}
		} else {
			if (*pList!=hpoj->resolutionList) {
				*pList=hpoj->resolutionList;
				reload|=SANE_INFO_RELOAD_OPTIONS;
			}
		}
		if (initValues ||
		    !hpojNumListIsInList(*pList,hpoj->currentResolution)) {
			hpojSetDefaultValue(hpoj,OPTION_SCAN_RESOLUTION);
			reload|=SANE_INFO_RELOAD_OPTIONS;
			reload|=SANE_INFO_RELOAD_PARAMS;
		}
	} else {
		if (initValues ||
		    hpoj->currentResolution<hpoj->resolutionRange.min ||
		    hpoj->currentResolution>hpoj->resolutionRange.max) {
			hpojSetDefaultValue(hpoj,OPTION_SCAN_RESOLUTION);
			reload|=SANE_INFO_RELOAD_OPTIONS;
			reload|=SANE_INFO_RELOAD_PARAMS;
		}
	}
	if (option==OPTION_SCAN_RESOLUTION) {
		reload|=SANE_INFO_RELOAD_PARAMS;
	}

	/* OPTION_CONTRAST: */
	if (initValues) {
		hpojSetDefaultValue(hpoj,OPTION_CONTRAST);
	}

	/* OPTION_COMPRESSION: */
    {
	int supportedCompression=hpoj->supportsScanMode[hpoj->currentScanMode];
	if (initValues || !(supportedCompression&hpoj->currentCompression) ||
	    (((supportedCompression&COMPRESSION_NONE)!=0) !=
	     (hpojStrListIsInList(hpoj->compressionList,
	      STR_COMPRESSION_NONE)!=0)) ||
	    (((supportedCompression&COMPRESSION_MH)!=0) !=
	     (hpojStrListIsInList(hpoj->compressionList,
	      STR_COMPRESSION_MH)!=0)) ||
	    (((supportedCompression&COMPRESSION_MR)!=0) !=
	     (hpojStrListIsInList(hpoj->compressionList,
	      STR_COMPRESSION_MR)!=0)) ||
	    (((supportedCompression&COMPRESSION_MMR)!=0) !=
	     (hpojStrListIsInList(hpoj->compressionList,
	      STR_COMPRESSION_MMR)!=0)) ||
	    (((supportedCompression&COMPRESSION_JPEG)!=0) !=
	     (hpojStrListIsInList(hpoj->compressionList,
	      STR_COMPRESSION_JPEG)!=0)) ) {
		hpojStrListClear(hpoj->compressionList);
		if (supportedCompression&COMPRESSION_NONE) {
			hpojStrListAdd(hpoj->compressionList,
				STR_COMPRESSION_NONE);
		}
		if (supportedCompression&COMPRESSION_MH) {
			hpojStrListAdd(hpoj->compressionList,
				STR_COMPRESSION_MH);
		}
		if (supportedCompression&COMPRESSION_MR) {
			hpojStrListAdd(hpoj->compressionList,
				STR_COMPRESSION_MR);
		}
		if (supportedCompression&COMPRESSION_MMR) {
			hpojStrListAdd(hpoj->compressionList,
				STR_COMPRESSION_MMR);
		}
		if (supportedCompression&COMPRESSION_JPEG) {
			hpojStrListAdd(hpoj->compressionList,
				STR_COMPRESSION_JPEG);
		}
		hpojSetDefaultValue(hpoj,OPTION_COMPRESSION);
		reload|=SANE_INFO_RELOAD_OPTIONS;
	}
    }

	/* OPTION_JPEG_COMPRESSION_FACTOR: */
	if (initValues ||
	    ((hpoj->currentCompression==COMPRESSION_JPEG) !=
	     ((hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].cap&
	       SANE_CAP_INACTIVE)==0)) ) {
		if (hpoj->currentCompression==COMPRESSION_JPEG) {
			hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].cap&=
				~SANE_CAP_INACTIVE;
		} else {
			hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].cap|=
				SANE_CAP_INACTIVE;
		}
		hpojSetDefaultValue(hpoj,OPTION_JPEG_COMPRESSION_FACTOR);
		reload|=SANE_INFO_RELOAD_OPTIONS;
	}

	/* OPTION_BATCH_SCAN: */
	if (initValues) {
		hpojSetDefaultValue(hpoj,OPTION_BATCH_SCAN);
		if (hpoj->preDenali) {
			hpoj->option[OPTION_BATCH_SCAN].cap|=SANE_CAP_INACTIVE;
		}
		reload|=SANE_INFO_RELOAD_OPTIONS;
	}
	if (!hpoj->currentBatchScan) {
		hpoj->noDocsConditionPending=0;
	}

	/* OPTION_ADF_MODE: */
	if (initValues) {
		hpojStrListClear(hpoj->adfModeList);
		if (hpoj->supportedAdfModes&ADF_MODE_AUTO) {
			hpojStrListAdd(hpoj->adfModeList,STR_ADF_MODE_AUTO);
		}
		if (hpoj->supportedAdfModes&ADF_MODE_FLATBED) {
			hpojStrListAdd(hpoj->adfModeList,STR_ADF_MODE_FLATBED);
		}
		if (hpoj->supportedAdfModes&ADF_MODE_ADF) {
			hpojStrListAdd(hpoj->adfModeList,STR_ADF_MODE_ADF);
		}
		hpojSetDefaultValue(hpoj,OPTION_ADF_MODE);
		reload|=SANE_INFO_RELOAD_OPTIONS;
	}

	/* OPTION_DUPLEX: */
	if (initValues ||
	    ((hpoj->supportsDuplex && hpoj->currentAdfMode!=ADF_MODE_FLATBED) !=
	     ((hpoj->option[OPTION_DUPLEX].cap&SANE_CAP_INACTIVE)==0)) ) {
		if (hpoj->supportsDuplex &&
		    hpoj->currentAdfMode!=ADF_MODE_FLATBED) {
			hpoj->option[OPTION_DUPLEX].cap&=~SANE_CAP_INACTIVE;
		} else {
			hpoj->option[OPTION_DUPLEX].cap|=SANE_CAP_INACTIVE;
		}
		hpojSetDefaultValue(hpoj,OPTION_DUPLEX);
		reload|=SANE_INFO_RELOAD_OPTIONS;
	}

	/* OPTION_LENGTH_MEASUREMENT: */
	if (initValues) {
		hpojSetDefaultValue(hpoj,OPTION_LENGTH_MEASUREMENT);
		hpojStrListClear(hpoj->lengthMeasurementList);
		hpojStrListAdd(hpoj->lengthMeasurementList,
			STR_LENGTH_MEASUREMENT_UNKNOWN);
		if (hpoj->scannerType==SCANNER_TYPE_PML) {
			hpojStrListAdd(hpoj->lengthMeasurementList,
				STR_LENGTH_MEASUREMENT_UNLIMITED);
		}
		hpojStrListAdd(hpoj->lengthMeasurementList,
			STR_LENGTH_MEASUREMENT_APPROXIMATE);
		hpojStrListAdd(hpoj->lengthMeasurementList,
			STR_LENGTH_MEASUREMENT_PADDED);
		/* TODO: hpojStrListAdd(hpoj->lengthMeasurementList,
			STR_LENGTH_MEASUREMENT_EXACT); */
	}

	/* OPTION_TL_X, OPTION_TL_Y, OPTION_BR_X, OPTION_BR_Y: */
	if (initValues) {
		hpojSetDefaultValue(hpoj,OPTION_TL_X);
		hpojSetDefaultValue(hpoj,OPTION_TL_Y);
		hpojSetDefaultValue(hpoj,OPTION_BR_X);
		hpojSetDefaultValue(hpoj,OPTION_BR_Y);
		reload|=SANE_INFO_RELOAD_OPTIONS;
		goto processGeometry;
	} else if (option==OPTION_TL_X || option==OPTION_TL_Y ||
		   option==OPTION_BR_X || option==OPTION_BR_Y) {
processGeometry:
		hpoj->effectiveTlx=hpoj->currentTlx;
		hpoj->effectiveBrx=hpoj->currentBrx;
		FIX_GEOMETRY(hpoj->effectiveTlx,hpoj->effectiveBrx,
			hpoj->brxRange.min,hpoj->brxRange.max);
		hpoj->effectiveTly=hpoj->currentTly;
		hpoj->effectiveBry=hpoj->currentBry;
		FIX_GEOMETRY(hpoj->effectiveTly,hpoj->effectiveBry,
			hpoj->bryRange.min,hpoj->bryRange.max);
		reload|=SANE_INFO_RELOAD_PARAMS;
	}
	if ((hpoj->currentLengthMeasurement!=LENGTH_MEASUREMENT_UNLIMITED)!=
	     ((hpoj->option[OPTION_BR_Y].cap&SANE_CAP_INACTIVE)==0)) {
		if (hpoj->currentLengthMeasurement==
		     LENGTH_MEASUREMENT_UNLIMITED) {
			hpoj->option[OPTION_BR_Y].cap|=SANE_CAP_INACTIVE;
		} else {
			hpoj->option[OPTION_BR_Y].cap&=~SANE_CAP_INACTIVE;
		}
		reload|=SANE_INFO_RELOAD_OPTIONS;
	}

	/* Pre-scan parameters: */
	if (reload&SANE_INFO_RELOAD_PARAMS) {
		switch (hpoj->currentScanMode) {
		   case SCAN_MODE_LINEART:
			hpoj->prescanParameters.format=SANE_FRAME_GRAY;
			hpoj->prescanParameters.depth=1;
			break;

		   case SCAN_MODE_GRAYSCALE:
			hpoj->prescanParameters.format=SANE_FRAME_GRAY;
			hpoj->prescanParameters.depth=8;
			break;

		   case SCAN_MODE_COLOR:
		   default:
			hpoj->prescanParameters.format=SANE_FRAME_RGB;
			hpoj->prescanParameters.depth=8;
			break;
		}
		hpoj->prescanParameters.last_frame=SANE_TRUE;

		hpoj->prescanParameters.lines=MILLIMETERS_TO_PIXELS(
			hpoj->effectiveBry-hpoj->effectiveTly,
			hpoj->currentResolution);
		hpoj->prescanParameters.pixels_per_line=MILLIMETERS_TO_PIXELS(
			hpoj->effectiveBrx-hpoj->effectiveTlx,
			hpoj->currentResolution);

		hpoj->prescanParameters.bytes_per_line=BYTES_PER_LINE(
			hpoj->prescanParameters.pixels_per_line,
			hpoj->prescanParameters.depth*
			 (hpoj->prescanParameters.format==SANE_FRAME_RGB?3:1));
	}

	return reload;
}

extern SANE_Status sane_hpoj_open(SANE_String_Const devicename,
    SANE_Handle *pHandle) {
	SANE_Status retcode=SANE_STATUS_INVAL;
	ptalDevice_t dev;
	hpojScanner_t hpoj=0;
	int r;
	char deviceIDString[LEN_DEVICE_ID_STRING];
	char *strVendor; int lenVendor;
	char *strModel; int lenModel;
	int forceJpegForGrayAndColor=0;
	int force300dpiForLineart=0;
	int force300dpiForGrayscale=0;
	int supportsMfpdtf=1;

	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_open\n",devicename);

	/* Create or lookup the device.  If we have already created it,
	 * then we're done. */
	dev=ptalDeviceOpen((char *)devicename);
	if (!dev) {
		retcode=SANE_STATUS_INVAL;
		goto abort;
	}
	hpoj=(hpojScanner_t)ptalDeviceGetAppInfo(dev);
	if (hpoj) {
		goto done;
	}
	hpoj=malloc(sizeof(struct hpojScanner_s));
	if (pHandle) *pHandle=hpoj;
	if (!hpoj) {
		retcode=SANE_STATUS_NO_MEM;
		goto abort;
	}
	memset(hpoj,0,sizeof(struct hpojScanner_s));

	/* Initialize (but don't yet open) the scan channel. */
	hpoj->dev=dev;
	hpoj->chan=ptalChannelFindOrAllocate(dev,PTAL_STYPE_SCAN,0,0);
	/* The OfficeJet T series needs a minimum of about 8K for reverse. */
	ptalChannelSetPacketSizes(hpoj->chan,256,8192);
	hpoj->chanReserve=ptalChannelFindOrAllocate(dev,PTAL_STYPE_GENERIC,0,
		"PTAL-MLCD-RESERVE-SCANNER");
	ptalChannelSetErrorHandling(hpoj->chanReserve,0,0);

	/* Get the device ID string and initialize the SANE_Device structure. */
	memset(deviceIDString,0,LEN_DEVICE_ID_STRING);
	ptalDeviceGetDeviceIDString(dev,deviceIDString,LEN_DEVICE_ID_STRING);
	PTAL_LOG_DEBUG("hpoj:%s: device ID string=<%s>\n",
		devicename,deviceIDString);
	hpoj->saneDevice.name=ptalDeviceGetName(dev);
	r=ptalDeviceIDGetManufacturer(deviceIDString,&strVendor,&lenVendor);
	if (r==PTAL_ERROR) {
		strVendor="(Hewlett-Packard)";
		lenVendor=strlen(strVendor);
	} else {
		ptalDeviceIDPruneField(&strVendor,&lenVendor);
	}
	hpoj->saneDevice.vendor=malloc(lenVendor+1);
	memcpy((char *)hpoj->saneDevice.vendor,strVendor,lenVendor);
	((char *)hpoj->saneDevice.vendor)[lenVendor]=0;
	r=ptalDeviceIDGetModel(deviceIDString,&strModel,&lenModel);
	if (r!=PTAL_ERROR) {
		ptalDeviceIDPruneField(&strModel,&lenModel);
		_SET_DEFAULT_MODEL(hpoj,strModel,lenModel);
	}
	hpoj->saneDevice.type="multi-function peripheral";

	/* Set up as much as we can of the option descriptors... */

	hpoj->option[OPTION_NUM_OPTIONS].name=SANE_NAME_NUM_OPTIONS;
	hpoj->option[OPTION_NUM_OPTIONS].title=SANE_TITLE_NUM_OPTIONS;
	hpoj->option[OPTION_NUM_OPTIONS].desc=SANE_DESC_NUM_OPTIONS;
	hpoj->option[OPTION_NUM_OPTIONS].type=SANE_TYPE_INT;
	hpoj->option[OPTION_NUM_OPTIONS].unit=SANE_UNIT_NONE;
	hpoj->option[OPTION_NUM_OPTIONS].size=sizeof(SANE_Int);
	hpoj->option[OPTION_NUM_OPTIONS].cap=SANE_CAP_SOFT_DETECT;
	hpoj->option[OPTION_NUM_OPTIONS].constraint_type=SANE_CONSTRAINT_NONE;

	hpoj->option[GROUP_SCAN_MODE].title="Scan mode";
	hpoj->option[GROUP_SCAN_MODE].type=SANE_TYPE_GROUP;

	hpoj->option[OPTION_SCAN_MODE].name=SANE_NAME_SCAN_MODE;
	hpoj->option[OPTION_SCAN_MODE].title=SANE_TITLE_SCAN_MODE;
	hpoj->option[OPTION_SCAN_MODE].desc=SANE_DESC_SCAN_MODE;
	hpoj->option[OPTION_SCAN_MODE].type=SANE_TYPE_STRING;
	hpoj->option[OPTION_SCAN_MODE].unit=SANE_UNIT_NONE;
	hpoj->option[OPTION_SCAN_MODE].size=LEN_STRING_OPTION_VALUE;
	hpoj->option[OPTION_SCAN_MODE].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT;
	hpoj->option[OPTION_SCAN_MODE].constraint_type=SANE_CONSTRAINT_STRING_LIST;
	hpoj->option[OPTION_SCAN_MODE].constraint.string_list=hpoj->scanModeList;

	hpoj->option[OPTION_SCAN_RESOLUTION].name=SANE_NAME_SCAN_RESOLUTION;
	hpoj->option[OPTION_SCAN_RESOLUTION].title=SANE_TITLE_SCAN_RESOLUTION;
	hpoj->option[OPTION_SCAN_RESOLUTION].desc=SANE_DESC_SCAN_RESOLUTION;
	hpoj->option[OPTION_SCAN_RESOLUTION].type=SANE_TYPE_INT;
	hpoj->option[OPTION_SCAN_RESOLUTION].unit=SANE_UNIT_DPI;
	hpoj->option[OPTION_SCAN_RESOLUTION].size=sizeof(SANE_Int);
	hpoj->option[OPTION_SCAN_RESOLUTION].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT;
	hpoj->option[OPTION_SCAN_RESOLUTION].constraint_type=SANE_CONSTRAINT_RANGE;
	hpoj->option[OPTION_SCAN_RESOLUTION].constraint.range=&hpoj->resolutionRange;
	hpoj->resolutionRange.quant=0;

	hpoj->option[GROUP_ADVANCED].title="Advanced";
	hpoj->option[GROUP_ADVANCED].type=SANE_TYPE_GROUP;
	hpoj->option[GROUP_ADVANCED].cap=SANE_CAP_ADVANCED;

	hpoj->option[OPTION_CONTRAST].name=SANE_NAME_CONTRAST;
	hpoj->option[OPTION_CONTRAST].title=SANE_TITLE_CONTRAST;
	hpoj->option[OPTION_CONTRAST].desc=SANE_DESC_CONTRAST;
	hpoj->option[OPTION_CONTRAST].type=SANE_TYPE_INT;
	hpoj->option[OPTION_CONTRAST].unit=SANE_UNIT_NONE;
	hpoj->option[OPTION_CONTRAST].size=sizeof(SANE_Int);
	hpoj->option[OPTION_CONTRAST].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT|SANE_CAP_ADVANCED|SANE_CAP_INACTIVE;
	hpoj->option[OPTION_CONTRAST].constraint_type=SANE_CONSTRAINT_RANGE;
	hpoj->option[OPTION_CONTRAST].constraint.range=&hpoj->contrastRange;
	hpoj->contrastRange.min=PML_CONTRAST_MIN;
	hpoj->contrastRange.max=PML_CONTRAST_MAX;
	hpoj->contrastRange.quant=0;
	hpoj->defaultContrast=PML_CONTRAST_DEFAULT;

	hpoj->option[OPTION_COMPRESSION].name="compression";
	hpoj->option[OPTION_COMPRESSION].title="Compression";
	hpoj->option[OPTION_COMPRESSION].desc=
		"Selects the scanner compression method for faster scans, "
		"possibly at the expense of image quality.";
	hpoj->option[OPTION_COMPRESSION].type=SANE_TYPE_STRING;
	hpoj->option[OPTION_COMPRESSION].unit=SANE_UNIT_NONE;
	hpoj->option[OPTION_COMPRESSION].size=LEN_STRING_OPTION_VALUE;
	hpoj->option[OPTION_COMPRESSION].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT|SANE_CAP_ADVANCED;
	hpoj->option[OPTION_COMPRESSION].constraint_type=SANE_CONSTRAINT_STRING_LIST;
	hpoj->option[OPTION_COMPRESSION].constraint.string_list=hpoj->compressionList;

	hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].name="jpeg-compression-factor";
	hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].title="JPEG compression factor";
	hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].desc=
		"Sets the scanner JPEG compression factor.  "
		"Larger numbers mean better compression, and "
		"smaller numbers mean better image quality.";
	hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].type=SANE_TYPE_INT;
	hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].unit=SANE_UNIT_NONE;
	hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].size=sizeof(SANE_Int);
	hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT|SANE_CAP_ADVANCED;
	hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].constraint_type=SANE_CONSTRAINT_RANGE;
	hpoj->option[OPTION_JPEG_COMPRESSION_FACTOR].constraint.range=&hpoj->jpegCompressionFactorRange;
	hpoj->jpegCompressionFactorRange.min=MIN_JPEG_COMPRESSION_FACTOR;
	hpoj->jpegCompressionFactorRange.max=MAX_JPEG_COMPRESSION_FACTOR;
	hpoj->jpegCompressionFactorRange.quant=0;
	hpoj->defaultJpegCompressionFactor=MIN_JPEG_COMPRESSION_FACTOR;

	hpoj->option[OPTION_BATCH_SCAN].name="batch-scan";
	hpoj->option[OPTION_BATCH_SCAN].title="Batch scan";
	hpoj->option[OPTION_BATCH_SCAN].desc=
		"Guarantees that a \"no documents\" condition will be "
		"returned after the last scanned page, to prevent "
		"endless flatbed scans after a batch scan.  "
		"For some models, option changes in the middle of a batch "
		"scan don't take effect until after the last page.";
	hpoj->option[OPTION_BATCH_SCAN].type=SANE_TYPE_BOOL;
	hpoj->option[OPTION_BATCH_SCAN].unit=SANE_UNIT_NONE;
	hpoj->option[OPTION_BATCH_SCAN].size=sizeof(SANE_Bool);
	hpoj->option[OPTION_BATCH_SCAN].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT|SANE_CAP_ADVANCED;
	hpoj->option[OPTION_BATCH_SCAN].constraint_type=SANE_CONSTRAINT_NONE;

	hpoj->option[OPTION_ADF_MODE].name="source";	// xsane expects this.
	hpoj->option[OPTION_ADF_MODE].title="Source";
	hpoj->option[OPTION_ADF_MODE].desc=
		"Selects the desired scan source for models with both "
		"flatbed and automatic document feeder (ADF) capabilities.  "
		"The \"Auto\" setting means that the ADF will be used "
		"if it's loaded, and the flatbed (if present) will be "
		"used otherwise.";
	hpoj->option[OPTION_ADF_MODE].type=SANE_TYPE_STRING;
	hpoj->option[OPTION_ADF_MODE].unit=SANE_UNIT_NONE;
	hpoj->option[OPTION_ADF_MODE].size=LEN_STRING_OPTION_VALUE;
	hpoj->option[OPTION_ADF_MODE].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT|SANE_CAP_ADVANCED;
	hpoj->option[OPTION_ADF_MODE].constraint_type=SANE_CONSTRAINT_STRING_LIST;
	hpoj->option[OPTION_ADF_MODE].constraint.string_list=hpoj->adfModeList;

	hpoj->option[OPTION_DUPLEX].name="duplex";
	hpoj->option[OPTION_DUPLEX].title="Duplex";
	hpoj->option[OPTION_DUPLEX].desc=
		"Enables scanning on both sides of the page for models "
		"with duplex-capable document feeders.  For pages printed "
		"in \"book\"-style duplex mode, one side will be scanned "
		"upside-down.  This feature is experimental.";
	hpoj->option[OPTION_DUPLEX].type=SANE_TYPE_BOOL;
	hpoj->option[OPTION_DUPLEX].unit=SANE_UNIT_NONE;
	hpoj->option[OPTION_DUPLEX].size=sizeof(SANE_Bool);
	hpoj->option[OPTION_DUPLEX].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT|SANE_CAP_ADVANCED;
	hpoj->option[OPTION_DUPLEX].constraint_type=SANE_CONSTRAINT_NONE;

	hpoj->option[GROUP_GEOMETRY].title="Geometry";
	hpoj->option[GROUP_GEOMETRY].type=SANE_TYPE_GROUP;
	hpoj->option[GROUP_GEOMETRY].cap=SANE_CAP_ADVANCED;

	hpoj->option[OPTION_LENGTH_MEASUREMENT].name="length-measurement";
	hpoj->option[OPTION_LENGTH_MEASUREMENT].title="Length measurement";
	hpoj->option[OPTION_LENGTH_MEASUREMENT].desc=
		"Selects how the scanned image length is measured and "
		"reported, which is impossible to know in advance for "
		"scrollfed scans.";
	hpoj->option[OPTION_LENGTH_MEASUREMENT].type=SANE_TYPE_STRING;
	hpoj->option[OPTION_LENGTH_MEASUREMENT].unit=SANE_UNIT_NONE;
	hpoj->option[OPTION_LENGTH_MEASUREMENT].size=LEN_STRING_OPTION_VALUE;
	hpoj->option[OPTION_LENGTH_MEASUREMENT].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT|SANE_CAP_ADVANCED;
	hpoj->option[OPTION_LENGTH_MEASUREMENT].constraint_type=SANE_CONSTRAINT_STRING_LIST;
	hpoj->option[OPTION_LENGTH_MEASUREMENT].constraint.string_list=hpoj->lengthMeasurementList;

	hpoj->option[OPTION_TL_X].name=SANE_NAME_SCAN_TL_X;
	hpoj->option[OPTION_TL_X].title=SANE_TITLE_SCAN_TL_X;
	hpoj->option[OPTION_TL_X].desc=SANE_DESC_SCAN_TL_X;
	hpoj->option[OPTION_TL_X].type=GEOMETRY_OPTION_TYPE;
	hpoj->option[OPTION_TL_X].unit=SANE_UNIT_MM;
	hpoj->option[OPTION_TL_X].size=sizeof(SANE_Int);
	hpoj->option[OPTION_TL_X].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT;
	hpoj->option[OPTION_TL_X].constraint_type=SANE_CONSTRAINT_RANGE;
	hpoj->option[OPTION_TL_X].constraint.range=&hpoj->tlxRange;
	hpoj->tlxRange.min=0;
	hpoj->tlxRange.quant=0;

	hpoj->option[OPTION_TL_Y].name=SANE_NAME_SCAN_TL_Y;
	hpoj->option[OPTION_TL_Y].title=SANE_TITLE_SCAN_TL_Y;
	hpoj->option[OPTION_TL_Y].desc=SANE_DESC_SCAN_TL_Y;
	hpoj->option[OPTION_TL_Y].type=GEOMETRY_OPTION_TYPE;
	hpoj->option[OPTION_TL_Y].unit=SANE_UNIT_MM;
	hpoj->option[OPTION_TL_Y].size=sizeof(SANE_Int);
	hpoj->option[OPTION_TL_Y].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT;
	hpoj->option[OPTION_TL_Y].constraint_type=SANE_CONSTRAINT_RANGE;
	hpoj->option[OPTION_TL_Y].constraint.range=&hpoj->tlyRange;
	hpoj->tlyRange.min=0;
	hpoj->tlyRange.quant=0;

	hpoj->option[OPTION_BR_X].name=SANE_NAME_SCAN_BR_X;
	hpoj->option[OPTION_BR_X].title=SANE_TITLE_SCAN_BR_X;
	hpoj->option[OPTION_BR_X].desc=SANE_DESC_SCAN_BR_X;
	hpoj->option[OPTION_BR_X].type=GEOMETRY_OPTION_TYPE;
	hpoj->option[OPTION_BR_X].unit=SANE_UNIT_MM;
	hpoj->option[OPTION_BR_X].size=sizeof(SANE_Int);
	hpoj->option[OPTION_BR_X].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT;
	hpoj->option[OPTION_BR_X].constraint_type=SANE_CONSTRAINT_RANGE;
	hpoj->option[OPTION_BR_X].constraint.range=&hpoj->brxRange;
	hpoj->brxRange.min=0;
	hpoj->brxRange.quant=0;

	hpoj->option[OPTION_BR_Y].name=SANE_NAME_SCAN_BR_Y;
	hpoj->option[OPTION_BR_Y].title=SANE_TITLE_SCAN_BR_Y;
	hpoj->option[OPTION_BR_Y].desc=SANE_DESC_SCAN_BR_Y;
	hpoj->option[OPTION_BR_Y].type=GEOMETRY_OPTION_TYPE;
	hpoj->option[OPTION_BR_Y].unit=SANE_UNIT_MM;
	hpoj->option[OPTION_BR_Y].size=sizeof(SANE_Int);
	hpoj->option[OPTION_BR_Y].cap=SANE_CAP_SOFT_SELECT|SANE_CAP_SOFT_DETECT;
	hpoj->option[OPTION_BR_Y].constraint_type=SANE_CONSTRAINT_RANGE;
	hpoj->option[OPTION_BR_Y].constraint.range=&hpoj->bryRange;
	hpoj->bryRange.min=0;
	hpoj->bryRange.quant=0;

#if 0	/* Use these 10 lines as a template for adding options: */
	hpoj->option[$].name=?;
	hpoj->option[$].title=?;
	hpoj->option[$].desc=?;
	hpoj->option[$].type=?;
	hpoj->option[$].unit=?;
	hpoj->option[$].size=?;
	hpoj->option[$].cap=?;
	hpoj->option[$].constraint_type=?;
	hpoj->option[$].constraint.?=?;

#endif	/* End of option-initialization template. */

	/* Guess the command language (SCL or PML) based on the model string. */
	if (UNDEFINED_MODEL(hpoj)) {
		hpoj->scannerType=SCANNER_TYPE_SCL;

	} else if (strstr(hpoj->saneDevice.model,"LaserJet") ||
	           strstr(hpoj->saneDevice.model,"Laserjet") ||
	           strstr(hpoj->saneDevice.model,"LASERJET") ||
	           strstr(hpoj->saneDevice.model,"laserjet")) {
		hpoj->scannerType=SCANNER_TYPE_PML;
		hpoj->pml.openFirst=1;
		if (strstr(hpoj->saneDevice.model,"HP LaserJet 1100")) {
			hpoj->pml.dontResetBeforeNextNonBatchPage=1;
		} else {
			hpoj->pml.startNextBatchPageEarly=1;
		}

	} else if (!strcmp(hpoj->saneDevice.model,"OfficeJet") ||
		   !strcmp(hpoj->saneDevice.model,"OfficeJet LX") ||
	           !strcmp(hpoj->saneDevice.model,"OfficeJet Series 300")) {
		hpoj->scannerType=SCANNER_TYPE_PML;
		hpoj->preDenali=1;

	} else if (!strcmp(hpoj->saneDevice.model,"OfficeJet Series 500") ||
	           !strcmp(hpoj->saneDevice.model,"All-in-One IJP-V100")) {
		hpoj->scannerType=SCANNER_TYPE_PML;
		hpoj->fromDenali=1;
		force300dpiForLineart=1;
		force300dpiForGrayscale=1;
		hpoj->defaultCompression[SCAN_MODE_LINEART]=COMPRESSION_MH;
		hpoj->defaultCompression[SCAN_MODE_GRAYSCALE]=COMPRESSION_JPEG;
		hpoj->defaultJpegCompressionFactor=
			SAFER_JPEG_COMPRESSION_FACTOR;

	} else if (!strcmp(hpoj->saneDevice.model,"OfficeJet Series 600")) {
		hpoj->scannerType=SCANNER_TYPE_PML;
		forceJpegForGrayAndColor=1;
		force300dpiForLineart=1;
		hpoj->defaultCompression[SCAN_MODE_LINEART]=COMPRESSION_MH;
		hpoj->defaultJpegCompressionFactor=
			SAFER_JPEG_COMPRESSION_FACTOR;

	} else if (!strcmp(hpoj->saneDevice.model,"Printer/Scanner/Copier 300")) {
		hpoj->scannerType=SCANNER_TYPE_PML;
		forceJpegForGrayAndColor=1;
		force300dpiForLineart=1;
		hpoj->defaultCompression[SCAN_MODE_LINEART]=COMPRESSION_MH;
		hpoj->defaultJpegCompressionFactor=
			SAFER_JPEG_COMPRESSION_FACTOR;

	} else if (!strcmp(hpoj->saneDevice.model,"OfficeJet Series 700")) {
		hpoj->scannerType=SCANNER_TYPE_PML;
		forceJpegForGrayAndColor=1;
		force300dpiForLineart=1;
		hpoj->defaultCompression[SCAN_MODE_LINEART]=COMPRESSION_MH;
		hpoj->defaultJpegCompressionFactor=
			SAFER_JPEG_COMPRESSION_FACTOR;

	} else if (!strcmp(hpoj->saneDevice.model,"OfficeJet T Series")) {
		hpoj->scannerType=SCANNER_TYPE_PML;
		forceJpegForGrayAndColor=1;

	} else {
		hpoj->scannerType=SCANNER_TYPE_SCL;
	}

	/* Open and reset scanner command channel.
	 * This also allocates the PML objects if necessary. */
	retcode=hpojConnOpen(hpoj);
	if (retcode!=SANE_STATUS_GOOD) {
		goto abort;
	}

    /* Probing and setup for SCL scanners... */
    if (hpoj->scannerType==SCANNER_TYPE_SCL) {
	/* Even though this isn't SCANNER_TYPE_PML, there are still a few
	 * interesting PML objects. */
	hpoj->scl.objSupportedFunctions=
		ptalPmlAllocateID(hpoj->dev,"\x1\x1\x2\x43");
	ptalPmlOpen(hpoj->dev);

	/* Probe the SCL model. */
	PTAL_LOG_DEBUG("hpoj:%s: Using SCL protocol.\n",hpoj->saneDevice.name);

	retcode=hpojSclInquire(hpoj,
		SCL_CMD_INQUIRE_DEVICE_PARAMETER,SCL_INQ_HP_MODEL_11,
		0,hpoj->scl.compat1150,LEN_MODEL_RESPONSE);
	if (retcode==SANE_STATUS_GOOD) {
		hpoj->scl.compat|=SCL_COMPAT_OFFICEJET;
	} else if (retcode!=SANE_STATUS_UNSUPPORTED) {
		goto abort;
	}
	PTAL_LOG_DEBUG("hpoj:%s: scl.compat1150=<%s>.\n",
		hpoj->saneDevice.name,hpoj->scl.compat1150);

	retcode=hpojSclInquire(hpoj,
		SCL_CMD_INQUIRE_DEVICE_PARAMETER,SCL_INQ_HP_MODEL_12,
		0,hpoj->scl.compatPost1150,LEN_MODEL_RESPONSE);
	if (retcode==SANE_STATUS_GOOD) {
		hpoj->scl.compat|=SCL_COMPAT_POST_1150;
	} else if (retcode!=SANE_STATUS_UNSUPPORTED) {
		goto abort;
	}
	PTAL_LOG_DEBUG("hpoj:%s: scl.compatPost1150=<%s>.\n",
		hpoj->saneDevice.name,hpoj->scl.compatPost1150);

	if (!hpoj->scl.compat) {
		SET_DEFAULT_MODEL(hpoj,"(unknown scanner)");

	} else if (hpoj->scl.compat==SCL_COMPAT_OFFICEJET) {
		hpoj->scl.compat|=SCL_COMPAT_1150;
		SET_DEFAULT_MODEL(hpoj,"(OfficeJet 1150)");

	} else if (!strcmp(hpoj->scl.compatPost1150,"5400A")) {
		hpoj->scl.compat|=SCL_COMPAT_1170;
		SET_DEFAULT_MODEL(hpoj,"(OfficeJet 1170)");

	} else if (!strcmp(hpoj->scl.compatPost1150,"5500A")) {
		hpoj->scl.compat|=SCL_COMPAT_R_SERIES;
		SET_DEFAULT_MODEL(hpoj,"(OfficeJet R Series)");

	} else if (!strcmp(hpoj->scl.compatPost1150,"5600A")) {
		hpoj->scl.compat|=SCL_COMPAT_G_SERIES;
		SET_DEFAULT_MODEL(hpoj,"(OfficeJet G Series)");

	} else if (!strcmp(hpoj->scl.compatPost1150,"5700A")) {
		hpoj->scl.compat|=SCL_COMPAT_K_SERIES;
		SET_DEFAULT_MODEL(hpoj,"(OfficeJet K Series)");

	} else if (!strcmp(hpoj->scl.compatPost1150,"5800A")) {
		hpoj->scl.compat|=SCL_COMPAT_D_SERIES;
		SET_DEFAULT_MODEL(hpoj,"(OfficeJet D Series)");

	} else if (!strcmp(hpoj->scl.compatPost1150,"5900A")) {
		hpoj->scl.compat|=SCL_COMPAT_6100_SERIES;
		SET_DEFAULT_MODEL(hpoj,"(OfficeJet 6100 Series)");

	} else {
		SET_DEFAULT_MODEL(hpoj,"(unknown OfficeJet)");
	}
	PTAL_LOG_DEBUG("hpoj:%s: scl.compat=0x%4.4X.\n",
		hpoj->saneDevice.name,hpoj->scl.compat);

	/* Decide which position/extent unit to use.  "Device pixels" works
	 * better on most models, but the 1150 requires "decipoints." */
	if (hpoj->scl.compat&(SCL_COMPAT_1150)) {
		hpoj->scl.decipixelChar=SCL_CHAR_DECIPOINTS;
		hpoj->decipixelsPerInch=DECIPOINTS_PER_INCH;
	} else {
		hpoj->scl.decipixelChar=SCL_CHAR_DEVPIXELS;
		hpoj->decipixelsPerInch=DEVPIXELS_PER_INCH;
		/* Check for non-default decipixelsPerInch definition. */
		hpojSclInquire(hpoj,SCL_CMD_INQUIRE_DEVICE_PARAMETER,
			SCL_INQ_DEVICE_PIXELS_PER_INCH,
			&hpoj->decipixelsPerInch,0,0);
	}
	PTAL_LOG_DEBUG("hpoj:%s: decipixelChar='%c', decipixelsPerInch=%d.\n",
		hpoj->saneDevice.name,hpoj->scl.decipixelChar,
		hpoj->decipixelsPerInch);

	/* Is MFPDTF supported? */
	if (hpojSclSendCommandCheckError(hpoj,SCL_CMD_SET_MFPDTF,
	     SCL_MFPDTF_ON)!=SANE_STATUS_GOOD) {
		PTAL_LOG_DEBUG("hpoj:%s: Doesn't support MFPDTF.\n",
			hpoj->saneDevice.name);
		supportsMfpdtf=0;
	}

	/* All scan modes are supported with no compression. */
	hpoj->supportsScanMode[SCAN_MODE_LINEART]=COMPRESSION_NONE;
	hpoj->supportsScanMode[SCAN_MODE_GRAYSCALE]=COMPRESSION_NONE;
	hpoj->supportsScanMode[SCAN_MODE_COLOR]=COMPRESSION_NONE;

	/* Is JPEG also supported for grayscale and color (requires MFPDTF)? */
	if (supportsMfpdtf) {
		if (hpojSclSendCommandCheckError(hpoj,SCL_CMD_SET_COMPRESSION,
		     SCL_COMPRESSION_JPEG)==SANE_STATUS_GOOD) {
			hpoj->supportsScanMode[SCAN_MODE_GRAYSCALE]|=
				COMPRESSION_JPEG;
			hpoj->supportsScanMode[SCAN_MODE_COLOR]|=
				COMPRESSION_JPEG;
		}
	}

	/* Determine the minimum and maximum resolution.
	 * Probe for both X and Y, and pick largest min and smallest max.
	 * For the 1150, set min to 50 to prevent scan head crashes (<42). */
	hpojSclInquire(hpoj,SCL_CMD_INQUIRE_MINIMUM_VALUE,
		SCL_CMD_SET_X_RESOLUTION,&hpoj->scl.minXRes,0,0);
	hpojSclInquire(hpoj,SCL_CMD_INQUIRE_MINIMUM_VALUE,
		SCL_CMD_SET_Y_RESOLUTION,&hpoj->scl.minYRes,0,0);
	if (hpoj->scl.compat&SCL_COMPAT_1150 &&
	    hpoj->scl.minYRes<SCL_MIN_Y_RES_1150) {
		hpoj->scl.minYRes=SCL_MIN_Y_RES_1150;
	}
	r=hpoj->scl.minXRes;
	if (r<hpoj->scl.minYRes) {
		r=hpoj->scl.minYRes;
	}
	hpoj->resolutionRange.min=r;
	hpojSclInquire(hpoj,SCL_CMD_INQUIRE_MAXIMUM_VALUE,
		SCL_CMD_SET_X_RESOLUTION,&hpoj->scl.maxXRes,0,0);
	hpojSclInquire(hpoj,SCL_CMD_INQUIRE_MAXIMUM_VALUE,
		SCL_CMD_SET_Y_RESOLUTION,&hpoj->scl.maxYRes,0,0);
	r=hpoj->scl.maxXRes;
	if (r>hpoj->scl.maxYRes) {
		r=hpoj->scl.maxYRes;
	}
	if (hpoj->scl.compat&(SCL_COMPAT_1150|SCL_COMPAT_1170) &&
	    r>SCL_MAX_RES_1150_1170) {
		r=SCL_MAX_RES_1150_1170;
	}
	hpoj->resolutionRange.max=r;

	/* Determine ADF/duplex capabilities. */
    {
	int flatbedCapability=1;
	hpojSclInquire(hpoj,SCL_CMD_INQUIRE_MAXIMUM_VALUE,
		SCL_PSEUDO_FLATBED_Y_RESOLUTION,&flatbedCapability,0,0);
	hpojSclInquire(hpoj,SCL_CMD_INQUIRE_DEVICE_PARAMETER,
		SCL_INQ_ADF_CAPABILITY,&hpoj->scl.adfCapability,0,0);
	PTAL_LOG_DEBUG("hpoj:%s: ADF capability=%d.\n",
		hpoj->saneDevice.name,hpoj->scl.adfCapability);
	if (!hpoj->scl.adfCapability) {
		hpoj->supportedAdfModes=ADF_MODE_FLATBED;
	} else if (hpoj->scl.compat&SCL_COMPAT_K_SERIES ||
	           !flatbedCapability) {
		hpoj->supportedAdfModes=ADF_MODE_ADF;
	} else {
		int supportedFunctions;

		hpoj->supportedAdfModes=
			ADF_MODE_AUTO|ADF_MODE_FLATBED|ADF_MODE_ADF;
		if (hpoj->scl.compat&(SCL_COMPAT_1170|
		     SCL_COMPAT_R_SERIES|SCL_COMPAT_G_SERIES)) {
			hpoj->scl.unloadAfterScan=1;
		}
		if (ptalPmlRequestGet(hpoj->scl.objSupportedFunctions,0)!=
		     PTAL_ERROR &&
		    ptalPmlGetIntegerValue(hpoj->scl.objSupportedFunctions,0,
		      &supportedFunctions)!=PTAL_ERROR &&
		    supportedFunctions&PML_SUPPFUNC_DUPLEX) {
			hpoj->supportsDuplex=1;
		}
	}
    }

	/* Determine maximum X and Y extents. */
	hpojSclInquire(hpoj,SCL_CMD_INQUIRE_MAXIMUM_VALUE,
		SCL_CMD_SET_X_EXTENT,&hpoj->scl.maxXExtent,0,0);
	hpojSclInquire(hpoj,SCL_CMD_INQUIRE_MAXIMUM_VALUE,
		SCL_CMD_SET_Y_EXTENT,&hpoj->scl.maxYExtent,0,0);
	PTAL_LOG_DEBUG("hpoj:%s: Maximum extents: x=%d, y=%d.\n",
		hpoj->saneDevice.name,
		hpoj->scl.maxXExtent,hpoj->scl.maxYExtent);
	hpoj->tlxRange.max=hpoj->brxRange.max=
		DECIPIXELS_TO_MILLIMETERS(hpoj->scl.maxXExtent);
	hpoj->tlyRange.max=hpoj->bryRange.max=
		DECIPIXELS_TO_MILLIMETERS(hpoj->scl.maxYExtent);

    /* Probing and setup for PML scanners... */
    } else /* if (hpoj->scannerType==SCANNER_TYPE_PML) */ {
	int comps=0;
	static struct timeval selectPollTimeout={PML_SELECT_POLL_TIMEOUT,0};
	PTAL_LOG_DEBUG("hpoj:%s: Using PML protocol.\n",hpoj->saneDevice.name);
	hpoj->decipixelsPerInch=DECIPOINTS_PER_INCH;

	ptalChannelSetSelectPollTimeout(hpoj->chan,&selectPollTimeout);
	ptalChannelSetSelectPollCallback(hpoj->chan,
		hpojPmlSelectCallback,hpoj);

	/* Determine supported scan modes and compression settings. */
	if (hpoj->preDenali) {
		comps|=COMPRESSION_MMR;
	}
	ptalPmlSetIntegerValue(hpoj->pml.objCompression,
		PTAL_PML_TYPE_ENUMERATION,PML_COMPRESSION_NONE);
	if (ptalPmlRequestSetRetry(hpoj->pml.objCompression,0,0)!=PTAL_ERROR) {
		comps|=COMPRESSION_NONE;
	}
	ptalPmlSetIntegerValue(hpoj->pml.objCompression,
		PTAL_PML_TYPE_ENUMERATION,PML_COMPRESSION_MH);
	if (ptalPmlRequestSetRetry(hpoj->pml.objCompression,0,0)!=PTAL_ERROR) {
		comps|=COMPRESSION_MH;
	}
	ptalPmlSetIntegerValue(hpoj->pml.objCompression,
		PTAL_PML_TYPE_ENUMERATION,PML_COMPRESSION_MR);
	if (ptalPmlRequestSetRetry(hpoj->pml.objCompression,0,0)!=PTAL_ERROR) {
		comps|=COMPRESSION_MR;
	}
	ptalPmlSetIntegerValue(hpoj->pml.objCompression,
		PTAL_PML_TYPE_ENUMERATION,PML_COMPRESSION_MMR);
	if (ptalPmlRequestSetRetry(hpoj->pml.objCompression,0,0)!=PTAL_ERROR) {
		comps|=COMPRESSION_MMR;
	}
	ptalPmlSetIntegerValue(hpoj->pml.objPixelDataType,
		PTAL_PML_TYPE_ENUMERATION,PML_DATA_TYPE_LINEART);
	if (ptalPmlRequestSetRetry(hpoj->pml.objPixelDataType,0,0)!=
	     PTAL_ERROR) {
		hpoj->supportsScanMode[SCAN_MODE_LINEART]=comps;
	}
	comps&=COMPRESSION_NONE;
	if (forceJpegForGrayAndColor) comps=0;
	ptalPmlSetIntegerValue(hpoj->pml.objCompression,
		PTAL_PML_TYPE_ENUMERATION,PML_COMPRESSION_JPEG);
	if (ptalPmlRequestSetRetry(hpoj->pml.objCompression,0,0)!=PTAL_ERROR) {
		comps|=COMPRESSION_JPEG;
	}
	ptalPmlSetIntegerValue(hpoj->pml.objPixelDataType,
		PTAL_PML_TYPE_ENUMERATION,PML_DATA_TYPE_GRAYSCALE);
	if (ptalPmlRequestSetRetry(hpoj->pml.objPixelDataType,0,0)!=
	     PTAL_ERROR) {
		hpoj->supportsScanMode[SCAN_MODE_GRAYSCALE]=comps;
	}
	ptalPmlSetIntegerValue(hpoj->pml.objPixelDataType,
		PTAL_PML_TYPE_ENUMERATION,PML_DATA_TYPE_COLOR);
	if (ptalPmlRequestSetRetry(hpoj->pml.objPixelDataType,0,0)!=
	     PTAL_ERROR) {
		hpoj->supportsScanMode[SCAN_MODE_COLOR]=comps;
	}

	/* Determine supported resolutions. */
	hpojNumListClear(hpoj->resolutionList);
	hpojNumListClear(hpoj->lineartResolutionList);
	if (hpoj->preDenali) {
		hpojNumListAdd(hpoj->lineartResolutionList,200);
	        if (!strcmp(hpoj->saneDevice.model,"OfficeJet Series 300")) {
			hpojNumListAdd(hpoj->lineartResolutionList,300);
		}
		hpoj->option[OPTION_SCAN_RESOLUTION].constraint_type=
			SANE_CONSTRAINT_WORD_LIST;

	} else if (ptalPmlRequestGet(hpoj->pml.objResolutionRange,0)==
	     PTAL_ERROR) {
pmlDefaultResRange:
		/* TODO: What are the correct X and Y resolution ranges
		 * for the OfficeJet T series? */
		hpoj->resolutionRange.min=75;
		hpoj->resolutionRange.max=600;

	} else {
		char resList[PTAL_PML_MAX_VALUE_LEN+1];
		int i,len,res,consumed;

		ptalPmlGetStringValue(hpoj->pml.objResolutionRange,
			0,resList,PTAL_PML_MAX_VALUE_LEN);
		resList[PTAL_PML_MAX_VALUE_LEN]=0;
		len=strlen(resList);

		/* Parse "(100)x(100),(150)x(150),(200)x(200),(300)x(300)".
		 * This isn't quite the right way to do it, but it'll do. */
		for (i=0;i<len;) {
			if (resList[i]<'0' || resList[i]>'9') {
				i++;
				continue;
			}
			if (sscanf(resList+i,"%d%n",&res,&consumed)!=1) break;
			i+=consumed;
		    if (!force300dpiForGrayscale || res>=300) {
			hpojNumListAdd(hpoj->resolutionList,res);
		    }
		    if (!force300dpiForLineart || res>=300) {
			hpojNumListAdd(hpoj->lineartResolutionList,res);
		    }
		}

		if (!hpojNumListGetCount(hpoj->resolutionList)) {
			goto pmlDefaultResRange;
		}
		hpoj->option[OPTION_SCAN_RESOLUTION].constraint_type=
			SANE_CONSTRAINT_WORD_LIST;
	}

	/* Determine contrast support. */
	if (ptalPmlRequestGet(hpoj->pml.objContrast,0)!=PTAL_ERROR) {
		hpoj->option[OPTION_CONTRAST].cap&=~SANE_CAP_INACTIVE;
	}

	/* Determine supported ADF modes. */
	if (ptalPmlRequestGet(hpoj->pml.objModularHardware,0)!=PTAL_ERROR) {
		int modularHardware=0;
		hpoj->pml.flatbedCapability=1;
		if (ptalPmlGetIntegerValue(hpoj->pml.objModularHardware,0,
		     &modularHardware)!=PTAL_ERROR &&
		    modularHardware&PML_MODHW_ADF) {
			goto adfAndAuto;
		}
		hpoj->supportedAdfModes=ADF_MODE_FLATBED;
	} else {
adfAndAuto:
		hpoj->supportedAdfModes=ADF_MODE_AUTO|ADF_MODE_ADF;
	}
	hpoj->supportsDuplex=0;

	hpoj->tlxRange.max=hpoj->brxRange.max=
		INCHES_TO_MILLIMETERS(PML_MAX_WIDTH_INCHES);
	hpoj->tlyRange.max=hpoj->bryRange.max=
		INCHES_TO_MILLIMETERS(PML_MAX_HEIGHT_INCHES);
    }

	PTAL_LOG_DEBUG("hpoj:%s: forceJpegForGrayAndColor=%d.\n",
		hpoj->saneDevice.name,forceJpegForGrayAndColor);
	PTAL_LOG_DEBUG("hpoj:%s: force300dpiForLineart=%d.\n",
		hpoj->saneDevice.name,force300dpiForLineart);
	PTAL_LOG_DEBUG("hpoj:%s: force300dpiForGrayscale=%d.\n",
		hpoj->saneDevice.name,force300dpiForGrayscale);
	PTAL_LOG_DEBUG("hpoj:%s: pml.openFirst=%d.\n",
		hpoj->saneDevice.name,hpoj->pml.openFirst);
	PTAL_LOG_DEBUG("hpoj:%s: fromDenali=%d.\n",
		hpoj->saneDevice.name,hpoj->fromDenali);
	PTAL_LOG_DEBUG("hpoj:%s: supportsScanMode: lineart=0x%2.2X, "
		"grayscale=0x%2.2X, color=0x%2.2X.\n",
		hpoj->saneDevice.name,
		hpoj->supportsScanMode[SCAN_MODE_LINEART],
		hpoj->supportsScanMode[SCAN_MODE_GRAYSCALE],
		hpoj->supportsScanMode[SCAN_MODE_COLOR]);
	PTAL_LOG_DEBUG("hpoj:%s: supportedAdfModes=0x%2.2X.\n",
		hpoj->saneDevice.name,hpoj->supportedAdfModes);
	PTAL_LOG_DEBUG("hpoj:%s: supportsDuplex=%d.\n",
		hpoj->saneDevice.name,hpoj->supportsDuplex);

	/* Allocate MFPDTF parser if supported. */
	if (supportsMfpdtf) {
		hpoj->mfpdtf=ptalMfpdtfAllocate(hpoj->chan);
		if (hpoj->preDenali) {
			ptalMfpdtfReadSetSimulateImageHeaders(hpoj->mfpdtf,1);
		}
	}
	hpoj->fdSaveCompressedData=PTAL_ERROR;

done:
	/* Finish setting up option descriptors. */
	hpojUpdateDescriptors(hpoj,OPTION_FIRST);

	if (pHandle) *pHandle=hpoj;
	ptalDeviceSetAppInfo(dev,hpoj);
	retcode=SANE_STATUS_GOOD;
abort:
	if (hpoj) hpojConnClose(hpoj);
	if (retcode!=SANE_STATUS_GOOD) {
		if (hpoj) {
			if (hpoj->saneDevice.vendor)
				free((void *)hpoj->saneDevice.vendor);
			if (hpoj->saneDevice.model)
				free((void *)hpoj->saneDevice.model);
			free(hpoj);
		}
	}
	return retcode;
}

extern void sane_hpoj_close(SANE_Handle handle) {
	hpojScanner_t hpoj=(hpojScanner_t)handle;

	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_close\n",hpoj->saneDevice.name);

	/* TODO */
}

static void hpojDeviceListReset(void) {
	if (hpojDeviceList) {
		free(hpojDeviceList);
		hpojDeviceList=0;
	}
	hpojDeviceListCount=0;
	hpojDeviceList=malloc(sizeof(SANE_Device *));
	if (hpojDeviceList) {
		hpojDeviceList[0]=0;
	}
}

static void hpojDeviceListAdd(SANE_Device *pSaneDevice) {
	SANE_Device **dl=realloc(hpojDeviceList,
		sizeof(SANE_Device *)*(hpojDeviceListCount+2));
	if (dl) {
		hpojDeviceList=dl;
		hpojDeviceList[hpojDeviceListCount]=pSaneDevice;
		hpojDeviceListCount++;
		hpojDeviceList[hpojDeviceListCount]=0;
	}
}

static int hpojDeviceEnumerateCallback(ptalDevice_t dev,void *cbd) {
	hpojScanner_t hpoj;

	if (sane_hpoj_open(ptalDeviceGetName(dev),(SANE_Handle)(&hpoj))!=
	     SANE_STATUS_GOOD) {
		return 0;
	}

	hpojDeviceListAdd(&hpoj->saneDevice);
	return 1;
}

extern SANE_Status sane_hpoj_get_devices(const SANE_Device ***pDeviceList,
    SANE_Bool localOnly) {

	PTAL_LOG_DEBUG("hpoj: sane_hpoj_get_devices\n");

	hpojDeviceListReset();
	ptalDeviceEnumerate(0,hpojDeviceEnumerateCallback,0);

	*pDeviceList=(const SANE_Device **)hpojDeviceList;
	return SANE_STATUS_GOOD;
}

extern SANE_Status sane_hpoj_init(SANE_Int *pVersionCode,
    SANE_Auth_Callback authorize) {
	PTAL_LOG_DEBUG("hpoj: sane_hpoj_init\n");

	ptalInit();
	hpojDeviceListReset();

	if (pVersionCode) *pVersionCode=SANE_VERSION_CODE(1,0,6);

	return SANE_STATUS_GOOD;
}

extern void sane_hpoj_exit(void) {
	PTAL_LOG_DEBUG("hpoj: sane_hpoj_exit\n");

	/* TODO: Clean up to avoid memory leaks.  Maybe have a libptal
	 * callback when a device appInfo pointer is set to notify that
	 * the device is going away. */

	hpojDeviceListReset();

	ptalDone();
}

extern const SANE_Option_Descriptor *sane_hpoj_get_option_descriptor(
    SANE_Handle handle,SANE_Int option) {
	hpojScanner_t hpoj=(hpojScanner_t)handle;
	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_get_option_descriptor(option=%d)\n",
		hpoj->saneDevice.name,option);
	if (option<0 || option>=OPTION_LAST) return 0;
	PTAL_LOG_DEBUG("hpoj:%s: option=%d name=<%s>\n"
		"\ttype=%d unit=%d size=%d cap=0x%2.2X ctype=%d\n",
		hpoj->saneDevice.name,option,
		hpoj->option[option].name,
		hpoj->option[option].type,
		hpoj->option[option].unit,
		hpoj->option[option].size,
		hpoj->option[option].cap,
		hpoj->option[option].constraint_type);
	if (hpoj->option[option].constraint_type==SANE_CONSTRAINT_RANGE) {
		PTAL_LOG_DEBUG("\tmin=%d=0x%8.8X, max=%d=0x%8.8X, quant=%d\n",
			hpoj->option[option].constraint.range->min,
			hpoj->option[option].constraint.range->min,
			hpoj->option[option].constraint.range->max,
			hpoj->option[option].constraint.range->max,
			hpoj->option[option].constraint.range->quant);
	}

	return &hpoj->option[option];
}

extern SANE_Status sane_hpoj_control_option(SANE_Handle handle,SANE_Int option,
    SANE_Action action,void *pValue,SANE_Int *pInfo) {
	hpojScanner_t hpoj=(hpojScanner_t)handle;
	SANE_Int _info;
	SANE_Int *pIntValue=pValue;
	SANE_String pStrValue=pValue;
	SANE_Status retcode;

	if (!pInfo) pInfo=&_info;
	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_control_option(option=%d,"
		"action=%d)\n",
		hpoj->saneDevice.name,option,action);

	switch (action) {
	   case SANE_ACTION_GET_VALUE:
		switch (option) {
		   case OPTION_NUM_OPTIONS:
			*pIntValue=OPTION_LAST;
			break;

		   case OPTION_SCAN_MODE:
			switch (hpoj->currentScanMode) {
			   case SCAN_MODE_LINEART:
				strcpy(pStrValue,STR_SCAN_MODE_LINEART);
				break;
			   case SCAN_MODE_GRAYSCALE:
				strcpy(pStrValue,STR_SCAN_MODE_GRAYSCALE);
				break;
			   case SCAN_MODE_COLOR:
				strcpy(pStrValue,STR_SCAN_MODE_COLOR);
				break;
			   default:
				strcpy(pStrValue,STR_UNKNOWN);
				break;
			}
			break;

		   case OPTION_SCAN_RESOLUTION:
			*pIntValue=hpoj->currentResolution;
			break;

		   case OPTION_CONTRAST:
			*pIntValue=hpoj->currentContrast;
			break;

		   case OPTION_COMPRESSION:
			switch (hpoj->currentCompression) {
			   case COMPRESSION_NONE:
				strcpy(pStrValue,STR_COMPRESSION_NONE);
				break;
			   case COMPRESSION_MH:
				strcpy(pStrValue,STR_COMPRESSION_MH);
				break;
			   case COMPRESSION_MR:
				strcpy(pStrValue,STR_COMPRESSION_MR);
				break;
			   case COMPRESSION_MMR:
				strcpy(pStrValue,STR_COMPRESSION_MMR);
				break;
			   case COMPRESSION_JPEG:
				strcpy(pStrValue,STR_COMPRESSION_JPEG);
				break;
			   default:
				strcpy(pStrValue,STR_UNKNOWN);
				break;
			}
			break;

		   case OPTION_JPEG_COMPRESSION_FACTOR:
			*pIntValue=hpoj->currentJpegCompressionFactor;
			break;

		   case OPTION_BATCH_SCAN:
			*pIntValue=hpoj->currentBatchScan;
			break;

		   case OPTION_ADF_MODE:
			switch (hpoj->currentAdfMode) {
			   case ADF_MODE_AUTO:
				strcpy(pStrValue,STR_ADF_MODE_AUTO);
				break;
			   case ADF_MODE_FLATBED:
				strcpy(pStrValue,STR_ADF_MODE_FLATBED);
				break;
			   case ADF_MODE_ADF:
				strcpy(pStrValue,STR_ADF_MODE_ADF);
				break;
			   default:
				strcpy(pStrValue,STR_UNKNOWN);
				break;
			}
			break;

		   case OPTION_DUPLEX:
			*pIntValue=hpoj->currentDuplex;
			break;

		   case OPTION_LENGTH_MEASUREMENT:
			switch (hpoj->currentLengthMeasurement) {
			   case LENGTH_MEASUREMENT_UNKNOWN:
				strcpy(pStrValue,
					STR_LENGTH_MEASUREMENT_UNKNOWN);
				break;
			   case LENGTH_MEASUREMENT_UNLIMITED:
				strcpy(pStrValue,
					STR_LENGTH_MEASUREMENT_UNLIMITED);
				break;
			   case LENGTH_MEASUREMENT_APPROXIMATE:
				strcpy(pStrValue,
					STR_LENGTH_MEASUREMENT_APPROXIMATE);
				break;
			   case LENGTH_MEASUREMENT_PADDED:
				strcpy(pStrValue,
					STR_LENGTH_MEASUREMENT_PADDED);
				break;
			   case LENGTH_MEASUREMENT_EXACT:
				strcpy(pStrValue,
					STR_LENGTH_MEASUREMENT_EXACT);
				break;
			   default:
				strcpy(pStrValue,STR_UNKNOWN);
				break;
			}
			break;

		   case OPTION_TL_X:
			*pIntValue=hpoj->currentTlx;
			break;

		   case OPTION_TL_Y:
			*pIntValue=hpoj->currentTly;
			break;

		   case OPTION_BR_X:
			*pIntValue=hpoj->currentBrx;
			break;

		   case OPTION_BR_Y:
			*pIntValue=hpoj->currentBry;
			break;

		   default:
			return SANE_STATUS_INVAL;
		}
		break;

	   case SANE_ACTION_SET_VALUE:
		if (hpoj->option[option].cap&SANE_CAP_INACTIVE) {
			return SANE_STATUS_INVAL;
		}
		switch (option) {
		   case OPTION_SCAN_MODE:
			if (!strcasecmp(pStrValue,STR_SCAN_MODE_LINEART) &&
			    hpoj->supportsScanMode[SCAN_MODE_LINEART]) {
				hpoj->currentScanMode=SCAN_MODE_LINEART;
				break;
			}
			if (!strcasecmp(pStrValue,STR_SCAN_MODE_GRAYSCALE) &&
			    hpoj->supportsScanMode[SCAN_MODE_GRAYSCALE]) {
				hpoj->currentScanMode=SCAN_MODE_GRAYSCALE;
				break;
			}
			if (!strcasecmp(pStrValue,STR_SCAN_MODE_COLOR) &&
			    hpoj->supportsScanMode[SCAN_MODE_COLOR]) {
				hpoj->currentScanMode=SCAN_MODE_COLOR;
				break;
			}
			return SANE_STATUS_INVAL;

		   case OPTION_SCAN_RESOLUTION:
			if ((hpoj->option[option].constraint_type==
			      SANE_CONSTRAINT_WORD_LIST &&
			     !hpojNumListIsInList((SANE_Int *)hpoj->option[
			      option].constraint.word_list,*pIntValue)) ||
			    (hpoj->option[option].constraint_type==
			      SANE_CONSTRAINT_RANGE &&
			     (*pIntValue<hpoj->resolutionRange.min ||
			      *pIntValue>hpoj->resolutionRange.max))) {
				return SANE_STATUS_INVAL;
			}
			hpoj->currentResolution=*pIntValue;
			break;

		   case OPTION_CONTRAST:
			if (*pIntValue<hpoj->contrastRange.min ||
			    *pIntValue>hpoj->contrastRange.max) {
				return SANE_STATUS_INVAL;
			}
			hpoj->currentContrast=*pIntValue;
			break;

		   case OPTION_COMPRESSION:
		     {
			int supportedCompression=hpoj->supportsScanMode[
				hpoj->currentScanMode];
			if (!strcasecmp(pStrValue,STR_COMPRESSION_NONE) &&
			    supportedCompression&COMPRESSION_NONE) {
				hpoj->currentCompression=COMPRESSION_NONE;
				break;
			}
			if (!strcasecmp(pStrValue,STR_COMPRESSION_MH) &&
			    supportedCompression&COMPRESSION_MH) {
				hpoj->currentCompression=COMPRESSION_MH;
				break;
			}
			if (!strcasecmp(pStrValue,STR_COMPRESSION_MR) &&
			    supportedCompression&COMPRESSION_MR) {
				hpoj->currentCompression=COMPRESSION_MR;
				break;
			}
			if (!strcasecmp(pStrValue,STR_COMPRESSION_MMR) &&
			    supportedCompression&COMPRESSION_MMR) {
				hpoj->currentCompression=COMPRESSION_MMR;
				break;
			}
			if (!strcasecmp(pStrValue,STR_COMPRESSION_JPEG) &&
			    supportedCompression&COMPRESSION_JPEG) {
				hpoj->currentCompression=COMPRESSION_JPEG;
				break;
			}
			return SANE_STATUS_INVAL;
		     }

		   case OPTION_JPEG_COMPRESSION_FACTOR:
			if (*pIntValue<MIN_JPEG_COMPRESSION_FACTOR ||
			    *pIntValue>MAX_JPEG_COMPRESSION_FACTOR) {
				return SANE_STATUS_INVAL;
			}
			hpoj->currentJpegCompressionFactor=*pIntValue;
			break;

		   case OPTION_BATCH_SCAN:
			if (*pIntValue!=SANE_FALSE && *pIntValue!=SANE_TRUE) {
				return SANE_STATUS_INVAL;
			}
			hpoj->currentBatchScan=*pIntValue;
			if (!hpoj->currentBatchScan) {
				hpojResetScannerIfInNewPageState(hpoj);
			}
			break;

		   case OPTION_ADF_MODE:
			if (!strcasecmp(pStrValue,STR_ADF_MODE_AUTO) &&
			    hpoj->supportedAdfModes&ADF_MODE_AUTO) {
				hpoj->currentAdfMode=ADF_MODE_AUTO;
				break;
			}
			if (!strcasecmp(pStrValue,STR_ADF_MODE_FLATBED) &&
			    hpoj->supportedAdfModes&ADF_MODE_FLATBED) {
				hpoj->currentAdfMode=ADF_MODE_FLATBED;
				break;
			}
			if (!strcasecmp(pStrValue,STR_ADF_MODE_ADF) &&
			    hpoj->supportedAdfModes&ADF_MODE_ADF) {
				hpoj->currentAdfMode=ADF_MODE_ADF;
				break;
			}
			return SANE_STATUS_INVAL;

		   case OPTION_DUPLEX:
			if (*pIntValue!=SANE_FALSE && *pIntValue!=SANE_TRUE) {
				return SANE_STATUS_INVAL;
			}
			hpoj->currentDuplex=*pIntValue;
			break;

		   case OPTION_LENGTH_MEASUREMENT:
			if (!strcasecmp(pStrValue,
			      STR_LENGTH_MEASUREMENT_UNKNOWN)) {
				hpoj->currentLengthMeasurement=
					LENGTH_MEASUREMENT_UNKNOWN;
				break;
			}
			if (!strcasecmp(pStrValue,
			      STR_LENGTH_MEASUREMENT_UNLIMITED)) {
				if (hpoj->scannerType!=SCANNER_TYPE_PML) {
					return SANE_STATUS_INVAL;
				}
				hpoj->currentLengthMeasurement=
					LENGTH_MEASUREMENT_UNLIMITED;
				break;
			}
			if (!strcasecmp(pStrValue,
			      STR_LENGTH_MEASUREMENT_APPROXIMATE)) {
				hpoj->currentLengthMeasurement=
					LENGTH_MEASUREMENT_APPROXIMATE;
				break;
			}
			if (!strcasecmp(pStrValue,
			      STR_LENGTH_MEASUREMENT_PADDED)) {
				hpoj->currentLengthMeasurement=
					LENGTH_MEASUREMENT_PADDED;
				break;
			}
			if (!strcasecmp(pStrValue,
			      STR_LENGTH_MEASUREMENT_EXACT)) {
				hpoj->currentLengthMeasurement=
					LENGTH_MEASUREMENT_EXACT;
				break;
			}
			return SANE_STATUS_INVAL;

		   case OPTION_TL_X:
			if (*pIntValue<hpoj->tlxRange.min ||
			    *pIntValue>hpoj->tlxRange.max) {
				return SANE_STATUS_INVAL;
			}
			hpoj->currentTlx=*pIntValue;
			break;

		   case OPTION_TL_Y:
			if (*pIntValue<hpoj->tlyRange.min ||
			    *pIntValue>hpoj->tlyRange.max) {
				return SANE_STATUS_INVAL;
			}
			hpoj->currentTly=*pIntValue;
			break;

		   case OPTION_BR_X:
			if (*pIntValue<hpoj->brxRange.min ||
			    *pIntValue>hpoj->brxRange.max) {
				return SANE_STATUS_INVAL;
			}
			hpoj->currentBrx=*pIntValue;
			break;

		   case OPTION_BR_Y:
			if (*pIntValue<hpoj->bryRange.min ||
			    *pIntValue>hpoj->bryRange.max) {
				return SANE_STATUS_INVAL;
			}
			hpoj->currentBry=*pIntValue;
			break;

		   default:
			return SANE_STATUS_INVAL;
		}
		goto reload;

	   case SANE_ACTION_SET_AUTO:
		retcode=hpojSetDefaultValue(hpoj,option);
		if (retcode!=SANE_STATUS_GOOD) return retcode;
reload:
		*pInfo=hpojUpdateDescriptors(hpoj,option);
		PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_control_option(option=%d,"
			"action=%d): info=0x%2.2X\n",
			hpoj->saneDevice.name,option,action,*pInfo);
		break;

	   default:
		return SANE_STATUS_INVAL;
	}

	if ((action==SANE_ACTION_GET_VALUE || action==SANE_ACTION_SET_VALUE) &&
	    pValue) {
		if (hpoj->option[option].type==SANE_TYPE_STRING) {
			PTAL_LOG_DEBUG("hpoj:%s: %soption %d = <%s>\n",
				hpoj->saneDevice.name,
				(action==SANE_ACTION_SET_VALUE?"set ":""),
				option,pValue);
		} else {
			PTAL_LOG_DEBUG("hpoj:%s: %soption %d = %d = 0x%8.8X\n",
				hpoj->saneDevice.name,
				(action==SANE_ACTION_SET_VALUE?"set ":""),
				option,*(int *)pValue,*(int *)pValue);
		}
	}

	return SANE_STATUS_GOOD;
}

extern SANE_Status sane_hpoj_get_parameters(SANE_Handle handle,
    SANE_Parameters *pParams) {
	hpojScanner_t hpoj=(hpojScanner_t)handle;
	char *s="";
	if (!hpoj->hJob) {
		*pParams=hpoj->prescanParameters;
		s="pre";
	} else {
		*pParams=hpoj->scanParameters;
	}
	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_get_parameters(%sscan): "
		"format=%d, last_frame=%d, lines=%d, depth=%d, "
		"pixels_per_line=%d, bytes_per_line=%d.\n",
		hpoj->saneDevice.name,s,pParams->format,
		pParams->last_frame,pParams->lines,pParams->depth,
		pParams->pixels_per_line,pParams->bytes_per_line);
	return SANE_STATUS_GOOD;
}

static SANE_Status hpojProgramOptions(hpojScanner_t hpoj) {
	PTAL_LOG_DEBUG("hpoj:%s: hpojProgramOptions\n",hpoj->saneDevice.name);

	hpoj->effectiveScanMode=hpoj->currentScanMode;
	hpoj->effectiveResolution=hpoj->currentResolution;

    if (hpoj->scannerType==SCANNER_TYPE_SCL) {
	/* Set output data type and width. */
	switch (hpoj->currentScanMode) {
	   case SCAN_MODE_LINEART:
		hpojSclSendCommand(hpoj,SCL_CMD_SET_OUTPUT_DATA_TYPE,
			SCL_DATA_TYPE_LINEART);
		hpojSclSendCommand(hpoj,SCL_CMD_SET_DATA_WIDTH,
			SCL_DATA_WIDTH_LINEART);
		break;
	   case SCAN_MODE_GRAYSCALE:
		hpojSclSendCommand(hpoj,SCL_CMD_SET_OUTPUT_DATA_TYPE,
			SCL_DATA_TYPE_GRAYSCALE);
		hpojSclSendCommand(hpoj,SCL_CMD_SET_DATA_WIDTH,
			SCL_DATA_WIDTH_GRAYSCALE);
		break;
	   case SCAN_MODE_COLOR:
	   default:
		hpojSclSendCommand(hpoj,SCL_CMD_SET_OUTPUT_DATA_TYPE,
			SCL_DATA_TYPE_COLOR);
		hpojSclSendCommand(hpoj,SCL_CMD_SET_DATA_WIDTH,
			SCL_DATA_WIDTH_COLOR);
		break;
	}

	/* Set MFPDTF. */
	hpojSclSendCommand(hpoj,SCL_CMD_SET_MFPDTF,
		hpoj->mfpdtf?SCL_MFPDTF_ON:SCL_MFPDTF_OFF);

	/* Set compression. */
	hpojSclSendCommand(hpoj,SCL_CMD_SET_COMPRESSION,
		(hpoj->currentCompression==COMPRESSION_JPEG?
		 SCL_COMPRESSION_JPEG:SCL_COMPRESSION_NONE));

	/* Set JPEG compression factor. */
	hpojSclSendCommand(hpoj,SCL_CMD_SET_JPEG_COMPRESSION_FACTOR,
		hpoj->currentJpegCompressionFactor);

	/* Set X and Y resolution. */
	hpojSclSendCommand(hpoj,SCL_CMD_SET_X_RESOLUTION,
		hpoj->currentResolution);
	hpojSclSendCommand(hpoj,SCL_CMD_SET_Y_RESOLUTION,
		hpoj->currentResolution);

	/* Set X and Y position and extent. */
	hpojSclSendCommand(hpoj,SCL_CMD_SET_X_POSITION,
		MILLIMETERS_TO_DECIPIXELS(hpoj->effectiveTlx));
	hpojSclSendCommand(hpoj,SCL_CMD_SET_Y_POSITION,
		MILLIMETERS_TO_DECIPIXELS(hpoj->effectiveTly));
	hpojSclSendCommand(hpoj,SCL_CMD_SET_X_EXTENT,
		MILLIMETERS_TO_DECIPIXELS(
			hpoj->effectiveBrx-hpoj->effectiveTlx));
	hpojSclSendCommand(hpoj,SCL_CMD_SET_Y_EXTENT,
		MILLIMETERS_TO_DECIPIXELS(
			hpoj->effectiveBry-hpoj->effectiveTly));

	/* Download color map to OfficeJet Pro 11xx. */
	if (hpoj->scl.compat&(SCL_COMPAT_1150|SCL_COMPAT_1170)) {
		hpojSclSendCommand(hpoj,SCL_CMD_SET_DOWNLOAD_TYPE,
			SCL_DOWNLOAD_TYPE_COLORMAP);
		hpojSclSendCommand(hpoj,SCL_CMD_DOWNLOAD_BINARY_DATA,
			sizeof(hpoj11xxSeriesColorMap));
		ptalChannelWrite(hpoj->chan,(char *)hpoj11xxSeriesColorMap,
			sizeof(hpoj11xxSeriesColorMap));
	}

	/* For OfficeJet R and PSC 500 series, set CCD resolution to 600
	 * for lineart. */
	if (hpoj->scl.compat&SCL_COMPAT_R_SERIES &&
	    hpoj->currentScanMode==SCAN_MODE_LINEART) {
		hpojSclSendCommand(hpoj,SCL_CMD_SET_CCD_RESOLUTION,600);
	}

    } else /* if (hpoj->scannerType==SCANNER_TYPE_PML) */ {
	if (hpojScannerIsUninterruptible(hpoj,0)) {
		int pixelDataType;
		struct PmlResolution resolution;

		if (ptalPmlRequestGet(hpoj->pml.objPixelDataType,0)!=
		     PTAL_ERROR &&
		    ptalPmlGetIntegerValue(hpoj->pml.objPixelDataType,0,
		      &pixelDataType)!=PTAL_ERROR) {
			switch (pixelDataType) {
			   case PML_DATA_TYPE_LINEART:
				hpoj->effectiveScanMode=SCAN_MODE_LINEART;
				break;
			   case PML_DATA_TYPE_GRAYSCALE:
				hpoj->effectiveScanMode=SCAN_MODE_GRAYSCALE;
				break;
			   case PML_DATA_TYPE_COLOR:
				hpoj->effectiveScanMode=SCAN_MODE_COLOR;
				break;
			   default:
				break;
			}
		}

		if (ptalPmlRequestGet(hpoj->pml.objResolution,0)!=
		     PTAL_ERROR &&
		    ptalPmlGetValue(hpoj->pml.objResolution,0,
		      (char *)&resolution,sizeof(resolution))!=PTAL_ERROR) {
			hpoj->effectiveResolution=
				BEND_GET_LONG(resolution.x)>>16;
		}

		return SANE_STATUS_GOOD;
	}

	/* Set upload timeout. */
	if (ptalPmlRequestGet(hpoj->pml.objUploadTimeout,0)!=PTAL_ERROR) {
		ptalPmlSetIntegerValue(hpoj->pml.objUploadTimeout,
			PTAL_PML_TYPE_SIGNED_INTEGER,PML_UPLOAD_TIMEOUT);
		ptalPmlRequestSetRetry(hpoj->pml.objUploadTimeout,0,0);
	}

	/* Set pixel data type. */
	if (ptalPmlRequestGet(hpoj->pml.objPixelDataType,0)!=PTAL_ERROR) {
		int pixelDataType;
		switch (hpoj->currentScanMode) {
		   case SCAN_MODE_LINEART:
			pixelDataType=PML_DATA_TYPE_LINEART;
			break;
		   case SCAN_MODE_GRAYSCALE:
			pixelDataType=PML_DATA_TYPE_GRAYSCALE;
			break;
		   case SCAN_MODE_COLOR:
		   default:
			pixelDataType=PML_DATA_TYPE_COLOR;
			break;
		}
		ptalPmlSetIntegerValue(hpoj->pml.objPixelDataType,
			PTAL_PML_TYPE_ENUMERATION,pixelDataType);
		ptalPmlRequestSetRetry(hpoj->pml.objPixelDataType,0,0);
	}

	/* Set resolution. */
	if (ptalPmlRequestGet(hpoj->pml.objResolution,0)!=PTAL_ERROR) {
		struct PmlResolution resolution;
		BEND_SET_LONG(resolution.x,hpoj->currentResolution<<16);
		BEND_SET_LONG(resolution.y,hpoj->currentResolution<<16);
		ptalPmlSetValue(hpoj->pml.objResolution,
			PTAL_PML_TYPE_BINARY,(char *)&resolution,
			sizeof(resolution));
		ptalPmlRequestSetRetry(hpoj->pml.objResolution,0,0);
	}

	/* Set contrast. */
	if (!(hpoj->option[OPTION_CONTRAST].cap&SANE_CAP_INACTIVE)) {
		ptalPmlSetIntegerValue(hpoj->pml.objContrast,
			PTAL_PML_TYPE_SIGNED_INTEGER,hpoj->currentContrast);
		ptalPmlRequestSetRetry(hpoj->pml.objContrast,0,0);
	}

	/* Set compression. */
	if (ptalPmlRequestGet(hpoj->pml.objCompression,0)!=PTAL_ERROR) {
		int compression;
		switch (hpoj->currentCompression) {
		   case COMPRESSION_NONE:
			compression=PML_COMPRESSION_NONE;
			break;
		   case COMPRESSION_MH:
			compression=PML_COMPRESSION_MH;
			break;
		   case COMPRESSION_MR:
			compression=PML_COMPRESSION_MR;
			break;
		   case COMPRESSION_MMR:
			compression=PML_COMPRESSION_MMR;
			break;
		   case COMPRESSION_JPEG:
		   default:
			compression=PML_COMPRESSION_JPEG;
			break;
		}
		ptalPmlSetIntegerValue(hpoj->pml.objCompression,
			PTAL_PML_TYPE_ENUMERATION,compression);
		ptalPmlRequestSetRetry(hpoj->pml.objCompression,0,0);
	}

	/* Set JPEG compression factor. */
	if (ptalPmlRequestGet(hpoj->pml.objCompressionFactor,0)!=PTAL_ERROR) {
		ptalPmlSetIntegerValue(hpoj->pml.objCompressionFactor,
			PTAL_PML_TYPE_SIGNED_INTEGER,
			hpoj->currentJpegCompressionFactor);
		ptalPmlRequestSetRetry(hpoj->pml.objCompressionFactor,0,0);
	}

	/* Set Automatic Background Control thresholds. */
	if (ptalPmlRequestGet(hpoj->pml.objAbcThresholds,0)!=PTAL_ERROR) {
		static struct {
			unsigned char ceiling;
			unsigned char floor;
		} __attribute__((packed)) abcThresholds={ 0xE6,0x00 };
		ptalPmlSetValue(hpoj->pml.objAbcThresholds,
			PTAL_PML_TYPE_BINARY,(char *)&abcThresholds,
			sizeof(abcThresholds));
		ptalPmlRequestSetRetry(hpoj->pml.objAbcThresholds,0,0);
	}

	/* Set sharpening coefficient. */
	if (ptalPmlRequestGet(hpoj->pml.objSharpeningCoefficient,0)!=
	    PTAL_ERROR) {
		static int sharpeningCoefficient=0x37;
		ptalPmlSetIntegerValue(hpoj->pml.objSharpeningCoefficient,
			PTAL_PML_TYPE_SIGNED_INTEGER,sharpeningCoefficient);
		ptalPmlRequestSetRetry(hpoj->pml.objSharpeningCoefficient,0,0);
	}

	/* Set neutral clip thresholds. */
	if (ptalPmlRequestGet(hpoj->pml.objNeutralClipThresholds,0)!=
	    PTAL_ERROR) {
		static struct {
			unsigned char luminance;
			unsigned char chrominance;
		} __attribute__((packed)) neutralClipThresholds={ 0xFA,0x05 };
		ptalPmlSetValue(hpoj->pml.objNeutralClipThresholds,
			PTAL_PML_TYPE_BINARY,(char *)&neutralClipThresholds,
			sizeof(neutralClipThresholds));
		ptalPmlRequestSetRetry(hpoj->pml.objNeutralClipThresholds,0,0);
	}

	/* Set tone map if supported (needed for T series, breaks others). */
	if (ptalPmlRequestGet(hpoj->pml.objToneMap,0)!=PTAL_ERROR) {
		ptalPmlSetValue(hpoj->pml.objToneMap,
			PTAL_PML_TYPE_BINARY,(char *)hpojTSeriesToneMap,
			sizeof(hpojTSeriesToneMap));
		ptalPmlRequestSetRetry(hpoj->pml.objToneMap,0,0);
	}

	/* Set copier reduction. */
	if (ptalPmlRequestGet(hpoj->pml.objCopierReduction,0)!=PTAL_ERROR) {
		static int copierReduction=100;
		ptalPmlSetIntegerValue(hpoj->pml.objCopierReduction,
			PTAL_PML_TYPE_SIGNED_INTEGER,copierReduction);
		ptalPmlRequestSetRetry(hpoj->pml.objCopierReduction,0,0);
	}
    }

	return SANE_STATUS_GOOD;
}

static SANE_Status hpojAdvanceDocument(hpojScanner_t hpoj) {
	SANE_Status retcode;

	PTAL_LOG_DEBUG("\nhpoj:%s: hpojAdvanceDocument:\n"
		"beforeScan=%d, already{Pre,Post}AdvancedDocument={%d,%d}, "
		"noDocsConditionPending=%d, "
		"currentPageNumber=%d, currentSideNumber=%d.\n",
		hpoj->saneDevice.name,hpoj->beforeScan,
		hpoj->alreadyPreAdvancedDocument,
		hpoj->alreadyPostAdvancedDocument,hpoj->noDocsConditionPending,
		hpoj->currentPageNumber,hpoj->currentSideNumber);

	if (hpoj->beforeScan) {
		hpoj->alreadyPostAdvancedDocument=0;
		retcode=hpojScannerToSaneStatus(hpoj);
		if (retcode!=SANE_STATUS_GOOD ||
		    hpoj->alreadyPreAdvancedDocument) {
			goto abort;
		}
		hpoj->alreadyPreAdvancedDocument=1;

	} else {
		if (hpoj->alreadyPostAdvancedDocument) {
			retcode=SANE_STATUS_GOOD;
			goto abort;
		}
		hpoj->alreadyPreAdvancedDocument=0;
		hpoj->alreadyPostAdvancedDocument=1;
	}

    if (hpoj->scannerType==SCANNER_TYPE_SCL) {
	int documentLoaded=0;

	retcode=hpojSclInquire(hpoj,SCL_CMD_INQUIRE_DEVICE_PARAMETER,
		SCL_INQ_ADF_DOCUMENT_LOADED,&documentLoaded,0,0);
	if (retcode!=SANE_STATUS_GOOD && retcode!=SANE_STATUS_UNSUPPORTED) {
		goto abort;
	}

	if (hpoj->currentSideNumber==1) {
		if (hpoj->currentDuplex) {
			/* Duplex change side. */
			retcode=hpojSclSendCommandCheckError(hpoj,
				SCL_CMD_CHANGE_DOCUMENT,
				SCL_CHANGE_DOC_DUPLEX_SIDE);
			if (retcode!=SANE_STATUS_GOOD) goto abort;
			hpoj->alreadyPreAdvancedDocument=1;
			hpoj->currentSideNumber=2;

		} else if (hpoj->scl.unloadAfterScan && !hpoj->beforeScan) {
			/* Unload document. */
			retcode=hpojSclSendCommandCheckError(hpoj,
				SCL_CMD_UNLOAD_DOCUMENT,0);
			if (retcode!=SANE_STATUS_GOOD) goto abort;
			hpoj->currentSideNumber=0;
			if (!documentLoaded) {
				goto noDocs;
			}

		} else if (documentLoaded) {
			goto changeDoc;

		} else {
unloadDoc:
			/* Unload document. */
			retcode=hpojSclSendCommandCheckError(hpoj,
				SCL_CMD_UNLOAD_DOCUMENT,0);
			if (retcode!=SANE_STATUS_GOOD) goto abort;
			goto noDocs;
		}

	} else if (hpoj->currentSideNumber==2) {
		/* Duplex change side. */
		retcode=hpojSclSendCommandCheckError(hpoj,
			SCL_CMD_CHANGE_DOCUMENT,
			SCL_CHANGE_DOC_DUPLEX_SIDE);
		if (retcode!=SANE_STATUS_GOOD) goto abort;
		if (documentLoaded) {
			goto changeDoc;
		}
		goto unloadDoc;

	} else /* if (!hpoj->currentSideNumber) */ {
		if (documentLoaded && hpoj->beforeScan &&
		    hpoj->currentAdfMode!=ADF_MODE_FLATBED) {
changeDoc:
			if (hpoj->currentDuplex) {
				/* Duplex change document. */
				retcode=hpojSclSendCommandCheckError(hpoj,
					SCL_CMD_CHANGE_DOCUMENT,
					SCL_CHANGE_DOC_DUPLEX);
			} else {
				/* Simplex change document. */
				retcode=hpojSclSendCommandCheckError(hpoj,
					SCL_CMD_CHANGE_DOCUMENT,
					SCL_CHANGE_DOC_SIMPLEX);
			}
			if (retcode!=SANE_STATUS_GOOD) goto abort;
			hpoj->alreadyPreAdvancedDocument=1;
			hpoj->currentPageNumber++;
			hpoj->currentSideNumber=1;

		} else {
noDocs:
			hpoj->currentPageNumber=0;
			hpoj->currentSideNumber=0;
			if (hpoj->beforeScan) {
				if (hpoj->currentAdfMode==ADF_MODE_ADF) {
					retcode=SANE_STATUS_NO_DOCS;
					goto abort;
				}

			} else if (hpoj->currentBatchScan) {
				if (hpoj->endOfData) {
					hpoj->noDocsConditionPending=1;
				}
				retcode=SANE_STATUS_NO_DOCS;
				goto abort;
			}
		}
	}

	if (!hpoj->beforeScan && !hpoj->currentPageNumber) {
		retcode=SANE_STATUS_NO_DOCS;
		goto abort;
	}

    } else /* if (hpoj->scannerType==SCANNER_TYPE_PML) */ {
	if (hpoj->beforeScan) {
		hpoj->currentPageNumber++;
		hpoj->currentSideNumber=1;

	} else if (hpoj->currentSideNumber) {
		int uploadState;

		if (ptalPmlRequestGet(hpoj->pml.objUploadState,0)==PTAL_ERROR ||
		    ptalPmlGetIntegerValue(hpoj->pml.objUploadState,0,
		     &uploadState)==PTAL_ERROR ||
		    uploadState==PML_UPLOAD_STATE_IDLE) {
			hpoj->currentPageNumber=0;
			hpoj->currentSideNumber=0;
			if (hpoj->currentBatchScan) {
				if (hpoj->endOfData) {
					hpoj->noDocsConditionPending=1;
				}
			}
			retcode=SANE_STATUS_NO_DOCS;
			goto abort;
		}
	}
    }

	if (!hpoj->beforeScan && !hpoj->endOfData) {
		retcode=SANE_STATUS_NO_DOCS;
		goto abort;
	}

	retcode=SANE_STATUS_GOOD;
abort:
	PTAL_LOG_DEBUG("\nhpoj:%s: hpojAdvanceDocument returns %d:\n"
		"beforeScan=%d, already{Pre,Post}AdvancedDocument={%d,%d}, "
		"noDocsConditionPending=%d, "
		"currentPageNumber=%d, currentSideNumber=%d.\n\n",
		hpoj->saneDevice.name,retcode,hpoj->beforeScan,
		hpoj->alreadyPreAdvancedDocument,
		hpoj->alreadyPostAdvancedDocument,hpoj->noDocsConditionPending,
		hpoj->currentPageNumber,hpoj->currentSideNumber);

	if (retcode!=SANE_STATUS_GOOD) {
		hpoj->alreadyPreAdvancedDocument=0;
		hpoj->alreadyPostAdvancedDocument=0;
		hpoj->currentPageNumber=0;
		hpoj->currentSideNumber=0;
	}
	return retcode;
}

extern void sane_hpoj_cancel(SANE_Handle handle) {
	hpojScanner_t hpoj=(hpojScanner_t)handle;

	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_cancel\n",hpoj->saneDevice.name);

	if (hpoj->mfpdtf) {
		ptalMfpdtfLogToFile(hpoj->mfpdtf,0);
	}
	if (hpoj->hJob) {
		ipClose(hpoj->hJob);
		hpoj->hJob=0;
	}
	if (hpoj->fdSaveCompressedData>=0) {
		close(hpoj->fdSaveCompressedData);
		hpoj->fdSaveCompressedData=PTAL_ERROR;
	}

	if (hpojAdvanceDocument(hpoj)!=SANE_STATUS_GOOD) {
		hpojConnEndScan(hpoj);
	}
}

static SANE_Status hpojPmlCheckForScanFailure(hpojScanner_t hpoj) {
	if (hpoj->scannerType!=SCANNER_TYPE_PML ||
	    !hpoj->pml.scanDone ||
	    hpoj->pml.previousUploadState!=PML_UPLOAD_STATE_ABORTED) {
		return SANE_STATUS_GOOD;
	}

	return hpojScannerToSaneError(hpoj);
}

static void hpojMfpdtfPardonReadTimeout(hpojScanner_t hpoj,int *prService) {
	if (*prService&PTAL_MFPDTF_RESULT_READ_TIMEOUT &&
	    hpoj->scannerType==SCANNER_TYPE_PML &&
	    hpoj->endOfData &&
	    hpoj->pml.scanDone) {
		*prService&=~PTAL_MFPDTF_RESULT_READ_TIMEOUT;
	}
}

extern SANE_Status sane_hpoj_start(SANE_Handle handle) {
	hpojScanner_t hpoj=(hpojScanner_t)handle;
	SANE_Status retcode;
	IP_IMAGE_TRAITS traits;
	IP_XFORM_SPEC xforms[IP_MAX_XFORMS],*pXform=xforms;
	WORD wResult;
	char *saveCompressedFilename=getenv("SANE_HPOJ_SAVE_COMPRESSED");

	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_start\n",hpoj->saneDevice.name);
	hpoj->endOfData=0;

	/* If we just scanned the last page of a batch scan, then
	 * return the obligatory SANE_STATUS_NO_DOCS condition. */
	if (hpoj->noDocsConditionPending) {
		hpoj->noDocsConditionPending=0;
		retcode=SANE_STATUS_NO_DOCS;
		goto abort;
	}

	/* Open scanner command channel. */
	retcode=hpojConnPrepareScan(hpoj);
	if (retcode!=SANE_STATUS_GOOD) goto abort;

	/* Change document if needed. */
	hpoj->beforeScan=1;
	retcode=hpojAdvanceDocument(hpoj);
	hpoj->beforeScan=0;
	if (retcode!=SANE_STATUS_GOOD) goto abort;

	/* Some PML scanners need the scan channel opened now. */
	retcode=hpojConnBeginScan(hpoj,0);
	if (retcode!=SANE_STATUS_GOOD) goto abort;

	/* Program options. */
	retcode=hpojProgramOptions(hpoj);
	if (retcode!=SANE_STATUS_GOOD) goto abort;

	hpoj->scanParameters=hpoj->prescanParameters;
	memset(xforms,0,sizeof(xforms));
	traits.iPixelsPerRow=-1;
	switch (hpoj->effectiveScanMode) {
	   case SCAN_MODE_LINEART:
		hpoj->scanParameters.format=SANE_FRAME_GRAY;
		hpoj->scanParameters.depth=1;
		traits.iBitsPerPixel=1;
		break;
	   case SCAN_MODE_GRAYSCALE:
		hpoj->scanParameters.format=SANE_FRAME_GRAY;
		hpoj->scanParameters.depth=8;
		traits.iBitsPerPixel=8;
		break;
	   case SCAN_MODE_COLOR:
	   default:
		hpoj->scanParameters.format=SANE_FRAME_RGB;
		hpoj->scanParameters.depth=8;
		traits.iBitsPerPixel=24;
		break;
	}
	traits.lHorizDPI=hpoj->effectiveResolution<<16;
	traits.lVertDPI=hpoj->effectiveResolution<<16;
	traits.lNumRows=-1;
	traits.iNumPages=1;
	traits.iPageNum=1;

    if (hpoj->scannerType==SCANNER_TYPE_SCL) {
	int lines,pixelsPerLine;

	/* Inquire exact image dimensions. */
	if (hpojSclInquire(hpoj,SCL_CMD_INQUIRE_DEVICE_PARAMETER,
	     SCL_INQ_NUMBER_OF_SCAN_LINES,&lines,0,0)==SANE_STATUS_GOOD) {
		traits.lNumRows=lines;
	}
	hpojSclInquire(hpoj,SCL_CMD_INQUIRE_DEVICE_PARAMETER,
		SCL_INQ_PIXELS_PER_SCAN_LINE,&pixelsPerLine,0,0);
	traits.iPixelsPerRow=pixelsPerLine;

	hpojSclSendCommand(hpoj,SCL_CMD_CLEAR_ERROR_STACK,0);

	/* Start scanning. */
	hpojSclSendCommand(hpoj,SCL_CMD_SCAN_WINDOW,0);

    } else /* if (hpoj->scannerType==SCANNER_TYPE_PML) */ {
	int i,startedHere=0;

	hpoj->pml.scanDone=0;

	/* Start scanning. */
	ptalPmlRequestGet(hpoj->pml.objUploadState,0);
	ptalPmlGetIntegerValue(hpoj->pml.objUploadState,0,
		&hpoj->pml.previousUploadState);
	if (hpoj->pml.previousUploadState==PML_UPLOAD_STATE_IDLE ||
	    !hpoj->pml.alreadyRestarted) {
		ptalPmlSetIntegerValue(hpoj->pml.objUploadState,
			PTAL_PML_TYPE_ENUMERATION,PML_UPLOAD_STATE_START);
		ptalPmlRequestSetRetry(hpoj->pml.objUploadState,0,0);
		startedHere=1;
	}
	hpoj->pml.alreadyRestarted=0;

	/* Look for a confirmation that the scan started or failed. */
	for (i=0;;i++) {
		ptalPmlRequestGet(hpoj->pml.objUploadState,0);
		ptalPmlGetIntegerValue(hpoj->pml.objUploadState,0,
			&hpoj->pml.previousUploadState);
		if (hpoj->pml.previousUploadState==PML_UPLOAD_STATE_ACTIVE ||
		    !startedHere) {
			break;
		}
		if (i>PML_START_SCAN_WAIT_ACTIVE_MAX_RETRIES ||
		    hpoj->pml.previousUploadState!=PML_UPLOAD_STATE_START) {
			retcode=hpojScannerToSaneError(hpoj);
			if (retcode==SANE_STATUS_GOOD) {
				retcode=SANE_STATUS_IO_ERROR;
			}
			goto abort;
		}
		sleep(PML_START_SCAN_WAIT_ACTIVE_DELAY);
	}
    }

	/* Some PML scanners need the scan channel opened now. */
	retcode=hpojConnBeginScan(hpoj,1);
	if (retcode!=SANE_STATUS_GOOD) goto abort;

    if (hpoj->mfpdtf) {
	ptalMfpdtfReadSetTimeout(hpoj->mfpdtf,MFPDTF_EARLY_READ_TIMEOUT);
	ptalMfpdtfReadStart(hpoj->mfpdtf);
	ptalMfpdtfLogToFile(hpoj->mfpdtf,getenv("SANE_HPOJ_SAVE_MFPDTF"));
     while (42) {
	int rService,sopEncoding;

	rService=ptalMfpdtfReadService(hpoj->mfpdtf);
	retcode=hpojPmlCheckForScanFailure(hpoj);
	if (retcode!=SANE_STATUS_GOOD) goto abort;
	hpojMfpdtfPardonReadTimeout(hpoj,&rService);
	if (rService&PTAL_MFPDTF_RESULT_ERROR_MASK) {
		PTAL_LOG_WARN("hpoj:%s: sane_hpoj_start: "
			"ptalMfpdtfReadService() returns 0x%4.4X!\n",
			hpoj->saneDevice.name,rService);
		retcode=SANE_STATUS_IO_ERROR;
		goto abort;
	}
	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_start: "
		"ptalMfpdtfReadService() returns 0x%4.4X.\n",
		hpoj->saneDevice.name,rService);

      if (rService&PTAL_MFPDTF_RESULT_NEW_VARIANT_HEADER && hpoj->preDenali) {
	union ptalMfpdtfVariantHeader_u vheader;
	ptalMfpdtfReadGetVariantHeader(hpoj->mfpdtf,&vheader,sizeof(vheader));
	traits.iPixelsPerRow=LEND_GET_SHORT(vheader.faxArtoo.pixelsPerRow);
	traits.iBitsPerPixel=1;
	switch (vheader.faxArtoo.dataCompression) {
	   case PTAL_MFPDTF_RASTER_MH:
		sopEncoding=PTAL_MFPDTF_RASTER_MH;
		break;
	   case PTAL_MFPDTF_RASTER_MR:
		sopEncoding=PTAL_MFPDTF_RASTER_MR;
		break;
	   case PTAL_MFPDTF_RASTER_MMR:
	   default:
		sopEncoding=PTAL_MFPDTF_RASTER_MMR;
		break;
	}
	goto setupDecoder;

      } else if (rService&PTAL_MFPDTF_RESULT_NEW_START_OF_PAGE_RECORD) {
	struct ptalMfpdtfImageStartPageRecord_s sop;

	if (hpoj->scannerType==SCANNER_TYPE_SCL) {
		if (hpoj->currentCompression==COMPRESSION_NONE) {
			goto rawDecode;
		} else /* if (hpoj->currentCompression==COMPRESSION_JPEG) */ {
			goto jpegDecode;
		}
	}

	/* Read SOP record and set image pipeline input traits. */
	ptalMfpdtfReadGetStartPageRecord(hpoj->mfpdtf,&sop,sizeof(sop));
	traits.iPixelsPerRow=LEND_GET_SHORT(sop.black.pixelsPerRow);
	traits.iBitsPerPixel=LEND_GET_SHORT(sop.black.bitsPerPixel);
	traits.lHorizDPI=LEND_GET_LONG(sop.black.xres);
	traits.lVertDPI=LEND_GET_LONG(sop.black.yres);
	sopEncoding=sop.encoding;
setupDecoder:
	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_start: MFPDTF pixelsPerRow=%d, "
		"bitsPerPixel=%d, xres=0x%8.8X, yres=0x%8.8X, encoding=%d.\n",
		hpoj->saneDevice.name,
		traits.iPixelsPerRow,traits.iBitsPerPixel,
		traits.lHorizDPI,traits.lVertDPI,sopEncoding);

	/* Set up image-processing pipeline. */
	switch (sopEncoding) {
	   case PTAL_MFPDTF_RASTER_MH:
		pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MH;
		goto faxDecode;
	   case PTAL_MFPDTF_RASTER_MR:
		pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MR;
		goto faxDecode;
	   case PTAL_MFPDTF_RASTER_MMR:
		pXform->aXformInfo[IP_FAX_FORMAT].dword=IP_FAX_MMR;
faxDecode:
		ADD_XFORM(X_FAX_DECODE);
		break;

	   case PTAL_MFPDTF_RASTER_BITMAP:
	   case PTAL_MFPDTF_RASTER_GRAYMAP:
	   case PTAL_MFPDTF_RASTER_RGB:
rawDecode:
		break;

	   case PTAL_MFPDTF_RASTER_JPEG:
jpegDecode:
		pXform->aXformInfo[
			IP_JPG_DECODE_FROM_DENALI].dword=
			hpoj->fromDenali;
		ADD_XFORM(X_JPG_DECODE);

		pXform->aXformInfo[
			IP_CNV_COLOR_SPACE_WHICH_CNV].dword=
			IP_CNV_YCC_TO_SRGB;
		pXform->aXformInfo[
			IP_CNV_COLOR_SPACE_GAMMA].dword=
			0x00010000;
		ADD_XFORM(X_CNV_COLOR_SPACE);
		break;

	   case PTAL_MFPDTF_RASTER_YCC411:
	   case PTAL_MFPDTF_RASTER_PCL:
	   case PTAL_MFPDTF_RASTER_NOT:
	   default:
		/* Skip processing for unknown encodings. */
		PTAL_LOG_WARN("hpoj:%s: sane_hpoj_start: "
			"device specified unknown image encoding=%d!\n",
			hpoj->saneDevice.name,sopEncoding);
	}

	continue;
      }

      if (rService&PTAL_MFPDTF_RESULT_IMAGE_DATA_PENDING) {
	ptalMfpdtfReadSetTimeout(hpoj->mfpdtf,MFPDTF_LATER_READ_TIMEOUT);
	break;
      }

	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_start: "
		"Unhandled ptalMfpdtfReadService() result=0x%4.4X.\n",
		hpoj->saneDevice.name,rService);
     }
    }
	hpoj->scanParameters.pixels_per_line=traits.iPixelsPerRow;
	hpoj->scanParameters.lines=traits.lNumRows;
	if (hpoj->scanParameters.lines<0) {
		hpoj->scanParameters.lines=MILLIMETERS_TO_PIXELS(
			hpoj->bryRange.max,hpoj->effectiveResolution);
	}

	if (hpoj->scannerType==SCANNER_TYPE_SCL) {
		/* We have to invert bilevel data from SCL scanners. */
		if (hpoj->effectiveScanMode==SCAN_MODE_LINEART) {
			ADD_XFORM(X_INVERT);

		} else /* if (hpoj->effectiveScanMode==SCAN_MODE_COLOR) */ {
			/* Do gamma correction on OfficeJet Pro 11xx. */
			if (hpoj->scl.compat&(SCL_COMPAT_1150|SCL_COMPAT_1170)) {
				pXform->aXformInfo[
					IP_TABLE_WHICH].dword=
					IP_TABLE_USER;
				pXform->aXformInfo[
					IP_TABLE_OPTION].pvoid=
					(char *)hpoj11xxSeriesGammaTable;
				ADD_XFORM(X_TABLE);
			}
		}

	} else /* if (hpoj->scannerType==SCANNER_TYPE_PML) */ {
		int mmWidth=PIXELS_TO_MILLIMETERS(
			traits.iPixelsPerRow,hpoj->effectiveResolution);

		/* Set up X_CROP xform. */
		pXform->aXformInfo[IP_CROP_LEFT].dword=
			MILLIMETERS_TO_PIXELS(
				hpoj->effectiveTlx,
				hpoj->effectiveResolution);
		if (hpoj->effectiveBrx<hpoj->brxRange.max &&
		    hpoj->effectiveBrx<mmWidth) {
			pXform->aXformInfo[IP_CROP_RIGHT].dword=
				MILLIMETERS_TO_PIXELS(
					mmWidth-hpoj->effectiveBrx,
					hpoj->effectiveResolution);
		}
		pXform->aXformInfo[IP_CROP_TOP].dword=
			MILLIMETERS_TO_PIXELS(
				hpoj->effectiveTly,
				hpoj->effectiveResolution);
		if (hpoj->currentLengthMeasurement!=
		     LENGTH_MEASUREMENT_UNLIMITED) {
			hpoj->scanParameters.lines=
			    pXform->aXformInfo[IP_CROP_MAXOUTROWS].dword=
				MILLIMETERS_TO_PIXELS(
					hpoj->effectiveBry-hpoj->effectiveTly,
					hpoj->effectiveResolution);
		}
		hpoj->scanParameters.pixels_per_line-=
			pXform->aXformInfo[IP_CROP_LEFT].dword+
			pXform->aXformInfo[IP_CROP_RIGHT].dword;
		ADD_XFORM(X_CROP);
	}

	if (hpoj->currentLengthMeasurement==LENGTH_MEASUREMENT_PADDED) {
		pXform->aXformInfo[IP_PAD_LEFT].dword=0;
		pXform->aXformInfo[IP_PAD_RIGHT].dword=0;
		pXform->aXformInfo[IP_PAD_TOP].dword=0;
		pXform->aXformInfo[IP_PAD_BOTTOM].dword=0;
		pXform->aXformInfo[IP_PAD_VALUE].dword=
			(hpoj->effectiveScanMode==SCAN_MODE_LINEART)?
			PAD_VALUE_LINEART:PAD_VALUE_GRAYSCALE_COLOR;
		pXform->aXformInfo[IP_PAD_MIN_HEIGHT].dword=
			hpoj->scanParameters.lines;
		ADD_XFORM(X_PAD);
	}

	/* If we didn't set up any xforms by now, then add the dummy
	 * "skel" xform to simplify our subsequent code path. */
	if (pXform==xforms) {
		ADD_XFORM(X_SKEL);
	}

	wResult=ipOpen(pXform-xforms,xforms,0,&hpoj->hJob);
	if (wResult!=IP_DONE || !hpoj->hJob) {
		PTAL_LOG_WARN("hpoj:%s: sane_hpoj_start: "
			"ipOpen() returns 0x%4.4X!\n",
			hpoj->saneDevice.name,wResult);
		retcode=SANE_STATUS_INVAL;
		goto abort;
	}

	traits.iComponentsPerPixel=((traits.iBitsPerPixel%3)?1:3);
	wResult=ipSetDefaultInputTraits(hpoj->hJob,&traits);
	if (wResult!=IP_DONE) {
		PTAL_LOG_WARN("hpoj:%s: sane_hpoj_start: "
			"ipSetDefaultInputTraits() returns 0x%4.4X!\n",
			hpoj->saneDevice.name,wResult);
		retcode=SANE_STATUS_INVAL;
		goto abort;
	}

	hpoj->scanParameters.bytes_per_line=BYTES_PER_LINE(
		hpoj->scanParameters.pixels_per_line,
		hpoj->scanParameters.depth*
		 (hpoj->scanParameters.format==SANE_FRAME_RGB?3:1));
	hpoj->totalBytesRemaining=
		hpoj->scanParameters.bytes_per_line*
		hpoj->scanParameters.lines;
	hpoj->bufferOffset=0;
	hpoj->bufferBytesRemaining=0;

	if (saveCompressedFilename) {
		hpoj->fdSaveCompressedData=creat(saveCompressedFilename,0600);
		PTAL_LOG_ERROR("hpoj:%s: hpojip-test %s "
			"-defwidth %d -defbpp %d -defcpp %d "
			"-defhdpi %d -defvdpi %d "
			"-defheight %d -defpages %d -defpagenum %d\n",
			hpoj->saneDevice.name,
			saveCompressedFilename,
			traits.iPixelsPerRow,
			traits.iBitsPerPixel,
			traits.iComponentsPerPixel,
			traits.lHorizDPI>>16,
			traits.lVertDPI>>16,
			traits.lNumRows,
			traits.iNumPages,
			traits.iPageNum);
	}

	if (hpoj->currentLengthMeasurement==LENGTH_MEASUREMENT_UNKNOWN ||
	    hpoj->currentLengthMeasurement==LENGTH_MEASUREMENT_UNLIMITED) {
		hpoj->scanParameters.lines=-1;

	} else if (hpoj->currentLengthMeasurement==LENGTH_MEASUREMENT_EXACT) {
		/* TODO: Set up spool file, scan the whole image into it,
		 * and set "hpoj->scanParameters.lines" accordingly.
		 * Then in sane_hpoj_read, read out of the file. */

	}

	retcode=SANE_STATUS_GOOD;
abort:
	if (retcode!=SANE_STATUS_GOOD) {
		hpojConnEndScan(hpoj);
		sane_hpoj_cancel(handle);
	}

	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_start returns %d.\n",
		hpoj->saneDevice.name,retcode);
	return retcode;
}

extern SANE_Status sane_hpoj_read(SANE_Handle handle,SANE_Byte *data,
    SANE_Int maxLength,SANE_Int *pLength) {
	hpojScanner_t hpoj=(hpojScanner_t)handle;
	SANE_Status retcode;
	DWORD dwInputAvail;
	LPBYTE pbInputBuf;
	DWORD dwInputUsed,dwInputNextPos;
	DWORD dwOutputAvail=maxLength;
	LPBYTE pbOutputBuf=data;
	DWORD dwOutputUsed,dwOutputThisPos;
	WORD wResult;

	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_read(maxLength=%d)\n",
		hpoj->saneDevice.name,maxLength);

	if (!hpoj->hJob) {
		PTAL_LOG_WARN("hpoj:%s: sane_hpoj_read(maxLength=%d): "
			"No scan pending!\n",hpoj->saneDevice.name,maxLength);
		retcode=SANE_STATUS_EOF;
		goto abort;
	}

needMoreData:
    if (hpoj->bufferBytesRemaining<=0 && !hpoj->endOfData) {
	if (!hpoj->mfpdtf) {
		int r,len=hpoj->totalBytesRemaining;
		PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_read: "
			"totalBytesRemaining=%d.\n",
			hpoj->saneDevice.name,hpoj->totalBytesRemaining);
		if (len<=0) {
			hpoj->endOfData=1;
		} else {
			static struct timeval startTimeout=
				{NON_MFPDTF_READ_START_TIMEOUT,0};
			static struct timeval continueTimeout=
				{NON_MFPDTF_READ_CONTINUE_TIMEOUT,0};
			if (len>LEN_BUFFER) len=LEN_BUFFER;
			r=ptalChannelReadTimeout(hpoj->chan,hpoj->inBuffer,
				len,&startTimeout,&continueTimeout);
			if (r<0) {

				retcode=SANE_STATUS_IO_ERROR;
				goto abort;
			}
			hpoj->bufferBytesRemaining=r;
			hpoj->totalBytesRemaining-=r;
		}

	} else while (42) {
		int rService;
		rService=ptalMfpdtfReadService(hpoj->mfpdtf);
		retcode=hpojPmlCheckForScanFailure(hpoj);
		if (retcode!=SANE_STATUS_GOOD) goto abort;
		hpojMfpdtfPardonReadTimeout(hpoj,&rService);
		if (rService&PTAL_MFPDTF_RESULT_ERROR_MASK) {
			PTAL_LOG_WARN("hpoj:%s: sane_hpoj_read: "
				"ptalMfpdtfReadService() returns 0x%4.4X!\n",
				hpoj->saneDevice.name,rService);
			retcode=SANE_STATUS_IO_ERROR;
			goto abort;
		}
		PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_read: "
			"ptalMfpdtfReadService() returns 0x%4.4X.\n",
			hpoj->saneDevice.name,rService);

		if (rService&PTAL_MFPDTF_RESULT_IMAGE_DATA_PENDING) {
			hpoj->bufferBytesRemaining=ptalMfpdtfReadInnerBlock(
				hpoj->mfpdtf,hpoj->inBuffer,LEN_BUFFER);
			rService=ptalMfpdtfReadGetLastServiceResult(
				hpoj->mfpdtf);
			hpojMfpdtfPardonReadTimeout(hpoj,&rService);
			if (rService&PTAL_MFPDTF_RESULT_ERROR_MASK) {

				retcode=SANE_STATUS_IO_ERROR;
				goto abort;
			}
			break;
		}

		if (rService&PTAL_MFPDTF_RESULT_NEW_END_OF_PAGE_RECORD ||
		    (rService&PTAL_MFPDTF_RESULT_END_PAGE && hpoj->preDenali)) {
			hpoj->endOfData=1;
		}
		if (hpoj->endOfData) {
			if (hpoj->scannerType==SCANNER_TYPE_PML &&
		            !hpoj->pml.scanDone) {
				continue;
			}
			break;
		}

		PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_read: "
			"Unhandled ptalMfpdtfReadService() result=0x%4.4X.\n",
			hpoj->saneDevice.name,rService);
	}

	hpoj->bufferOffset=0;
	if (hpoj->preDenali) {
		ipMirrorBytes(hpoj->inBuffer,hpoj->bufferBytesRemaining);
	}
    }

	dwInputAvail=hpoj->bufferBytesRemaining;
	if (hpoj->bufferBytesRemaining<=0 && hpoj->endOfData) {
		pbInputBuf=0;
	} else {
		pbInputBuf=hpoj->inBuffer+hpoj->bufferOffset;
	}
	wResult=ipConvert(hpoj->hJob,
		dwInputAvail,pbInputBuf,&dwInputUsed,&dwInputNextPos,
		dwOutputAvail,pbOutputBuf,&dwOutputUsed,&dwOutputThisPos);
	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_read: "
		"ipConvert(dwInputAvail=%d,pbInputBuf=0x%8.8X,"
		"dwInputUsed=%d,dwInputNextPos=%d,dwOutputAvail=%d,"
		"dwOutputUsed=%d,dwOutputThisPos=%d) returns 0x%4.4X.\n",
		hpoj->saneDevice.name,dwInputAvail,pbInputBuf,
		dwInputUsed,dwInputNextPos,
		dwOutputAvail,dwOutputUsed,dwOutputThisPos,wResult);
	if (hpoj->fdSaveCompressedData>=0 && dwInputUsed>0) {
		write(hpoj->fdSaveCompressedData,pbInputBuf,dwInputUsed);
	}
	hpoj->bufferOffset+=dwInputUsed;
	hpoj->bufferBytesRemaining-=dwInputUsed;
	*pLength=dwOutputUsed;
	if (wResult&(IP_INPUT_ERROR|IP_FATAL_ERROR)) {
		PTAL_LOG_WARN("hpoj:%s: sane_hpoj_read: "
			"ipConvert() returns 0x%4.4X!\n",
			hpoj->saneDevice.name,wResult);
		retcode=SANE_STATUS_IO_ERROR;
		goto abort;
	}
	if (!dwOutputUsed) {
		if (wResult&IP_DONE) {
			retcode=SANE_STATUS_EOF;
			goto abort;
		}
		goto needMoreData;
	}

	retcode=SANE_STATUS_GOOD;
abort:
	if (retcode!=SANE_STATUS_GOOD) {
		*pLength=0;
		if (retcode!=SANE_STATUS_EOF) {
			hpojConnEndScan(hpoj);
		}
		sane_hpoj_cancel(handle);
	}

	PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_read(maxLength=%d) returns %d, "
		"*pLength=%d\n",
		hpoj->saneDevice.name,maxLength,retcode,*pLength);
	return retcode;
}

extern SANE_Status sane_hpoj_set_io_mode(SANE_Handle handle,SANE_Bool nonBlocking) {
	hpojScanner_t hpoj=(hpojScanner_t)handle;

	PTAL_LOG_WARN("hpoj:%s: sane_hpoj_set_io_mode unsupported!\n",
		hpoj->saneDevice.name);

	return SANE_STATUS_UNSUPPORTED;
}

extern SANE_Status sane_hpoj_get_select_fd(SANE_Handle handle,SANE_Int *pFd) {
	hpojScanner_t hpoj=(hpojScanner_t)handle;

	PTAL_LOG_WARN("hpoj:%s: sane_hpoj_get_select_fd unsupported!\n",
		hpoj->saneDevice.name);

	return SANE_STATUS_UNSUPPORTED;
}

static int hpojPmlSelectCallback(ptalChannel_t chan,void *cbd) {
	hpojScanner_t hpoj=(hpojScanner_t)cbd;
	int r=PTAL_OK;

	if (hpoj->pml.scanDone ||
	    ptalPmlRequestGet(hpoj->pml.objUploadState,0)==PTAL_ERROR ||
	    ptalPmlGetIntegerValue(hpoj->pml.objUploadState,0,
	     &hpoj->pml.previousUploadState)==PTAL_ERROR ||
	    (hpoj->pml.previousUploadState==PML_UPLOAD_STATE_ACTIVE &&
	     (!hpoj->preDenali || !hpoj->endOfData))) {
		goto done;
	}

	if (hpoj->pml.previousUploadState==PML_UPLOAD_STATE_NEWPAGE) {
		if (!hpoj->currentBatchScan) {
			if (!hpoj->pml.dontResetBeforeNextNonBatchPage) {
				goto setIdle;
			}

		} else if (hpoj->pml.startNextBatchPageEarly) {
			PTAL_LOG_DEBUG("hpoj:%s: hpojPmlSelectCallback: "
				"restarting scan early.\n",
				hpoj->saneDevice.name);
			ptalPmlSetIntegerValue(hpoj->pml.objUploadState,
				PTAL_PML_TYPE_ENUMERATION,
				PML_UPLOAD_STATE_START);
			ptalPmlRequestSetRetry(hpoj->pml.objUploadState,0,0);
			hpoj->pml.alreadyRestarted=1;
		}
	} else {
setIdle:
		hpojResetScanner(hpoj);
	}

	hpoj->pml.scanDone=1;
done:
	if (hpoj->pml.scanDone && hpoj->endOfData) {
		r=PTAL_ERROR;
	}
	PTAL_LOG_DEBUG("hpoj:%s: hpojPmlSelectCallback returns %d, "
		"scanDone=%d, endOfData=%d, alreadyRestarted=%d.\n",
		hpoj->saneDevice.name,r,hpoj->pml.scanDone,hpoj->endOfData,
		hpoj->pml.alreadyRestarted);
	return r;
}
