/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "nl-drag-server.h"

#include "nl-drag-dest.h"

#include "nl-defines.h"

G_DEFINE_TYPE (NlDragServer, nl_drag_server, G_TYPE_OBJECT);

#define NL_DRAG_SERVER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  CLUTTER_TYPE_DRAG_SERVER, \
  NlDragServerPrivate))

struct _NlDragServerPrivate
{
  GList *dests;

  ClutterActor *context;
  gpointer data;
  gint startx;
  gint starty;

  guint drag_tag;
  guint drop_tag;
};

enum
{
  DRAG_STARTED,
  DRAG_FINISHED,

  LAST_SIGNAL
};
static guint _server_signals[LAST_SIGNAL] = { 0 };

void
nl_drag_server_add_drag_dest (NlDragServer *server,
                              NlDragDest   *dest)
{
  NlDragServerPrivate *priv;

  g_return_if_fail (CLUTTER_IS_DRAG_SERVER (server));
  g_return_if_fail (CLUTTER_IS_DRAG_DEST (dest));
  priv = server->priv;

  priv->dests = g_list_append (priv->dests, dest);
}

static gboolean
on_drag (ClutterActor       *stage,
         ClutterMotionEvent *event,
         NlDragServer  *server)
{
  NlDragServerPrivate *priv;

  g_return_val_if_fail (CLUTTER_IS_DRAG_SERVER (server), FALSE);
  priv = server->priv;

  if (!priv->context)
    {
      return FALSE;
    }

  if (event->modifier_state & CLUTTER_BUTTON1_MASK)
    {
      clutter_actor_set_position (priv->context, event->x, event->y);
    }
  else
    {
      clutter_actor_destroy (priv->context);
      priv->context = NULL;
      clutter_ungrab_pointer ();
      g_signal_handler_disconnect (stage, priv->drag_tag);
      g_signal_handler_disconnect (stage, priv->drop_tag);

      g_signal_emit (server, _server_signals[DRAG_FINISHED], 0);
    }

  return TRUE;
}


void
nl_drag_server_remove_drag_dest (NlDragServer *server,
                                 NlDragDest   *dest)
{
  NlDragServerPrivate *priv;

  g_return_if_fail (CLUTTER_IS_DRAG_SERVER (server));
  g_return_if_fail (CLUTTER_IS_DRAG_DEST (dest));
  priv = server->priv;

  priv->dests = g_list_remove (priv->dests, dest);
}


static void
finish_drop (ClutterActor *context, ClutterAnimation *old_anim)
{
  ClutterAnimation *anim =
    clutter_actor_animate (context, CLUTTER_EASE_OUT_SINE, SLOW_TIME,
                           "scale-x", 0.3,
                           "scale-y", 0.3,
                           "opacity", 0,
                           NULL);
  g_signal_connect_swapped (anim, "completed",
                            G_CALLBACK (clutter_actor_destroy), context);
}

static gboolean
on_drop (ClutterActor       *stage,
         ClutterButtonEvent *event,
         NlDragServer  *server)
{
  NlDragServerPrivate *priv;
  ClutterActor *dest = NULL;
  GList *d;

  g_return_val_if_fail (CLUTTER_IS_DRAG_SERVER (server), FALSE);
  priv = server->priv;

  if (!priv->context)
    {
      g_debug (G_STRLOC " - No context");
      return FALSE;
    }

  g_debug ("Dropped : %f %f\n", event->x, event->y);

  clutter_actor_set_position (priv->context, event->x, event->y);

  dest = clutter_stage_get_actor_at_pos (CLUTTER_STAGE (stage), TRUE,
                                         event->x, event->y);

  for (d = priv->dests; d; d = d->next)
    {
      ClutterActor *actor = d->data;
      gfloat x = 0, y = 0;
      gfloat width = 0, height = 0;

      if (!CLUTTER_IS_ACTOR (actor))
        continue;

      clutter_actor_get_transformed_position (actor, &x, &y);
      clutter_actor_get_transformed_size (actor, &width, &height);

      if (event->x > x && event->x < (x + width))
        {
          if (event->y > y && event->y < (y + height))
            {
              dest = actor;
              break;
            }
        }
    }

  if (!CLUTTER_IS_DRAG_DEST (dest))
    {
      ClutterAnimation *anim =
        clutter_actor_animate (priv->context, CLUTTER_EASE_OUT_SINE, SLOW_TIME,
                               "x", priv->startx,
                               "y", priv->starty,
                               "opacity", 0,
                               NULL);
      g_signal_connect_swapped (anim, "completed",
                                G_CALLBACK (clutter_actor_destroy), priv->context);
    }
  else if (!nl_drag_dest_drop (NL_DRAG_DEST (dest), priv->data))
    {
      /* Drop was unsuccessful */
      ClutterAnimation *anim =
        clutter_actor_animate (priv->context, CLUTTER_EASE_OUT_SINE, SLOW_TIME,
                               "x", priv->startx,
                               "y", priv->starty,
                               "opacity", 0,
                               NULL);
      g_signal_connect_swapped (anim, "completed",
                                G_CALLBACK (clutter_actor_destroy), priv->context);
    }
  else
    {
      /* Drop animation */
      ClutterAnimation *anim =
        clutter_actor_animate (priv->context, CLUTTER_EASE_OUT_SINE, SLOW_TIME,
                               "scale-x", 1.1, "scale-y", 1.1,
                               NULL);
      g_signal_connect_swapped (anim, "completed",
                                G_CALLBACK (finish_drop), priv->context);
    }
  priv->context = NULL;
  clutter_ungrab_pointer ();
  g_signal_handler_disconnect (stage, priv->drag_tag);
  g_signal_handler_disconnect (stage, priv->drop_tag);

  g_signal_emit (server, _server_signals[DRAG_FINISHED], 0);
  return TRUE;
}
/*
 * @context is the visual representaion of the source, it will be destroyed
 * by the server. Must be added to the stage before passing to this function.
 * @udata is whats passed to a NlDragDest if the drop is successful.
 */
void
nl_drag_server_begin_drag (NlDragServer *server,
                           ClutterActor      *context,
                           gpointer           udata)
{
  NlDragServerPrivate *priv;
  ClutterActor *stage = clutter_stage_get_default ();

  g_return_if_fail (CLUTTER_IS_DRAG_SERVER (server));
  g_return_if_fail (CLUTTER_IS_ACTOR (context));
  priv = server->priv;

  g_debug ("Begin drag");

  if (CLUTTER_IS_ACTOR (priv->context))
    {
      g_debug (G_STRLOC " - No context");
      nl_drag_server_finish_drag (server);
    }

  priv->context = context;
  priv->data = udata;

  priv->startx = clutter_actor_get_x (context);
  priv->starty = clutter_actor_get_y (context);

  clutter_grab_pointer (stage);
  priv->drag_tag = g_signal_connect (stage, "motion-event",
                                     G_CALLBACK (on_drag), server);
  priv->drop_tag =g_signal_connect (stage, "button-release-event",
                                    G_CALLBACK (on_drop), server);
  g_signal_emit (server, _server_signals[DRAG_STARTED], 0);
}

void
nl_drag_server_finish_drag (NlDragServer *server)
{
  NlDragServerPrivate *priv;
  ClutterActor *stage = clutter_stage_get_default ();

  g_return_if_fail (CLUTTER_IS_DRAG_SERVER (server));
  priv = server->priv;

  clutter_actor_destroy (priv->context);
  priv->context = NULL;

  clutter_ungrab_pointer ();
  g_signal_handler_disconnect (stage, priv->drag_tag);
  g_signal_handler_disconnect (stage, priv->drop_tag);

  g_signal_emit (server, _server_signals[DRAG_FINISHED], 0);
}


/* GObject stuff */
static void
nl_drag_server_finalize (GObject *object)
{
  NlDragServerPrivate *priv;

  priv = NL_DRAG_SERVER_GET_PRIVATE (object);

  G_OBJECT_CLASS (nl_drag_server_parent_class)->finalize (object);
}

static void
nl_drag_server_class_init (NlDragServerClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = nl_drag_server_finalize;

  _server_signals[DRAG_STARTED] =
    g_signal_new ("drag-started",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (NlDragServerClass, drag_started),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  _server_signals[DRAG_FINISHED] =
    g_signal_new ("drag-finished",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (NlDragServerClass, drag_finished),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  g_type_class_add_private (obj_class, sizeof (NlDragServerPrivate));
}

static void
nl_drag_server_init (NlDragServer *server)
{
  NlDragServerPrivate *priv;

  priv = server->priv = NL_DRAG_SERVER_GET_PRIVATE (server);
}

NlDragServer *
nl_drag_server_get_default (void)

{
  static NlDragServer *drag_server = NULL;

  if (!drag_server)
    drag_server = g_object_new (CLUTTER_TYPE_DRAG_SERVER,
                                NULL);

  return drag_server;
}
