/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *             Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */
using Dee;
using Zeitgeist;
using Zeitgeist.Timestamp;
using Config;
using Gee;
using GMenu;

namespace Unity.ApplicationsLens {

  /* Number of 'Apps available for download' to show if no search query is provided AND a filter is active.
     It shouldn't be too high as this may impact lens performance.
   */
  const uint MAX_APP_FOR_DOWNLOAD_FOR_EMPTY_QUERY = 100;

  /* Number of top rated apps to show in 'Apps available for download' if no search query is provided AND NO filter is active. */
  const uint MAX_TOP_RATED_APPS_FOR_EMPTY_QUERY = 12;

  /* Number of "What's new" apps to show in 'Apps available for download' if no search query is provided AND NO filter is active. */
  const uint MAX_WHATS_NEW_APPS_FOR_EMPTY_QUERY = 10;

  /* Time between queries to SoftwareCenterDataProvider */
  const int64 TOP_RATED_ITEMS_CACHE_LIFETIME = 24*3600; // 24 hours

  const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";
  const string GENERIC_APP_ICON = "applications-other";

  public class Daemon : GLib.Object
  {
    private Zeitgeist.Log log;
    private Zeitgeist.Index zg_index;
    private Zeitgeist.Monitor monitor;

    private Map<string, int> popularity_map;
    private bool popularities_dirty;

    /* The searcher for online material may be null if it fails to load
     * the Xapian index from the Software Center */
    private Unity.Package.Searcher? pkgsearcher;
    public Unity.Package.Searcher appsearcher;

    /* Read the app ratings dumped by the Software Center */
    private bool ratings_db_initialized = false;
    private Unity.Ratings.Database? ratings = null;

    private Unity.Lens lens;
    private Unity.Scope scope;

    /* Support aptd dbus interface; created when application install/remove was requested by preview action */
    private AptdProxy aptdclient;
    private AptdTransactionProxy aptd_transaction;

    private SoftwareCenterUtils.MangledDesktopFileLookup sc_mangler;

    /* Used for adding launcher icon on app installation from the preview */
    private LauncherProxy launcherservice;

    /* Desktop file & icon name for unity-install:// install candidate displayed in last preview; we store
     * them here to avoid extra query for app details if app install action is activated */
    private string preview_installable_desktop_file;
    private string preview_installable_icon_file;

    private string preview_developer_website;

    /* SoftwareCenter data provider used for app preview details */
    private SoftwareCenterDataProviderProxy sc_data_provider;

    private Unity.ApplicationsLens.Runner runner;

    /* Keep references to the FilterOptions for sources */
    private FilterOption local_apps_option;
    private FilterOption usc_apps_option;

    private Gee.List<string> image_extensions;
    private HashTable<string,Icon> file_icon_cache;

    /* Monitor the favorite apps in the launcher, so we can filter them
     * out of the results for Recent Apps */
    private Unity.LauncherFavorites favorite_apps;
    private AppWatcher app_watcher;

    private PtrArray zg_templates;

    /* Gnome menu structure - also used to check whether apps are installed */
    private uint app_menu_changed_reindex_timeout = 0;
    private GMenu.Tree app_menu = null;

    private Regex? uri_regex;
    private Regex  mountable_regex;

    private Settings gp_settings;

    private const string DISPLAY_RECENT_APPS_KEY = "display-recent-apps";
    private const string DISPLAY_AVAILABLE_APPS_KEY = "display-available-apps";

    public bool display_recent_apps { get; set; default = true; }
    public bool display_available_apps { get; set; default = true; }
    public bool force_small_icons_for_suggestions { get; set; default = true; }

    private PurchaseInfoHelper purchase_info = null;

    construct
    {
      populate_zg_templates ();

      log = new Zeitgeist.Log();
      zg_index = new Zeitgeist.Index();
      monitor = new Zeitgeist.Monitor (new Zeitgeist.TimeRange.from_now (),
                                       zg_templates);
      monitor.events_inserted.connect (mark_dirty);
      monitor.events_deleted.connect (mark_dirty);
      log.install_monitor (monitor);

      popularity_map = new HashMap<string, int> ();
      popularities_dirty = true;
      // refresh the popularities every now and then
      Timeout.add_seconds (30 * 60, () =>
      {
        popularities_dirty = true;
        return true;
      });

      this.gp_settings = new Settings ("com.canonical.Unity.ApplicationsLens");
      this.gp_settings.bind(DISPLAY_RECENT_APPS_KEY, this, "display_recent_apps", SettingsBindFlags.GET);
      this.gp_settings.bind(DISPLAY_AVAILABLE_APPS_KEY, this, "display_available_apps", SettingsBindFlags.GET);

      pkgsearcher = new Unity.Package.Searcher ();
      if (pkgsearcher == null)
      {
        critical ("Failed to load Software Center index. 'Apps Available for Download' will not be listed");
      }

      /* Image file extensions in order of popularity */
      image_extensions = new Gee.ArrayList<string> ();
      image_extensions.add ("png");
      image_extensions.add ("xpm");
      image_extensions.add ("svg");
      image_extensions.add ("tiff");
      image_extensions.add ("ico");
      image_extensions.add ("tif");
      image_extensions.add ("jpg");

      build_app_menu_index ();

      file_icon_cache = new HashTable<string,Icon>(str_hash, str_equal);
      sc_mangler = new SoftwareCenterUtils.MangledDesktopFileLookup ();

      scope = new Unity.Scope ("/com/canonical/unity/scope/applications");
      scope.provides_personal_content = true;
      //scope.icon = @"$(Config.PREFIX)/share/unity/themes/applications.png";

      // TRANSLATORS: Please make sure this string is short enough to fit
      // into the filter button
      local_apps_option = scope.sources.add_option ("local", _("Local Apps"));
      if (display_available_apps)
      {
        // TRANSLATORS: Please make sure this string is short enough to fit
        // into the filter button
        usc_apps_option = scope.sources.add_option ("usc", _("Software Center"));
      }

      scope.generate_search_key.connect ((lens_search) =>
      {
        return lens_search.search_string.strip ();
      });
      /* Listen for changes to the lens scope search */
      scope.search_changed.connect ((lens_search, search_type, cancellable) =>
      {
        dispatch_search (lens_search, search_type, cancellable);
      });

      /* Re-do the search if the filters changed */
      scope.filters_changed.connect (() =>
      {
        scope.queue_search_changed (SearchType.DEFAULT);
      });

      /* And also if the sources change */
      scope.active_sources_changed.connect (() =>
      {
        scope.queue_search_changed (SearchType.DEFAULT);
      });

      scope.activate_uri.connect (activate);
      scope.preview_uri.connect (preview);

      /* Listen for changes in the installed applications */
      AppInfoManager.get_default().changed.connect (mark_dirty);

      /* Now start the RunEntry */
      runner = new Unity.ApplicationsLens.Runner (this);

      try {
        uri_regex = new Regex ("^[a-z]+:.+$");
        mountable_regex = new Regex ("((ftp|ssh|sftp|smb|dav)://).+");
      } catch (RegexError e) {
        uri_regex = null;
        critical ("Failed to compile URI regex. URL launching will be disabled");
      }

      favorite_apps = Unity.LauncherFavorites.get_default ();
      favorite_apps.changed.connect(mark_dirty);

      app_watcher = new AppWatcher ();
      app_watcher.running_applications_changed.connect (mark_dirty);

      aptdclient = new AptdProxy ();
      launcherservice = new LauncherProxy ();

      lens = new Unity.Lens ("/com/canonical/unity/lens/applications", "applications");
      lens.search_hint = _("Search Applications");
      lens.visible = true;
      lens.search_in_global = true;
      lens.sources_display_name = _("Sources");
      populate_categories ();
      populate_filters();
      lens.add_local_scope (scope);
      lens.export ();
    }

    private void init_ratings_db ()
    {
      if (ratings_db_initialized) return;
      try
      {
        ratings = new Unity.Ratings.Database ();
      }
      catch (FileError e)
      {
        warning ("%s", e.message);
        ratings = null;
      }
      ratings_db_initialized = true;
    }

    private async void dispatch_search (LensSearch lens_search,
                                        SearchType search_type,
                                        Cancellable cancellable)
    {
      if (popularities_dirty)
      {
        popularities_dirty = false;
        // we're not passing the cancellable, cause cancelling this search
        // shouldn't cancel getting most popular apps
        yield update_popularities ();
        if (cancellable.is_cancelled ()) return;
      }

      if (search_type == SearchType.DEFAULT)
        yield update_scope_search (lens_search, cancellable);
      else
        yield update_global_search (lens_search, cancellable);
    }

    private void populate_categories ()
    {
      GLib.List<Unity.Category> categories = new GLib.List<Unity.Category> ();
      File icon_dir = File.new_for_path (ICON_PATH);

      var cat = new Unity.Category (_("Applications"),
                                    new FileIcon (icon_dir.get_child ("group-apps.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("Recently Used"),
                                new FileIcon (icon_dir.get_child ("group-recent.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("Recent Apps"),
                                new FileIcon (icon_dir.get_child ("group-apps.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("Installed"),
                                new FileIcon (icon_dir.get_child ("group-installed.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("More suggestions"),
                                new FileIcon (icon_dir.get_child ("group-treat-yourself.svg")),
                                Unity.CategoryRenderer.FLOW);
      categories.append (cat);

      lens.categories = categories;
    }

    private void populate_filters()
    {
      GLib.List<Unity.Filter> filters = new GLib.List<Unity.Filter> ();

      /* Type filter */
      {
        var filter = new CheckOptionFilter ("type", _("Type"));
        filter.sort_type = Unity.OptionsFilter.SortType.DISPLAY_NAME;

        filter.add_option ("accessories", _("Accessories"));
        filter.add_option ("education", _("Education"));
        filter.add_option ("game", _("Games"));
        filter.add_option ("graphics", _("Graphics"));
        filter.add_option ("internet", _("Internet"));
        filter.add_option ("fonts", _("Fonts"));
        filter.add_option ("office", _("Office"));
        filter.add_option ("media", _("Media"));
        filter.add_option ("customization", _("Customization"));
        filter.add_option ("accessibility", _("Accessibility"));
        filter.add_option ("developer", _("Developer"));
        filter.add_option ("science-and-engineering", _("Science & Engineering"));
        filter.add_option ("system", _("System"));

        filters.append (filter);
      }

      lens.filters = filters;
    }

    private bool local_apps_active ()
    {
      if (scope.sources.filtering && local_apps_option != null)
        return local_apps_option.active;
      return true;
    }

    private bool usc_apps_active ()
    {
      if (scope.sources.filtering && usc_apps_option != null)
        return usc_apps_option.active;
      return true;
    }

    /* Load xdg menu info and build a Xapian index over it.
     * Do throttled re-index if the menu changes */
    private bool build_app_menu_index ()
    {
      if (app_menu == null)
      {
        debug ("Building initial application menu");

        /* We need INCLUDE_NODISPLAY to employ proper de-duping between
         * the Installed and Availabale categorys. If a NoDisplay app is installed,
         * eg. Evince, it wont otherwise be in the menu index, only in the
         * S-C index - thus show up in the Available category */
        app_menu = new GMenu.Tree ("unity-lens-applications.menu",
                                        GMenu.TreeFlags.INCLUDE_NODISPLAY);

        app_menu.changed.connect (() => {
            /* Reschedule the timeout if we already have one. The menu tree triggers
             * many change events during app installation. This way we wait the full
             * delay *after* the last change event has triggered */
            if (app_menu_changed_reindex_timeout != 0)
            {
              Source.remove (app_menu_changed_reindex_timeout);
            }

            app_menu_changed_reindex_timeout = Timeout.add_seconds (5, build_app_menu_index_and_result_models);
          });
      }

      // gnome menu tree needs to be loaded on startup and after receiving 'changed' signal - see gmenu-tree.c in gnome-menus-3.6.0.
      try
      {
        app_menu.load_sync (); //FIXME: libgnome-menu doesn't provide async method (yet?)
      }
      catch (GLib.Error e)
      {
        warning ("Failed to load menu entries: %s", e.message);
      }

      debug ("Indexing application menu");
      appsearcher = new Unity.Package.Searcher.for_menu (app_menu);
      app_menu_changed_reindex_timeout = 0;

      return false;
    }

    /* Called when our app_menu structure changes - probably because something
     * has been installed or removed. We reload gnome menu tree,
     * rebuild the index and update the result models for global and scope.
     * We need to update both because
     * we can't know exactly what Unity may be showing */
    private bool build_app_menu_index_and_result_models ()
    {
      build_app_menu_index ();

      mark_dirty ();

      return false;
    }

    private void populate_zg_templates ()
    {
      /* Create a template that activation of applications */
      zg_templates = new PtrArray.sized(1);
      var ev = new Zeitgeist.Event.full (ZG_ACCESS_EVENT, ZG_USER_ACTIVITY, "",
                             new Subject.full ("application*",
                                               "", //NFO_SOFTWARE,
                                               "",
                                               "", "", "", ""));
      zg_templates.add ((ev as GLib.Object).ref());
    }

    private void mark_dirty ()
    {
      scope.queue_search_changed (SearchType.DEFAULT);
      scope.queue_search_changed (SearchType.GLOBAL);
    }

    private async void update_popularities ()
    {
      try
      {
        // simulate a kind of frecency
        var end_ts = Timestamp.now ();
        var start_ts = end_ts - Timestamp.WEEK * 3;
        var rs = yield log.find_events (new TimeRange (start_ts, end_ts),
                                        zg_templates,
                                        StorageState.ANY,
                                        256,
                                        ResultType.MOST_POPULAR_SUBJECTS,
                                        null);

        // most popular apps must have high value, so unknown apps (where
        // popularity_map[app] == 0 aren't considered super popular
        int relevancy = 256;
        foreach (unowned Event event in rs)
        {
          for (int i = 0; i < event.num_subjects (); i++)
          {
            unowned string uri = event.get_subject (i).get_uri ();
            if (uri == null) continue;
            popularity_map[uri] = relevancy;
          }
          relevancy--;
        }
      }
      catch (GLib.Error err)
      {
        warning ("%s", err.message);
      }
    }

    /* Returns TRUE if application is NOT installed */
    public bool filter_cb (Unity.Package.PackageInfo pkginfo)
    {
      var appmanager = AppInfoManager.get_default();
      AppInfo? app = appmanager.lookup (pkginfo.desktop_file);
      return app == null;
    }

    private async void update_scope_search (Unity.LensSearch search,
                                            Cancellable cancellable)
    {
      var model = search.results_model;
      /* We'll clear the model once we finish waiting for the dbus-call
       * to finish, to prevent flicker. */

      debug ("Searching for: %s", search.search_string);

      var filter = scope.get_filter ("type") as OptionsFilter;

      string pkg_search_string = XapianUtils.prepare_pkg_search_string (search.search_string, filter);

      bool has_filter = (filter != null && filter.filtering);
      bool has_search = !Utils.is_search_empty (search.search_string);

      Timer timer = new Timer ();

      var transaction = new Dee.Transaction (model);
      transaction.clear ();

      /* Even though the Installed apps search is super fast, we wait here
       * for the Most Popular search to finish, because otherwise we'll update
       * the Installed category too soon and this will cause flicker
       * in the Dash. (lp:868192) */

      Set<string> installed_uris = new HashSet<string> ();
      Set<string> available_uris = new HashSet<string> ();
      var appresults = appsearcher.search (pkg_search_string, 0,
                                           Unity.Package.SearchType.PREFIX,
                                           has_search ?
                                              Unity.Package.Sort.BY_RELEVANCY :
                                              Unity.Package.Sort.BY_NAME);
      if (local_apps_active ())
      {
        if (has_search) resort_pkg_search_results (appresults);
        add_pkg_search_result (appresults, installed_uris, available_uris,
                               transaction, Category.INSTALLED);
      }

      timer.stop ();
      debug ("Entry search listed %i Installed apps in %fms for query: %s",
             appresults.num_hits, timer.elapsed ()*1000, pkg_search_string);

      if (local_apps_active () && display_recent_apps)
      {
        try
        {
          timer.start ();
          /* Ignore the search string, we want to keep displaying the same apps
           * in the recent category and just filter out those that don't match
           * the search query */
          var zg_search_string = XapianUtils.prepare_zg_search_string ("", filter);

          var results = yield zg_index.search (zg_search_string,
                                               new Zeitgeist.TimeRange.anytime(),
                                               zg_templates,
                                               0,
                                               20,
                                               Zeitgeist.ResultType.MOST_RECENT_SUBJECTS,
                                               cancellable);

          append_events_with_category (results, transaction, Category.RECENT,
                                       false, 6, installed_uris);

          timer.stop ();
          debug ("Entry search found %u/%u Recently Used apps in %fms for query '%s'",
                 results.size (), results.estimated_matches (),
                 timer.elapsed()*1000, zg_search_string);

        } catch (IOError.CANCELLED ioe) {
          // no need to bother
          return;
        } catch (GLib.Error e) {
          warning ("Error performing search '%s': %s", search.search_string, e.message);
        }
      }

      transaction.commit ();

      purchase_info = new PurchaseInfoHelper ();

      /* If we don't have a search we display 6 random apps */
      if (usc_apps_active () && display_available_apps && pkgsearcher != null)
      {
        if (has_search)
        {
          timer.start ();
          var pkgresults = pkgsearcher.search (pkg_search_string, 50,
                                               Unity.Package.SearchType.PREFIX,
                                               Unity.Package.Sort.BY_RELEVANCY);
          add_pkg_search_result (pkgresults, installed_uris, available_uris,
                                 model, Category.AVAILABLE);
          timer.stop ();
          debug ("Entry search listed %i Available apps in %fms for query: %s",
                 pkgresults.num_hits, timer.elapsed ()*1000, pkg_search_string);
        }
        else if (has_filter) /* Empty search string + active filters should get lots of results from selected categories */
        {
          timer.start ();
          string? filter_query = XapianUtils.prepare_pkg_search_string (search.search_string, filter);

          var pkgresults = pkgsearcher.get_apps (filter_query, MAX_APP_FOR_DOWNLOAD_FOR_EMPTY_QUERY, filter_cb);
          purchase_info.from_pkgresults (pkgresults);
          add_pkg_search_result (pkgresults, installed_uris, available_uris, model, Category.AVAILABLE, MAX_APP_FOR_DOWNLOAD_FOR_EMPTY_QUERY);
          timer.stop ();
          debug ("Entry search listed %i Available apps in %fms",
                 pkgresults.num_hits, timer.elapsed ()*1000);
        }
        else
        {
          timer.start ();

          uint hits = 0;
          try
          {
            Set<string> duplicates_lookup = new HashSet<string> ();

            if (sc_data_provider == null)
              sc_data_provider = new SoftwareCenterDataCache (TOP_RATED_ITEMS_CACHE_LIFETIME);

            var whats_new = sc_data_provider.get_items_for_category ("unity-whats-new");
            var query = purchase_info.create_pkgsearch_query (whats_new);
            var tmpresults = pkgsearcher.get_by_exact_names (query);
            purchase_info.from_pkgresults (tmpresults);
            hits = add_sc_category_results (whats_new, model, Category.AVAILABLE, ref duplicates_lookup, MAX_WHATS_NEW_APPS_FOR_EMPTY_QUERY);

            var top_rated = sc_data_provider.get_items_for_category ("unity-top-rated");
            query = purchase_info.create_pkgsearch_query (top_rated);
            tmpresults = pkgsearcher.get_by_exact_names (query);
            purchase_info.from_pkgresults (tmpresults);
            hits += add_sc_category_results (top_rated, model, Category.AVAILABLE, ref duplicates_lookup, MAX_TOP_RATED_APPS_FOR_EMPTY_QUERY);
          }
          catch (GLib.Error e)
          {
            warning ("Failed to get top rated apps: %s", e.message);
          }

          timer.stop ();
          debug ("Entry search listed %u top rated/new available apps in %fms",
                 hits, timer.elapsed ()*1000);
        }
      }

      if (model.get_n_rows () == 0)
      {
        search.set_reply_hint ("no-results-hint",
          _("Sorry, there are no applications that match your search."));
      }

      search.finished ();
    }

    private async void update_global_search (Unity.LensSearch search,
                                             Cancellable cancellable)
    {
      /*
       * In global search, with a non-empty search string, we collate all
       * hits under one Applications category
       */

      if (Utils.is_search_empty (search.search_string))
      {
        yield update_global_without_search (search, cancellable);
        return;
      }

      var model = search.results_model;

      model.clear ();

      var search_string = XapianUtils.prepare_pkg_search_string (search.search_string, null);
      Set<string> installed_uris = new HashSet<string> ();
      Set<string> available_uris = new HashSet<string> ();
      var timer = new Timer ();
      var appresults = appsearcher.search (search_string, 0,
                                           Unity.Package.SearchType.PREFIX,
                                           Unity.Package.Sort.BY_RELEVANCY);
      resort_pkg_search_results (appresults);
      add_pkg_search_result (appresults, installed_uris, available_uris, model,
                             Category.APPLICATIONS);

      timer.stop ();
      debug ("Global search listed %i Installed apps in %fms for query: %s",
             appresults.num_hits, timer.elapsed ()*1000, search_string);

      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      search.finished ();
    }

    private async void update_global_without_search (Unity.LensSearch search,
                                                     Cancellable cancellable)
    {
      /*
       * In global search, with an empty search string, we show just Recent Apps
       * Excluding apps with icons in the launcher (be they running or faves)
       */
      var model = search.results_model;

      Timer timer = new Timer ();

      if (local_apps_active () && display_recent_apps)
      {
        try
        {
          var zg_search_string = XapianUtils.prepare_zg_search_string (search.search_string,
                                                           null);

          var time_range = new Zeitgeist.TimeRange.anytime ();
          var results = yield log.find_events (time_range,
                                               zg_templates,
                                               Zeitgeist.StorageState.ANY,
                                               40,
                                               Zeitgeist.ResultType.MOST_RECENT_SUBJECTS,
                                               cancellable);

          model.clear ();
          append_events_with_category (results, model, Category.RECENT_APPS,
                                       false);

          timer.stop ();
          debug ("Entry search found %u/%u Recently Used apps in %fms for query '%s'",
                 results.size (), results.estimated_matches (),
                 timer.elapsed()*1000, zg_search_string);

        } catch (IOError.CANCELLED ioe) {
          // no need to bother
          return;
        } catch (GLib.Error e) {
          warning ("Error performing search '%s': %s",
                   search.search_string, e.message);
        }
      }

      search.finished ();
    }

    public Icon find_pkg_icon (string? desktop_file, string icon_name)
    {
      if (desktop_file != null)
      {
        string desktop_id = Path.get_basename (desktop_file);
        bool installed = AppInfoManager.get_default().lookup (desktop_id) != null;

        /* If the app is already installed we should be able to pull the
         * icon from the theme */
        if (installed)
          return new ThemedIcon (icon_name);
      }

      /* App is not installed - we need to find the right icon in the bowels
       * of the software center */
      if (icon_name.has_prefix ("/"))
      {
        return new FileIcon (File.new_for_path (icon_name));
      }
      else
      {
        Icon icon = file_icon_cache.lookup (icon_name);

        if (icon != null)
          return icon;

        /* If the icon name contains a . it probably already have a
         * type postfix - so test icon name directly */
        string path;
        if ("." in icon_name)
        {
          path = @"$(Config.DATADIR)/app-install/icons/$(icon_name)";
          if (FileUtils.test (path, FileTest.EXISTS))
          {
            icon = new FileIcon (File.new_for_path (path));
            file_icon_cache.insert (icon_name, icon);
            return icon;
          }
          /* Try also software center cache dir */
          path = Path.build_filename (Environment.get_user_cache_dir (),
                                      "software-center",
                                      "icons",
                                      icon_name);
          if (FileUtils.test (path, FileTest.EXISTS))
          {
            icon = new FileIcon (File.new_for_path (path));
            file_icon_cache.insert (icon_name, icon);
            return icon;
          }
        }

        /* Now try appending all the image extensions we know */
        foreach (var ext in image_extensions)
        {
          path = @"$(Config.DATADIR)/app-install/icons/$(icon_name).$(ext)";
          if (FileUtils.test (path, FileTest.EXISTS))
          {
            /* Got it! Cache the icon path and return the icon */
            icon = new FileIcon (File.new_for_path (path));
            file_icon_cache.insert (icon_name, icon);
            return icon;
          }
        }
      }

      /* Cache the fact that we couldn't find this icon */
      var icon = new ThemedIcon (GENERIC_APP_ICON);
      file_icon_cache.insert (icon_name, icon);

      return icon;
    }

    /*
     * Performs secondary level sorting of the results according to popularity
     * of individual desktop files.
     */
    private void resort_pkg_search_results (Unity.Package.SearchResult results)
    {
      results.results.sort_with_data ((a, b) =>
      {
        /* We'll cluster the relevancies into a couple of bins, because
         * there can be multiple terms adding to the relevancy (especially
         * when doing one/two character prefix searches - ie a "f*" search
         * will have slightly higher relevancy for item with name "Search
         * _f_or _f_iles" than "_F_irefox", and that's not what we want)
         */
        int rel_a = a.relevancy;
        int rel_b = b.relevancy;
        int delta = (rel_a - rel_b).abs ();
        if (delta < 10)
        {
          string id_a = sc_mangler.extract_desktop_id (a.desktop_file);
          string id_b = sc_mangler.extract_desktop_id (b.desktop_file);
          rel_a = popularity_map["application://" + id_a];
          rel_b = popularity_map["application://" + id_b];
        }
        return rel_b - rel_a; // we want higher relevancy first
      });
    }

    /**
     * Sanitize executable name -- make it suitable for Home Lens.
     */
    private static string sanitize_binary_name (string name)
    {
      return GLib.Path.get_basename (name);
    }

    private string get_annotated_icon (Icon app_icon, string price, bool paid,
                                       bool use_small_icon = true)
    {
      var annotated_icon = new AnnotatedIcon (app_icon);
      annotated_icon.category = CategoryType.APPLICATION;
      if (price != null && price != "")
      {
        if (paid)
          annotated_icon.ribbon = _("Paid");
        else
          annotated_icon.ribbon = price;
      }
      else
      {
        annotated_icon.ribbon = _("Free");
      }
      if (force_small_icons_for_suggestions || use_small_icon
          || app_icon.to_string () == GENERIC_APP_ICON)
      {
        annotated_icon.size_hint = IconSizeHint.SMALL;
      }

      return annotated_icon.to_string ();
    }

    /**
     * Add all results obtained from SoftwareCenterDataProvider
     */
    private uint add_sc_category_results (SoftwareCenterData.AppInfo?[] results,
                                          Dee.Model model,
                                          Category category,
                                          ref Set<string> duplicates_lookup,
                                          uint max_results)
    {
      uint i = 0;
      foreach (SoftwareCenterData.AppInfo app in results)
      {
        string uri = @"unity-install://$(app.package_name)/$(app.application_name)";
        if (uri in duplicates_lookup)
          continue;

        string icon_obj;
        Icon app_icon = find_pkg_icon (app.desktop_file, app.icon);
        var pinfo = purchase_info.find (app.application_name, app.package_name);
        if (pinfo != null)
        {
          // magazines need to use large icons
          bool use_small_icon = app.desktop_file.has_suffix (".desktop");
          var annotated_icon = get_annotated_icon (app_icon,
                                                   pinfo.formatted_price,
                                                   pinfo.paid,
                                                   use_small_icon);
          icon_obj = annotated_icon.to_string ();
        }
        else
        {
          warning ("No purchase info for: %s, %s", app.application_name, app.package_name);
          icon_obj = app_icon.to_string ();
        }

        model.append (uri, icon_obj,
                      category,
                      "application/x-desktop",
                      app.application_name,
                      "", //comment
                      "file://" + app.desktop_file);
        duplicates_lookup.add (uri);
        i++;
        if (i == max_results)
          break;
      }
      return i;
    }

    private void add_pkg_search_result (Unity.Package.SearchResult results,
                                        Set<string> installed_uris,
                                        Set<string> available_uris,
                                        Dee.Model model,
                                        Category category,
                                        uint max_add=0)
    {
      var appmanager = AppInfoManager.get_default();
      uint n_added = 0;

      foreach (unowned Unity.Package.PackageInfo pkginfo in results.results)
      {
        if (pkginfo.desktop_file == null)
          continue;

        string desktop_id = sc_mangler.extract_desktop_id (pkginfo.desktop_file,
                                                category == Category.AVAILABLE);
        string full_path;

        AppInfo? app = appmanager.lookup (desktop_id);
        full_path = appmanager.get_path (desktop_id);

        /* De-dupe by 'application://foo.desktop' URI. Also note that we need
         * to de-dupe before we chuck out NoDisplay app infos, otherwise they'd
         * show up from alternate sources */
        string uri = @"application://$(desktop_id)";
        if (uri in installed_uris || uri in available_uris)
          continue;

        /* Extract basic metadata and register de-dupe keys */
        string display_name;
        string comment;
        switch (category)
        {
          case Category.INSTALLED:
          case Category.APPLICATIONS:
            installed_uris.add (uri);
            display_name = app.get_display_name ();
            comment = sanitize_binary_name (app.get_executable ());
            break;
          case Category.AVAILABLE:
            available_uris.add (uri);
            display_name = pkginfo.application_name;
            comment = "";
            break;
          default:
            warning (@"Illegal category for package search $(category)");
            continue;
        }

        /* We can only chuck out NoDisplay and OnlyShowIn app infos after
         * we have registered a de-dupe key for them - which is done in the
         * switch block above) */
        if (app != null && !app.should_show ())
          continue;

        Icon icon = find_pkg_icon (pkginfo.desktop_file, pkginfo.icon);
        string icon_str;
        if (category == Category.AVAILABLE)
        {
          /* If we have an available item, which is not a dupe, but is
           * installed anyway, we weed it out here, because it's probably
           * left out from the Installed section because of some rule in the
           * .menu file */
          if (app != null)
            continue;

          /* Apps that are not installed, ie. in the Available category
           * use the 'unity-install://pkgname/Full App Name' URI scheme,
           * but only use that after we've de-duped the results.
           * But only change the URI *after* we've de-duped the results! */
          uri = @"unity-install://$(pkginfo.package_name)/$(pkginfo.application_name)";
          available_uris.add (uri);

          // magazines need to use large icons
          bool use_small_icon = pkginfo.desktop_file.has_suffix (".desktop");
          icon_str = get_annotated_icon (icon, pkginfo.price,
                                         !pkginfo.needs_purchase,
                                         use_small_icon);
        }
        else
        {
          icon_str = icon.to_string ();
        }

        model.append (uri, icon_str,
                      category,"application/x-desktop",
                      display_name != null ? display_name : "",
                      comment != null ? comment : "",
                      full_path != null ? "file://" + full_path : "");

        /* Stop if we added the number of items requested */
        n_added++;
        if (max_add > 0 && n_added >= max_add)
          return;
      }
    }

    private async void call_install_packages (string package_name, out string tid) throws IOError
    {
      tid = yield aptdclient.install_packages ({package_name});
    }

    private async void call_remove_packages (string package_name, out string tid) throws IOError
    {
      tid = yield aptdclient.remove_packages ({package_name});
    }

    /**
     * Handler for free apps installation.
     * Triggers package installation via apt-daemon DBus service
     */
    private Unity.ActivationResponse app_preview_install (string uri)
    {
      if (uri.has_prefix ("unity-install://"))
      {
        string app = uri.substring (16); // trim "unity-install://"
        string[] parts = app.split ("/");

        if (parts.length > 1)
        {
          string pkgname = parts[0];
          string appname = parts[1];
          try
          {
            aptdclient.connect_to_aptd ();
          }
          catch (IOError e)
          {
            warning ("Failed to connect to aptd: '%s'", e.message);
            return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
          }
          call_install_packages.begin (pkgname, (obj, res) =>
          {
            try {
              string tid;
              call_install_packages.end (res, out tid);
              debug ("transaction started: %s, pkg: %s\n", tid, pkgname);
              aptd_transaction = new AptdTransactionProxy ();
              aptd_transaction.connect_to_aptd (tid);
              aptd_transaction.simulate ();
              aptd_transaction.run ();

              launcherservice.connect_to_launcher ();
              string desktop_file = preview_installable_desktop_file;
              Icon icon = find_pkg_icon (null, preview_installable_icon_file);
              launcherservice.add_launcher_item_from_position (appname, icon.to_string (), 0, 0, 32, desktop_file, tid);
            }
            catch (IOError e)
            {
              warning ("Package '%s' installation failed: %s", pkgname, e.message);
            }
          });
        }
        else
        {
          warning ("Bad install uri: '%s'", uri);
        }
      }
      else
      {
        warning ("Can't handle '%s' in app_preview_install handler", uri);
        return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
      }
      return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);
    }

    private Unity.ActivationResponse app_preview_install_commercial (string uri)
    {
      return activate (uri); //this will just launch Software-Center
    }

    private Unity.ActivationResponse app_preview_buy (string uri)
    {
      return activate (uri); //this will just launch Software-Center
    }

    private Unity.ActivationResponse app_preview_website (string uri)
    {
      try
      {
        AppInfo.launch_default_for_uri (preview_developer_website, null);
        return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);
      }
      catch (Error e)
      {
        warning ("Failed to launch a web browser for uri '%s': '%s'", uri, e.message);
      }
      return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
    }

    private Unity.ActivationResponse app_preview_uninstall (string uri)
    {
      if (uri.has_prefix ("application://"))
      {
        string desktopfile = uri.substring (14); // trim "application://"

        // de-mangle desktop file names back to what S-C expects
        if (sc_mangler.contains (desktopfile))
        {
          desktopfile = sc_mangler.get (desktopfile);
        }

        var pkginfo = pkgsearcher.get_by_desktop_file (desktopfile);

        if (pkginfo != null && pkginfo.package_name != null)
        {
          try
          {
            aptdclient.connect_to_aptd ();
          }
          catch (IOError e)
          {
            warning ("Failed to connect to aptd: '%s'", e.message);
            return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
          }

          call_remove_packages.begin (pkginfo.package_name, (obj, res) =>
          {
            try {
              string tid;
              call_remove_packages.end (res, out tid);
              debug ("transaction started: %s, pkg: %s\n", tid, pkginfo.package_name);
              aptd_transaction = new AptdTransactionProxy ();
              aptd_transaction.connect_to_aptd (tid);
              aptd_transaction.simulate ();
              aptd_transaction.run ();
            }
            catch (IOError e)
            {
              warning (@"Package '$(pkginfo.package_name)' removal failed: $(e.message)");
            }
          });
          return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);
        }
        else
        {
          warning (@"Cannot find package info for $uri");
        }
      }
      warning (@"Can't handle '%s' in app_preview_uninstall handler", uri);
      return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
    }

    public Unity.Preview preview (string uri)
    {
      Unity.ApplicationPreview? preview = null;

      string pkgname = "";
      string appname = "";
      bool installed = uri.has_prefix ("application://");

      if (installed || uri.has_prefix ("unity-install://"))
      {
        string desktopfile = null;
        if (installed)
        {
          desktopfile = uri.substring (14); //remove "application://" prefix

          // de-mangle desktop file names back to what S-C expects
          if (sc_mangler.contains (desktopfile))
          {
            desktopfile = sc_mangler.get (desktopfile);
          }

          Unity.Package.PackageInfo pkginfo = pkgsearcher.get_by_desktop_file (desktopfile);
          if (pkginfo != null)
          {
            appname = pkginfo.application_name;
            pkgname = pkginfo.package_name;
          }
        }
        else // unity-install
        {
          string app = uri.substring (16); //remove "unity-install://" prefix
          string[] parts = app.split ("/");
          if (parts.length > 1)
          {
            pkgname = parts[0];
            appname = parts[1];
          }
        }

        if (pkgname != "")
        {
          try {
            if (sc_data_provider == null)
            {
              sc_data_provider = new SoftwareCenterDataCache (TOP_RATED_ITEMS_CACHE_LIFETIME);
              sc_data_provider.connect_to ();
            }

            debug ("Requesting pkg info: %s, %s\n", pkgname, appname);
            var details = sc_data_provider.get_app_details (appname, pkgname);

            Icon? icon = null;
            if (installed)
              icon = new GLib.ThemedIcon (details.icon);
            else
            {
              icon = find_pkg_icon (null, details.icon);
              if (icon.to_string () == GENERIC_APP_ICON && details.icon_url != null && details.icon_url != "")
              {
                icon = new GLib.FileIcon (File.new_for_uri (details.icon_url));
              }
            }

            Icon? screenshot = null;

            if (details.screenshot != null)
            {
              File scr_file = File.new_for_uri (details.screenshot);
              screenshot = new FileIcon (scr_file);
            }

            string subtitle = "";
            if (details.version != "")
              subtitle = _("Version %s").printf (details.version);
            if (details.size > 0)
            {
              if (subtitle != "")
                subtitle += ", ";
              subtitle += ("Size %s").printf (GLib.format_size (details.size));
            }
            preview = new Unity.ApplicationPreview (details.name, subtitle, details.description, icon, screenshot);
            preview.license = details.license;

            init_ratings_db ();
            if (ratings != null)
            {
              Unity.Ratings.Result result;
              ratings.query (pkgname, out result);
              preview.set_rating (result.average_rating / 5.0f, result.total_rating);
            }

            if (details.hardware_requirements != "")
              preview.add_info (new InfoHint ("hardware-requirements", _("Hardware requirements"), null, details.hardware_requirements));

            if (uri.has_prefix ("unity-install://")) // application needs to be purchased/installed
            {
              // uninstalled and not purchased before
              if (details.pkg_state == SoftwareCenterData.PackageState.NEEDS_PURCHASE)
              {
                var buy_action = new Unity.PreviewAction ("buy", _("Buy"), null);
                if (details.price != null && details.price != "")
                {
                  buy_action.extra_text = details.price;
                }

                buy_action.activated.connect (app_preview_buy);
                preview.add_action (buy_action);
              }
              else // uninstalled, purchased before
              {

                Unity.PreviewAction install_action = null;
                if (details.raw_price == null || details.raw_price == "")
                {
                  install_action = new Unity.PreviewAction ("install", _("Free Download"), null);
                  install_action.activated.connect (app_preview_install);
                }
                else
                {
                  install_action = new Unity.PreviewAction ("install", _("Install"), null);
                  install_action.activated.connect (app_preview_install_commercial);
                  install_action.activated.connect (app_preview_install);
                }
                preview.add_action (install_action);
              }

              if (details.website != null && details.website != "")
              {
                preview_developer_website = details.website;
                var website_action = new Unity.PreviewAction ("website", _("Developer Site"), null);
                website_action.activated.connect (app_preview_website);
                preview.add_action (website_action);
              }
            }
            else // application is already installed
            {
              preview.add_info (new InfoHint ("date-installed", _("Installed on"), null, details.installation_date));
              var launch_action = new Unity.PreviewAction ("launch", _("Launch"), null);
              preview.add_action (launch_action);
              if (!details.is_desktop_dependency)
              {
                var uninstall_action = new Unity.PreviewAction ("uninstall", _("Uninstall"), null);
                uninstall_action.activated.connect (app_preview_uninstall);
                preview.add_action (uninstall_action);
              }
            }

            preview_installable_desktop_file = details.desktop_file;
            preview_installable_icon_file = details.icon;
          }
          catch (Error e)
          {
            warning ("Failed to get package details for '%s': %s", uri, e.message);
            preview = null;
          }
        }

        // xapian db doesn't know this .desktop file or S-C dbus data provider fails,
        // fallback to DesktopAppInfo (based on installed .desktop file) if available
        if (preview == null && desktopfile != null)
        {
          var app_info = new DesktopAppInfo (desktopfile);
          if (app_info != null)
          {
            preview = new Unity.ApplicationPreview (app_info.get_display_name (), "", app_info.get_description () ?? "", app_info.get_icon (), null);
            var launch_action = new Unity.PreviewAction ("launch", _("Launch"), null);
            preview.add_action (launch_action);
          }
        }

        if (preview == null)
        {
          warning ("No pksearcher nor desktop app info for '%s'", uri);
        }
      }
      return preview;
    }

    /**
     * Override of the default activation handler. The apps lens daemon
     * can handle activation of installable apps using the Software Center
     */
    public Unity.ActivationResponse activate (string uri)
    {
      string[] args;
      string exec_or_dir = null;
      if (uri.has_prefix ("unity-install://"))
      {
        unowned string pkg = uri.offset (16); // strip off "unity-install://" prefix
        debug ("Installing: %s", pkg);
        args = new string[2];
        args[0] = "software-center";
        args[1] = pkg;
      }
      else if (uri.has_prefix ("unity-runner://"))
      {
        string orig;
        orig = uri.offset (15);
        if (orig.has_prefix("\\\\"))
          orig = orig.replace ("\\\\","smb://");
        if (uri_regex != null && uri_regex.match (orig)) {
          try {
            /* this code ensures that a file manager will be used
             * if uri it's a remote location that should be mounted */
            if (mountable_regex.match (orig)) {
              var muris = new GLib.List<string>();
              muris.prepend (orig);
              var file_manager = AppInfo.get_default_for_type("inode/directory", true);
              file_manager.launch_uris(muris,null);
            } else {
              AppInfo.launch_default_for_uri (orig, null);
            }
          } catch (GLib.Error error) {
            warning ("Failed to launch URI %s", orig);
            return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
          }
          return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);

        } else {
          exec_or_dir = Utils.subst_tilde (orig);
          args = exec_or_dir.split (" ", 0);
          for (int i = 0; i < args.length; i++)
            args[i] = Utils.subst_tilde (args[i]);
        }
        this.runner.add_history (orig);
      }
      else
      {
        /* Activation of standard application:// uris */

        /* Make sure fresh install learns quickly */
        if (popularity_map.size <= 5) popularities_dirty = true;

        return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
      }

      if ((exec_or_dir != null) && FileUtils.test (exec_or_dir, FileTest.IS_DIR))
      {
        try {
          AppInfo.launch_default_for_uri ("file://" + exec_or_dir, null);
        } catch (GLib.Error err) {
          warning ("Failed to open current folder '%s' in file manager: %s",
                   exec_or_dir, err.message);
          return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
        }
      }
      else
      {
        try {
          unowned string home_dir = GLib.Environment.get_home_dir ();
          Process.spawn_async (home_dir, args, null, SpawnFlags.SEARCH_PATH, null, null);
        } catch (SpawnError e) {
          warning ("Failed to spawn software-center or direct URI activation '%s': %s",
                   uri, e.message);
          return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
        }
      }

      return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);

    }

    /* Appends the subject URIs from a set of Zeitgeist.Events to our Dee.Model
     * assuming that these events are already sorted */
    public void append_events_with_category (Zeitgeist.ResultSet events,
                                             Dee.Model results,
                                             uint category_id,
                                             bool include_favorites,
                                             int max_results = int.MAX,
                                             Set<string>? allowed_uris = null)
    {
      int num_results = 0;
      foreach (var ev in events)
      {
        string? app_uri = null;
        if (ev.num_subjects () > 0)
          app_uri = ev.get_subject (0).get_uri ();

        if (app_uri == null)
        {
          warning ("Unexpected event without subject");
          continue;
        }

        /* Assert that we indeed have a known application as actor */
        string desktop_id = Utils.get_desktop_id_for_actor (app_uri);

        /* Discard Recently Used apps that are in the launcher */
        if ((category_id == Category.RECENT ||
             category_id == Category.RECENT_APPS) &&
            !include_favorites &&
            (favorite_apps.has_app_id (desktop_id)
            || app_watcher.has_app_id (desktop_id)))
          continue;

        var appmanager = AppInfoManager.get_default ();
        AppInfo? app = appmanager.lookup (desktop_id);

        if (app == null)
          continue;

        if (!app.should_show ())
          continue;

        /* HACK! when using the max_results together with allowed_uris,
         * the limit serves as a truncation point - therefore results which
         * are not displayed the first time won't be displayed later
         * (consider results A, B, C - all of them are allowed and we use
         *  limit 2 - so only A and B is displayed, later we change allowed to
         *  B and C, so normally we would display both B and C, but that's not
         *  desired because C wasn't shown in the first place, so we display
         *  only B) */
        if (num_results++ >= max_results) break;

        if (allowed_uris != null && !(app_uri in allowed_uris)) continue;

        string full_path = appmanager.get_path (desktop_id);
        string full_uri = full_path != null ? "file://" + full_path : app_uri;

        results.append (app_uri, app.get_icon().to_string(), category_id,
                        "application/x-desktop", app.get_display_name (),
                        sanitize_binary_name (app.get_executable ()), full_uri);
      }
    }

  } /* END: class Daemon */

} /* namespace */
