/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Didier Roche <didrocks@ubuntu.com>
 *
 */

using Dee;
using Gee;


namespace Unity.ApplicationsPlace {

  private class AboutEntry {
    public string name;
    public string exec;
    public Icon   icon;
    
    public AboutEntry (string name, string exec, Icon icon)
    {
      this.name = name;
      this.exec = exec;
      this.icon = icon;
    }
  }

  public class Runner: GLib.Object
  {  
  
    private Unity.ApplicationsPlace.Daemon daemon;    
    private const string BUS_NAME_PREFIX = "com.canonical.Unity.ApplicationsPlace.Runner";
    
    /* for now, load the same keys as gnome-panel */
    private const string HISTORY_KEY = "/apps/gnome-settings/gnome-panel/history-gnome-run";
    private const int    MAX_HISTORY = 10;
    
    public Unity.PlaceEntryInfo place_entry;

    private bool all_models_synced;

    /* We remember the previous search so we can figure out if we should do
     * incremental filtering of the result models */
    private PlaceSearch? previous_search;    
    
    private Gee.HashMap<string,AboutEntry> about_entries;
    private Gee.List<string> executables;
    private Gee.List<string> history;

    private GConf.Client     client;
    
    public Runner (Unity.ApplicationsPlace.Daemon daemon)
    {
      var sections_model = new Dee.SharedModel(BUS_NAME_PREFIX + "SectionsModel");
      sections_model.set_schema ("s", "s");

      var groups_model = new Dee.SharedModel(BUS_NAME_PREFIX + "GroupsModel");
      groups_model.set_schema ("s", "s", "s");

      var results_model = new Dee.SharedModel(BUS_NAME_PREFIX + "ResultsModel");
      results_model.set_schema ("s", "s", "u", "s", "s", "s");
    
      place_entry = new PlaceEntryInfo ("/com/canonical/unity/applicationsplace/runner");
      place_entry.sections_model = sections_model;
      place_entry.entry_renderer_info.groups_model = groups_model;
      place_entry.entry_renderer_info.results_model = results_model;

      previous_search = null;
      
      /* create the private about entries */
      about_entries = new Gee.HashMap<string,AboutEntry> ();
      load_about_entries ();
      
      executables = new Gee.ArrayList<string> ();
      find_system_executables.begin ();
      
      client = GConf.Client.get_default ();
      history = new Gee.ArrayList<string> ();
      load_history ();
      
      /* Listen for changes to the place entry search */
      place_entry.notify["active-search"].connect (
        (obj, pspec) => {
          if (!all_models_synced)
            return;

          var search = place_entry.active_search;
          
          if (!Utils.search_has_really_changed (previous_search, search))
            return;
          
          update_search.begin(search);
          previous_search = search;
        }
      );

      /* We should not start manipulating any of our models before they are
       * all synchronized. When they are we set all_models_synced = true */
      sections_model.notify["synchronized"].connect (check_models_synced);
      groups_model.notify["synchronized"].connect (check_models_synced);
      results_model.notify["synchronized"].connect (check_models_synced);
      all_models_synced = false;

      this.daemon = daemon;
    
    }

    private async void update_search (PlaceSearch? search)
    {
      var model = place_entry.entry_renderer_info.results_model;
      var executables_match = new Gee.ArrayList<string> ();
      var dirs_match = new Gee.ArrayList<string> ();
      model.clear ();

      var search_string = search.get_search_string ();
      bool has_search = !Utils.search_is_invalid (search);

      string uri;      
      Icon icon;
      string mimetype;
      string display_name;
      var group_id = RunnerGroup.HISTORY;

      foreach (var command in this.history)
      {          
        display_name = get_icon_uri_and_mimetype (command, out icon, out uri, out mimetype);   
        model.append (uri, icon.to_string (),
                      group_id, mimetype,
                      display_name,
                      null);
      }
              
      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remember to thaw the notifys with release_entrylock()! */
      place_entry.freeze_notify ();

      if (!has_search)
      {
        release_entrylock ();
        search.finished ();
        return;
      }

      Timer timer = new Timer ();
      
      /* no easter egg in unity */
      if (search_string == "free the fish")
      {
        uri = "no-easter-egg";
        string commenteaster = _("There is no easter egg in Unity");
        icon = new ThemedIcon ("gnome-panel-fish");
        model.append (uri, icon.to_string (),
                      0, "no-mime",
                      commenteaster,
                      null);
        release_entrylock ();
        search.finished ();
        return;
      }
      else if (search_string == "gegls from outer space")
      {
        uri = "really-no-easter-egg";
        string commentnoeaster = _("Still no easter egg in Unity");
        icon = new ThemedIcon ("gnome-panel-fish");
        model.append (uri, icon.to_string (),
                      0, "no-mime",
                      commentnoeaster,
                      null);
        release_entrylock ();
        search.finished ();
        return;
         
      }
      
      /* manual seek with directory and executables result */
      if (search_string.has_prefix ("/") || search_string.has_prefix ("~"))
      {
        search_string = Utils.subst_tilde (search_string);
        var search_dirname = Path.get_dirname (search_string);
        var directory = File.new_for_path (search_dirname);
        var search_dirname_in_path = false;
        
        /* strip path_directory if in executable in path */
        foreach (var path_directory in Environment.get_variable ("PATH").split(":"))
        {
          if (search_dirname == path_directory || search_dirname == path_directory + "/")
          {
            search_dirname_in_path = true;
            break;
          }
        }
                
        try {
          var iterator = directory.enumerate_children (FILE_ATTRIBUTE_STANDARD_NAME + "," + FILE_ATTRIBUTE_STANDARD_TYPE + "," + FILE_ATTRIBUTE_ACCESS_CAN_EXECUTE,
                                                 0, null);
          while (true)
          {
            var subelem_info = iterator.next_file ();
            if (subelem_info == null)
              break;
            
            var complete_path = Path.build_filename (search_dirname, subelem_info.get_name ());
            if (complete_path.has_prefix (search_string) && complete_path != search_string)
            {
              if (subelem_info.get_file_type () == FileType.DIRECTORY)
              {
                dirs_match.add (complete_path);
              }
              else if (subelem_info.get_attribute_boolean (FILE_ATTRIBUTE_ACCESS_CAN_EXECUTE))
              {
                // only include exec name if we are in the PATH
                if (search_dirname_in_path)
                  executables_match.add (subelem_info.get_name ());
                else
                  executables_match.add (complete_path);
              }
            }
          }
        }
        catch (Error err) {
          warning("Error listing directory executables: %s\n", err.message);
        }

      }
      /* complete again system executables */
      else
      {
        foreach (var exec_candidate in this.executables)
        {
          if (exec_candidate.has_prefix (search_string) && exec_candidate != search_string)
          {
            executables_match.add (exec_candidate);
          }
        }
      }
      
      executables_match.sort ();
      dirs_match.sort ();
      
      group_id = RunnerGroup.RESULTS;

      // populate results
      // 1. enable launching the exact search string
      display_name = get_icon_uri_and_mimetype (search_string, out icon, out uri, out mimetype);
      model.append (uri, icon.to_string (),
                    group_id, mimetype,
                    display_name,
                    null);
      
      // 2. add possible directories (we don't store them)
      mimetype = "inode/directory";
      icon = ContentType.get_icon (mimetype);
      foreach (var dir in dirs_match)
      {
        uri = @"unity-runner://$(dir)";
        model.append (uri, icon.to_string (),
                      group_id, mimetype,
                      dir,
                      null);        
      }
              
      // 3. add available exec
      foreach (var final_exec in executables_match)
      {
        // TODO: try to match to a desktop file for the icon
        uri = @"unity-runner://$(final_exec)";
        display_name = get_icon_uri_and_mimetype (final_exec, out icon, out uri, out mimetype);   
        
        model.append (uri, icon.to_string (),
                      group_id, mimetype,
                      display_name,
                      null);
      }
      
      timer.stop ();
      debug ("Entry search listed %i dir matches and %i exec matches in %fms for search: %s",
             dirs_match.size, executables_match.size, timer.elapsed ()*1000, search_string);
      

      release_entrylock ();
      search.finished ();
    }
    
    private void release_entrylock ()
    {
       /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        place_entry.thaw_notify ();
        return false;
      });       
    }

    // TODO: add reload
    private async void find_system_executables ()
    {

      if (this.executables.size > 0)
        return;
       
      foreach (var path_directory in Environment.get_variable ("PATH").split(":"))
      {
        var dir = File.new_for_path (path_directory);
        try {
          var e = yield dir.enumerate_children_async (FILE_ATTRIBUTE_STANDARD_NAME + "," + FILE_ATTRIBUTE_ACCESS_CAN_EXECUTE,
                                                      0, Priority.DEFAULT, null);
          while (true) {
            var files = yield e.next_files_async (10, Priority.DEFAULT, null);
            if (files == null)
                break;

            foreach (var info in files) {
              if (info.get_attribute_boolean (FILE_ATTRIBUTE_ACCESS_CAN_EXECUTE))
              {
                this.executables.add (info.get_name ());
              }
            }
          }
        }
        catch (Error err) {
          warning("Error listing directory executables: %s\n", err.message);
        }
      }
    }
    
    private string get_icon_uri_and_mimetype (string exec_string, out Icon? icon, out string? uri, out string? mimetype)
    {
      
      AboutEntry? entry = null;
      
      mimetype = "application/x-unity-run";      
      entry = about_entries[exec_string];
      if (entry != null)
      {
        uri = @"unity-runner://$(entry.exec)";
        icon = entry.icon;
        return entry.name;
      }

      uri = @"unity-runner://$(exec_string)";

      
      // if it's a folder, show… a folder icone! + right exec
      if (FileUtils.test (exec_string, FileTest.IS_DIR))
      {
        mimetype = "inode/directory";
        icon = ContentType.get_icon (mimetype);
        return exec_string;
      }
                        
      var s = exec_string.delimit ("-", '_').split (" ", 0)[0];      
      var appresults = this.daemon.appsearcher.search (@"type:Application AND exec:$s", 0,
                                                       Unity.Package.SearchType.EXACT,
                                                       Unity.Package.Sort.BY_NAME);
      foreach (var pkginfo in appresults.results)
      {

        if (pkginfo.desktop_file == null)
          continue;
        
        // pick the first one
        icon = this.daemon.find_pkg_icon (pkginfo);
        return exec_string;
        
      }
      
      // if no result, default icon
      icon = new ThemedIcon ("gtk-execute");
      return exec_string;
      
    }

    
    /* The check_models_synced() method acts like a latch - once all models
     * have reported themselves to be synchronized we set
     * all_models_synced = true and tell the searches to re-check their state
     * as they should refuse to run when all_models_synced == false */
    private void check_models_synced (Object obj, ParamSpec pspec)
    {
      if ((place_entry.sections_model as Dee.SharedModel).synchronized &&
          (place_entry.entry_renderer_info.groups_model as Dee.SharedModel).synchronized &&
          (place_entry.entry_renderer_info.results_model as Dee.SharedModel).synchronized) {
        if (all_models_synced == false)
          {
            all_models_synced = true;
            
            populate_groups (place_entry.entry_renderer_info.groups_model);
            
            /* Emitting notify here will make us recheck if the search results
             * need update. In the negative case this is a noop */
            place_entry.notify_property ("active-search");
          }
      }
    }
    
    private void populate_groups (Dee.Model groups)
    {
      if (groups.get_n_rows() != 0)
        {
          debug ("The groups model already populated. We probably cloned it off Unity. Rebuilding.");
          groups.clear ();
        }

      // TODO: needs asset
      groups.append ("UnityDefaultRenderer",
                     _("Results"),
                     ICON_PATH + "group-installed.svg");
      groups.append ("UnityDefaultRenderer",
                     _("History"),
                     ICON_PATH + "group-available.svg");

      /* expand the history */
      place_entry.entry_renderer_info.set_hint ("ExpandedGroups",
                                                 @"$((uint)RunnerGroup.HISTORY)");
    }
    
    public void add_history (string last_command)
    {

      // new history list: better, greatest, latest!
      var new_history = new Gee.ArrayList<string> ();
      int i = 0;
      
      new_history.insert (i, last_command);
      for (var j = 0; (j < this.history.size) && (i < MAX_HISTORY); j++)
      {
        if (this.history[j] == last_command)
           continue;
        
        new_history.add(history[j]);
        i++;
      }
      this.history = new_history;
      
      // store in gconf
      SList<string> history_store = null;
      foreach (var command in this.history)
      {
        history_store.prepend(command);
      }
      try
      {
        this.client.set_list (HISTORY_KEY, GConf.ValueType.STRING, history_store);
      }
      catch (Error e)
      {
        warning ("Can't store the history: %s", e.message);
      }
      
      // force a search to refresh history order (TODO: be more clever in the future)
      update_search.begin(place_entry.active_search);
    }
    
    private void load_history ()
    {
      try
      {
        int i = 0;
        SList<string> history_store = this.client.get_list (HISTORY_KEY, GConf.ValueType.STRING);
        foreach (var command in history_store)
        {
          if (i >= MAX_HISTORY)
            break;
          this.history.insert(0, (string) command.data);
          i++;
        }
      }
      catch (Error e)
      {
        warning ("Can't access to the history: %s", e.message);
      }
    }
    
    private void load_about_entries ()
    { 
      AboutEntry entry;
      string name;
      string exec;
      Icon icon;
      
      // first about:config
      name = "about:config";
      exec = "ccsm -p unityshell";
      try {
        icon = Icon.new_for_string (@"$(Config.PREFIX)/share/ccsm/icons/hicolor/64x64/apps/plugin-unityshell.png");      
      }
      catch (Error err) {
        warning ("Can't find unityshell icon: %s", err.message);
        icon = new ThemedIcon ("gtk-execute");
      }   
      entry = new AboutEntry (name, exec, icon);
      
      about_entries[name] = entry;
      about_entries[exec] = entry;
      
      // second about:robots
      name = "Robots have a plan.";
      exec = "firefox about:robots";
      entry = new AboutEntry (name, exec, icon = new ThemedIcon ("battery"));
      
      about_entries["about:robots"] = entry;
      about_entries[exec] = entry;
      
    }
      
  }
  
}
