/* AbstractNamedFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories;

import java.io.IOException;

import org.grinvin.params.ParameterList;
import org.grinvin.util.InternationalizedProperties;
import org.grinvin.xml.XMLUtils;

import org.jdom.Element;

/**
 * Common superclass of various visual graph factory classes.
 * Provides support for loading various properties from an xml configuration file
 */
public class AbstractNamedFactory extends AbstractFactory implements NamedFactory {
    
    /**
     * Default constructor.
     * @param type String identifying the type of this factory.
     */
    public AbstractNamedFactory(String type) {
        load(getClass().getName().replace('.', '/') + ".xml", type);
    }
    
    /**
     * Internationalized properties associated with this factory.
     */
    protected InternationalizedProperties factoryProps;
    
    //
    private String factoryDescription;
    
    //
    private String factoryName;
    
    //
    private String factoryLongName;
    
    //
    private String iconPath;
    
    //
    private String selectedIconPath;
    
    /**
     * Return the URI of the factory icon when the factory is not selected.
     */
    public String getIconPath() {
        return iconPath;
    }
    
    /**
     * Return the URI of the factory icon when the factory is selected.
     */
    public String getSelectedIconPath() {
        return selectedIconPath;
    }
    
    public String getName() {
        return factoryName;
    }
    
    public String getLongName() {
        return factoryLongName;
    }
    
    public String getDescription() {
        return factoryDescription;
    }
    
    /**
     * Return the fully qualified class name of this factory
     * as its id.
     */
    public String getId() {
        return getClass().getName ();
    }
    
    /**
     * Load factory information from the description in the given XML-config file.
     */
    private void load(String config, String prefix) {
        try {
            Element root = XMLUtils.loadFromClassPath(config);
            // resources
            Element el = root.getChild("i18n");
            if (el == null)
                throw new RuntimeException("Invalid input format: cannot find 'i18n' element");
            factoryProps = new InternationalizedProperties();
            factoryProps.fromElement(el);
            
            // parameters
            list = new ParameterList();
            list.fromElement(root);
            
            // cell properties
            factoryLongName = factoryProps.getProperty(prefix + ".name");
            factoryName = factoryProps.getProperty(prefix + ".text");
            if (factoryName == null)
                factoryName = factoryLongName;
            factoryDescription = factoryProps.getProperty(prefix + ".description");
            if (factoryDescription == null)
                factoryDescription = factoryName;
            
            el = root.getChild("icon");
            if (el != null)
                iconPath = el.getText();
            el = root.getChild("selectedIcon");
            if (el != null)
                selectedIconPath = el.getText();
            if (selectedIconPath == null)
                selectedIconPath = iconPath;
        } catch (IOException ex) {
            throw new RuntimeException("Invalid input format", ex);
        }
    }
    
    
}
