/* DefaultAnnotation.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphs;

import java.util.HashMap;
import java.util.Map;

/**
 *
 */
public class DefaultAnnotation implements Annotation {
    
    //
    private Map<Element, Object> map;
    
    //
    private GraphView peer;

    public DefaultAnnotation(GraphView peer) {
        this.peer = peer;
        map = new HashMap<Element, Object>();
    }

    public GraphView getPeer() {
        return peer;
    }

    public Object getAnnotation(Vertex vertex) {
        if (!peer.contains(vertex))
            throw new IllegalArgumentException("Vertex does not belong to the graph");
        return map.get(vertex);
    }

    public Object getAnnotation(Edge edge) {
        if (!peer.contains(edge))
            throw new IllegalArgumentException("Edge does not belong to the graph");
        return map.get(edge);
    }

    public void setAnnotation(Vertex vertex, Object annotation) {
        map.put(vertex, annotation);
    }

    public void setAnnotation(Edge edge, Object annotation) {
        map.put(edge, annotation);
    }
    
    // implements Annotation
    public void copy(AnnotationView original) {
        if (this == original)
            return;
        this.peer = original.getPeer();
        map.clear();
        for (Vertex v : peer.vertices())
            map.put(v, original.getAnnotation(v));
        for (Edge e : peer.edges())
            map.put(e, original.getAnnotation(e));
    }
    
    // implements Annotation
    public void copy(AnnotationView original, GraphView newPeer) {
        if (this == original)
            throw new IllegalArgumentException("Cannot make copy of self with different peer");
        GraphView originalPeer = original.getPeer();
        if (newPeer == originalPeer) {
            copy(original);
        } else {
            this.peer = newPeer;
            this.map.clear();
            for (Vertex v: originalPeer.vertices())
                map.put(newPeer.getVertex(v.getIndex()), original.getAnnotation(v));
            for (Edge e : originalPeer.edges())
                map.put(newPeer.getEdge(newPeer.getVertex(e.getFirstEndpoint().getIndex()), newPeer.getVertex(e.getSecondEndpoint().getIndex())), original.getAnnotation(e));
        }
    }

}
