/* FilterListComponent.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.components;

import be.ugent.caagt.swirl.actions.SimpleAction;
import be.ugent.caagt.swirl.dnd.LocalTransferHandler;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.util.ResourceBundle;
import javax.swing.AbstractAction;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;
import javax.swing.KeyStroke;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.grinvin.conjecture.filter.BooleanValueGraphFilter;
import org.grinvin.conjecture.filter.CompoundGraphFilter;
import org.grinvin.conjecture.filter.CompoundGraphFilter.Operator;
import org.grinvin.conjecture.filter.GraphFilter;
import org.grinvin.gui.components.AcceptsInvariant.Reason;
import org.grinvin.graphs.GraphURIType;
import org.grinvin.gui.WatermarkPainter;
import org.grinvin.gui.dnd.FilterDropHandler;
import org.grinvin.gui.dnd.HasAccessControl;
import org.grinvin.gui.dnd.InvariantDropHandler;
import org.grinvin.gui.dnd.InvariantFactoryDropHandler;
import org.grinvin.gui.dnd.SelectionDragHandler;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.invariants.values.BooleanValue;
import org.grinvin.list.DefaultFilterList;
import org.grinvin.list.DefaultFilterListModel;
import org.grinvin.list.FilterList;
import org.grinvin.list.FilterListModel;
import org.grinvin.list.HasSelectableValues;



/**
 * List component with entries of type {@link GraphFilter}.
 */
public class FilterListComponent extends JList implements HasAccessControl, HasSelectableValues, AcceptsInvariant, AcceptsFilter {
    
    // shared instance
    private static final ListCellRenderer RENDERER = new Renderer ();
    
    //
    private static final String BUNDLE_NAME = "org.grinvin.worksheet.resources";
    
    // shared watermark painter
    private static final WatermarkPainter WATERMARK_PAINTER = new WatermarkPainter(
            20,
            ResourceBundle.getBundle(BUNDLE_NAME).getString("FilterList.emptytext"),
            new Color(241, 227, 241)
            );
    
    // shared transfer handler
    protected static final LocalTransferHandler TRANSFER_HANDLER;
    
    static {
        TRANSFER_HANDLER = new LocalTransferHandler();
        TRANSFER_HANDLER.addDropHandler(InvariantDropHandler.getInstance());
        TRANSFER_HANDLER.addDropHandler(InvariantFactoryDropHandler.getInstance());
        TRANSFER_HANDLER.addDropHandler(FilterDropHandler.getInstance());
        TRANSFER_HANDLER.setDragHandler(new SelectionDragHandler(GraphFilter.class));
    }
    
    public FilterListComponent(FilterListModel model) {
        this(model, LocalTransferHandler.COPY_OR_MOVE, LocalTransferHandler.COPY_OR_MOVE);
    }
    
    /**
     * Creates a new instance of InvariantListComponent
     */
    public FilterListComponent(FilterListModel model, int dragOperations, int dropOperations) {
        super(model);
        
        this.dragOperations = dragOperations;
        this.dropOperations = dropOperations;
        
        setTransferHandler(TRANSFER_HANDLER);
        setDragEnabled(true);
        setCellRenderer(RENDERER);
        setOpaque(false); // we paint our own background
        setPaintWatermark(true);
        
        // Keyboard interaction
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("DELETE"), "deleteSelectedElements");
        getActionMap().put("deleteSelectedElements", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                if(!isReadOnly())
                    deleteSelectedElements();
            }
        });
        
        //CSH.addManager(new InvariantListHelpManager(this));
    }
    
    //
    private int dragOperations;
    
    //
    private int dropOperations;
    
    // paint the watermark?
    private boolean paintWatermark;
    
    /**
     * Configure whether to paint the watermark or not. If no watermark
     * is painted, the list background is completely transparent.
     * Note that this behavious is different from a {@link GraphCellListComponent}
     * which has no watermark.
     */
    public void setPaintWatermark(boolean paintWatermark) {
        this.paintWatermark = paintWatermark;
    }
    
    /**
     * Adds a watermark to the list.
     */
    @Override
    protected void paintComponent(Graphics g) {
        if(paintWatermark){
            g.setColor(getBackground());
            g.fillRect(0, 0, getWidth(), getHeight());
            WATERMARK_PAINTER.paint(this, g);
        }
        super.paintComponent(g);
    }
    
    //
    private FilterList getFilterList() {
        return (FilterList)super.getModel();
    }
    
    /**
     * Delete currently selected elements.
     */
    public void deleteSelectedElements() {
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        // gather all elements to be deleted
        FilterList list = new DefaultFilterList();
        for(int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                list.add(getFilterList().get(i));
        
        // and remove them
        getFilterList().removeAll(list);
        //TODO: make sure changed messages are sent
    }

    /**
     * Toggle the negation of the currently selected filters.
     */
    public void toggleNegagtionSelectedElements() {
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        // gather all elements to be deleted
        for(int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                getFilterList().get(i).negate(!getFilterList().get(i).isNegated());
        
        repaint();
                
        //TODO: make sure changed messages are sent
    }

    /**
     * Combine the currently selected filters into one filter.
     */
    public void combineSelectedFilters(Operator operator) {
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin + 1)
            return ; // no selection or just one filter selected
        
        // gather all elements to be deleted
        FilterListModel list = new DefaultFilterListModel();
        for(int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                list.add(getFilterList().get(i));
        
        
        //combine them
        GraphFilter compound = new CompoundGraphFilter(list, operator);
        
        // and remove them
        getFilterList().removeAll(list);
        
        // and add combined filter
        getFilterList().add(compound);
        
        //TODO: make sure changed messages are sent
    }
    
    /**
     * Split the selected filters again.
     */
    public void splitSelectedElements() {
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        FilterList removeList = new DefaultFilterList();
        FilterList addList = new DefaultFilterList();
        
        // gather all elements to be deleted
        for(int i = iMax; i >= iMin; i--) {
            if (selectionModel.isSelectedIndex(i)) {
                GraphFilter filter = getFilterList().get(i);
                if (filter instanceof CompoundGraphFilter) {
                    removeList.add(filter);
                    addList.addAll(((CompoundGraphFilter)filter).getFilters());
                }
            }
        }
        
        getFilterList().removeAll(removeList);
        getFilterList().addAll(addList);
        
        repaint();
                
        //TODO: make sure changed messages are sent
    }


    public int getDragOperations() {
        return dragOperations;
    }
    
    public int getDropOperations() {
        return dropOperations;
    }
    
    //
    private boolean isReadOnly = false;
    
    //
    public void setIsReadOnly(boolean isReadOnly) {
        this.isReadOnly = isReadOnly;
    }
    
    //
    public boolean isReadOnly() {
        return isReadOnly;
    }

    public Reason addInvariant(Invariant invariant) {
        getFilterList().add(new BooleanValueGraphFilter(invariant));
        return Reason.SUCCESS;
    }

    public Reason acceptsInvariant(Invariant invariant) {
        return invariant.getType().equals(BooleanValue.class) ? Reason.SUCCESS : Reason.WRONG_TYPE;
    }
    
    public Reason acceptsInvariant(Class<? extends InvariantValue> clazz) {
        return clazz.equals(BooleanValue.class) ? Reason.SUCCESS : Reason.WRONG_TYPE;
    }

    public boolean addFilter(GraphFilter filter) {
        return getFilterList().add(filter);
    }
    
    //
    private static class Renderer extends DefaultListCellRenderer {
        
        Renderer() {
            // avoid creation of access type
        }
        
        @Override
        public Component getListCellRendererComponent
                (JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            
            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            setOpaque(isSelected);
            Color bg = getBackground();
            setBackground(new Color(bg.getRed(), bg.getGreen(), bg.getBlue(), 128));
            if (value instanceof GraphFilter) {
                GraphFilter filter = (GraphFilter)value;
                setText(filter.getDescription());
                setIcon(GraphURIType.FILTER.getSmallIcon());
            } else
                throw new IllegalArgumentException("ListCellRenderer can only be used with lists of GraphFilter");
            return this;
        }
    }
    
    public static class NegateSelectedElements extends FilterListAction {

        public NegateSelectedElements(FilterListComponent component) {
            super(component, "action.negateselectedelements");
        }

        public void performAction() {
            component.toggleNegagtionSelectedElements();
        }
        
        protected boolean isActive() {
            return getSelectionCount() > 0;
        }

    }

    public static class CombineSelectedAND extends FilterListAction {

        public CombineSelectedAND(FilterListComponent component) {
            super(component, "action.combineselectedAND");
        }

        public void performAction() {
            component.combineSelectedFilters(Operator.AND);
        }

        protected boolean isActive() {
            return getSelectionCount() > 1;
        }

    }

    public static class CombineSelectedOR extends FilterListAction {

        public CombineSelectedOR(FilterListComponent component) {
            super(component, "action.combineselectedOR");
        }

        protected void performAction() {
            component.combineSelectedFilters(Operator.OR);
        }
    
        protected boolean isActive() {
            return getSelectionCount() > 1;
        }

    }

    public static class SplitSelectedElements extends FilterListAction {

        public SplitSelectedElements(FilterListComponent component) {
            super(component, "action.splitselectedelements");
        }

        public void performAction() {
            component.splitSelectedElements();
        }
        
        protected boolean isActive() {
            return getSelectionCount() > 0;
        }

    }
    
    private abstract static class FilterListAction extends SimpleAction implements ListSelectionListener {
        
        //
        protected FilterListComponent component;
        
        public FilterListAction(FilterListComponent component, String id) {
            super(ResourceBundle.getBundle(BUNDLE_NAME), id, null);
            this.component = component;
            component.addListSelectionListener(this);
            setEnabled(isActive());
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            if (!component.isReadOnly()) {
                performAction();
            }
        }
        
        public void valueChanged(ListSelectionEvent e) {
            setEnabled(isActive());
        }
        
        protected int getSelectionCount() {
            return component.getSelectedIndices().length;
        }
        
        protected abstract void performAction();
        
        protected abstract boolean isActive();
        
    }

}
