/* FilteredInvariantValueTableCellRenderer.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.components.render;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Polygon;
import java.util.ResourceBundle;
import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;

import org.grinvin.conjecture.filter.GraphFilter;
import org.grinvin.worksheet.HasRequestedCell;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.invariants.InvariantValueDelegate;
import org.grinvin.list.graphs.GraphList;

/**
 * Table cell renderer for objects of type {@link InvariantValue}.
 */
public class FilteredInvariantValueTableCellRenderer extends DefaultTableCellRenderer {
    
    //
    private GraphFilter filter;
    
    //
    private GraphList graphList;
    
    public FilteredInvariantValueTableCellRenderer (GraphFilter filter, GraphList graphList) {
        this.filter = filter;
        this.graphList = graphList;
    }
    
    //
    private boolean hasDetails;
    
    /**
     * Renders an invariant value.
     */
    @Override
    public void setValue(Object value) {
        ResourceBundle rbundle = ResourceBundle.getBundle ("org.grinvin.list.resources");
        if (value instanceof InvariantValue) {
            InvariantValue invariantValue = (InvariantValue)value;
            InvariantValueDelegate delegate = invariantValue.getType().getDelegate();
            delegate.initLabel(this, invariantValue);
            this.hasDetails = delegate.hasDetails(invariantValue);
        } else if (value == null) {
            setIcon(null);
            setText(rbundle.getString("InvariantValue.notYetCalculated"));
            this.hasDetails = false;
        } else
            throw new IllegalArgumentException
                    ("InvariantValueTableCellRenderer can only be used with columns of type InvariantValue");
    }
    
    private static final int SIDE = 10;
    
    private static final Polygon polygon = new Polygon (
            new int[] { 0, SIDE, SIDE}, new int[] {SIDE, 0, SIDE}, 3);
    
    protected void paintDetailsIndicator (Graphics g1) {
        Graphics g = g1.create (getWidth() - SIDE, getHeight() - SIDE, SIDE, SIDE);
        g.setColor(Color.GRAY);
        g.fillPolygon(polygon);
    }

    protected void paintRequestedIndicator (Graphics g1) {
        Graphics g = g1.create ();
        g.setColor(Color.RED);
        g.drawRect(0,0,getWidth()-1, getHeight()-1);
    }

    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        if (hasDetails)
            paintDetailsIndicator (g);
        if (isRequested)
            paintRequestedIndicator(g);
    }

    private boolean isRequested;
    
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
        isRequested = (table instanceof HasRequestedCell) && ((HasRequestedCell)table).isRequestedCell(row, column);
        super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
        setOpaque(isSelected);
        Color bg = getBackground();
        if(! filter.accepts(graphList.get(row).getBundle())){
            setOpaque(true);
            bg = isSelected ? Color.RED.darker() : Color.RED;
        }
        setBackground(new Color(bg.getRed(), bg.getGreen(), bg.getBlue(), 128));
        return this;
    }
    
}
