/* GraphListElementManager.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list.graphs;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.graphs.DefaultGraphBundle;
import org.grinvin.graphs.GraphBundle;
import org.grinvin.graphs.GraphURI;
import org.grinvin.graphs.GraphURIException;
import org.grinvin.io.IOFormatException;
import org.grinvin.io.SectionLoader;
import org.grinvin.xml.XMLUtils;

import org.jdom.Element;

/**
 * Manages the mapping between Graph URI's and the GraphListElement objects representing them.
 */
public class GraphListElementManager {
    
    //
    private static final GraphListElementManager SINGLETON = new GraphListElementManager();

    /**
     * Maps a Graph uri to the corresponding object of type GraphListElement.
     */
    private Map<URI, GraphListElement> graphlistelements;
    
    //
    private GraphGroup standardGraphNodes;
    
    //
    private GraphListElementManager() {
        graphlistelements = new HashMap<URI, GraphListElement>();
        standardGraphNodes = processStandardGraphNodes();
    }

    /**
     * Get the singleton instance of this class
     * @return the single shared instance of this class
     */
    public static GraphListElementManager getInstance() {
        return SINGLETON;
    }
    
    //
    private void registerGraphListElement(GraphListElement element) {
        graphlistelements.put(element.getURI(), element);
    }
    
    /**
     * Get the {@link GraphListElement} for the given {@link URI}.
     * @param directory Base directory to be used for (relative) session URIs, or null
     * if session URIs are not allowed
     * @return the graph list element
     */
    public GraphListElement getGraphListElement(URI uri, SectionLoader sloader) {
        GraphListElement element = graphlistelements.get(uri);
        if (element == null) {
            GraphBundle bundle = new DefaultGraphBundle ();
            try {
                GraphURI.load(uri, bundle, sloader);
            } catch (GraphURIException ex) {
                Logger.getLogger("org.grinvin.io", "org.grinvin.main.resources").log(Level.WARNING, "Failed to load graph from " + uri, ex);
                bundle = null;
            }
            element = new GraphListElement(uri, bundle);
        }
        registerGraphListElement(element);
        return element;
    }
    
    /**
     * Convenience method for retrieving a graph list element with given
     * URI. Same as <tt>getGraphListElement(uri,null)</tt>.
     */
    public GraphListElement getGraphListElement(URI uri) {
        return getGraphListElement (uri,null);
    }
    
    //
    public GraphListElement createGraphListElement(GraphBundle bundle) {
        return new GraphListElement(bundle);
    }
    
    public GraphGroup getStandardGraphNodes() {
        return standardGraphNodes;
    }
        
    private GraphGroup processStandardGraphNodes() {
        String path = "org/grinvin/graphs/resources/standardgraphs.xml";
        String bundle = "org.grinvin.graphs.resources.standardgraphs";
        return loadGroups(path, bundle);
    }

    private GraphGroup loadGroups(String path, String bundle) {
        GraphGroup result = new GraphGroup("Dummy");
        try {
            load(XMLUtils.loadFromClassPath(path), result, ResourceBundle.getBundle(bundle));
        } catch (MissingResourceException ex) {
                Logger.getLogger("org.grinvin.io").log(Level.WARNING, "Failed to load standard graphs", ex);
        } catch (IOException ex) {
            Logger.getLogger("org.grinvin.io").log(Level.WARNING, "Failed to load standard graphs", ex);
        }
        return result;
    }

    
    private void load(Element root, GraphGroup parent, ResourceBundle bundle) throws IOFormatException {
        for (Object o : root.getChildren()) {
            Element element = (Element) o;
            String name = element.getName();
            if ("group".equals(name)) {
                String caption = bundle.getString(element.getAttributeValue("i18n"));
                GraphGroup group = new GraphGroup(caption);
                load(element, group, bundle); // process children
                parent.addChild(group);
            } else if ("graphlistelement".equals(name)) {
                try {
                    parent.addChild(getGraphListElement(new URI(element.getAttributeValue("uri"))));
                } catch (URISyntaxException ex) {
                    throw new IOFormatException("invalid graph URI");
                }
            } else {
                throw new IOFormatException("Unknown element: " + name);
            }
        }
    }


    
}
