/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <dvdread/ifo_read.h>
#include <string.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "dvd.h"
#include "dvd-time.h"

static GObjectClass *dvd_parent_class = NULL;

static void   dvd_class_init		(DvdClass	*class);
static void   dvd_instance_init		(GTypeInstance	*instance,
					 gpointer	 g_class);
static void   dvd_dispose		(GObject	*object);

static void
dvd_class_init		(DvdClass	*class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	dvd_parent_class = g_type_class_ref (G_TYPE_OBJECT);
	object_class->dispose = dvd_dispose;
}

static void
dvd_instance_init	(GTypeInstance	*instance,
			 gpointer	 g_class)
{
	Dvd *dvd;
	
	dvd = DVD (instance);
	dvd->name = NULL;
	dvd->provider_id = NULL;
	dvd->title_count = 0;
	dvd->titles = NULL;
}

static void
dvd_dispose		(GObject	*object)
{
	Dvd *dvd = DVD (object);
	
	dvd_delete_all (dvd);
	
	G_OBJECT_CLASS (dvd_parent_class)->dispose (G_OBJECT (dvd));
}

/**
 * dvd_get_type:
 *
 * Returns the GType for the Dvd class.
 */

GType
dvd_get_type		(void)
{
	static GType dvd_type = 0;

	if (dvd_type == 0) {
		GTypeInfo dvd_info = {
			sizeof (DvdClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (Dvd),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_instance_init
	    	};
		dvd_type = g_type_register_static (G_TYPE_OBJECT,
						    "Dvd",
						    &dvd_info, 0);
	}
	return dvd_type;
}

DvdSource
dvd_get_source			(Dvd		*dvd)
{
	g_assert (dvd != NULL);
	
	return dvd->source;
}

void
dvd_set_region		 (Dvd	        *dvd,
			  DvdRegion	 region)
{
	g_assert (dvd != NULL);
	
	dvd->region = region;
	g_message ("Dvd Region = 0x%x", (guint8) region);
}

DvdRegion
dvd_get_region		(Dvd	        *dvd)
{
	guint8 region;
	
	g_assert (dvd != NULL);
	
	region = dvd->region;
	
	return region;
}

void
dvd_set_name			(Dvd	        *dvd,
				 const gchar	*name)
{
	g_assert (dvd != NULL);
	
	g_free (dvd->name);
	if (strlen (name) > 32) {
		g_warning ("truncating dvd name to 32 chars");
		dvd->name = g_strndup (name, 32);
	} else {
		dvd->name = g_strdup (name);
	}
	g_message ("Dvd Name = %s", name);
}

/**
 * Gets a constant pointer to the DVD name string.
 * @param dvd A #Dvd.
 * @returns A constant string if dvd name is set or NULL.
 */
G_CONST_RETURN
gchar
*dvd_get_name			(Dvd		*dvd)
{
	g_assert (dvd != NULL);
	
	return dvd->name;
}

void
dvd_set_volume_set		(Dvd	        *dvd,
				 const gchar	*volume_set)
{
	gint i;
	gchar *tmp;
	g_assert (dvd != NULL);
	
	tmp = g_strdup (volume_set);
	g_strstrip (tmp);
	for (i = 0;
	     tmp[i] != '\0';
	     i++) {
		if (g_ascii_isalnum (tmp[i] == FALSE)) {
			tmp[i] = '_';			
		}
		tmp[i] = g_ascii_toupper (tmp[i]);
	}
	g_free (dvd->volume_set);
	if (i > 128) {
		g_warning ("Trucating volume set to 128");
		dvd->volume_set = g_strndup (tmp, 128);
	} else {
		dvd->volume_set = g_strndup (tmp, i);
	}
	g_message ("Dvd Volume Set = %s", dvd->volume_set);
	g_free (tmp);
}

/**
 * Gets a constant pointer to the volume set ID specified by the disk.
 * @param dvd A #Dvd.
 * @returns A constant string if volume set ID is set or NULL.
 */
G_CONST_RETURN
gchar
*dvd_get_volume_set		(Dvd		*dvd)
{
	g_assert (dvd != NULL);
	
	return dvd->volume_set;
}

void
dvd_set_provider		(Dvd	        *dvd,
				 const gchar	*provider)
{
	g_assert (dvd != NULL);
	
	g_free (dvd->provider_id);
	dvd->provider_id = g_strdup (provider);
	g_message ("DVD Provider = %s", dvd->provider_id);
}

/**
 * Gets a constant pointer to the provider string specified by the disk.
 * @param dvd A #Dvd.
 * @returns A constant string if provider is set or NULL.
 */
G_CONST_RETURN
gchar
*dvd_get_provider		 (Dvd		*dvd)
{
	g_assert (dvd != NULL);
	
	return dvd->provider_id;
}

/**
 * Gets the nuber of titles in the dvd.
 * @param dvd A #Dvd.
 * @returns The number of titles as a guint8.
 */
guint8
dvd_get_title_count		(Dvd		*dvd)
{
	g_assert (dvd != NULL);
	
	return dvd->title_count;
}

void
dvd_append_title		(Dvd	        *dvd,
				 DvdTitle	*title)
{
	g_assert (dvd != NULL);
	g_assert (title != NULL);
	
	/* ToDo - check title and dvd source is the same! */
	g_object_ref (title);
	dvd->titles = g_slist_append (dvd->titles, title);
	++dvd->title_count;
}

void
dvd_delete_all			(Dvd	        *dvd)
{
	g_assert (dvd != NULL);
	
	GSList *iter;
	
	g_free (dvd->name);
	g_free (dvd->provider_id);
	g_free (dvd->volume_set);
	dvd->name = NULL;
	dvd->provider_id = NULL;
	dvd->volume_set = NULL;
	for (iter = dvd->titles;
	     iter != NULL;
	     iter = g_slist_next (iter)) {
		g_object_unref (G_OBJECT (iter->data));
	}
	g_slist_free (dvd->titles);
	dvd->titles = NULL;
	dvd->title_count = 0;
}

/**
 * Gets a pointer to the title specified by the title_id parameter.
 * You must call g_object_unref () on the returned title when you are finished with it.
 * @param dvd A #Dvd.
 * @param title_id A valid title number id of the requested title.
 * @returns A referenced #DvdTitle object.
 */
DvdTitle
*dvd_get_title			(Dvd		*dvd,
				 guint8		 title_id)
{
	DvdTitle *title;
	
	g_assert (dvd != NULL);
	g_assert (title_id < dvd->title_count);
	
	title = g_slist_nth_data (dvd->titles, title_id);
	g_assert (title != NULL);
	g_object_ref (title);
	
	return title;
}

/**
 * Gets the the title number of the feature title.
 * This makes the assumption that the longest title is the feature title.
 * @param dvd A #Dvd.
 * @returns void.
 */
guint8
dvd_get_feature_title_id	(Dvd		*dvd)
{
	guint32 longest = 0;
	guint8 title_id;
	guint8 feature_id = 0;
	
	g_assert (dvd != NULL);
	
	for (title_id = 0;
	     title_id < dvd->title_count;
	     title_id++) {
		DvdTitle *title;
		const DvdTime *time;

		title = dvd_get_title (dvd, title_id);
		time = dvd_title_get_time (title);
		if (time->ts > longest) {
			longest = time->ts;
			feature_id = title_id;
		}
		g_object_unref (title);
	}
	return feature_id;
}
