// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/PreMacroManager.h"
#include "Puma/ErrorStream.h"
#include "Puma/PreMacro.h"
#include "Puma/StrCol.h"
#include <string.h>
#include <stdio.h>

namespace Puma {


void PreMacroManager::clear () {
        PreMacro* m;
        int idx;
        
        for( idx=0; idx < StrHashTable::STRHASHTABLE_SIZE; idx++ ) {
                while( (m = (PreMacro*)_tab[idx].unlink()) != 0 ) {
                        delete m;
                }
        }
}

void PreMacroManager::add (PreMacro* m) {
        _tab[m->getName().magic()].insert(m);
}

PreMacro* PreMacroManager::get (const DString& ds) const {
        PreMacro* m = (PreMacro*)_tab[ds.magic()].select();
        while( m ) {
                if( m->getName() == ds )
                        return m;
                m = (PreMacro*)m->select();
        }
        return 0;
}

void PreMacroManager::kill (const DString& ds) {
        PreMacro* prev = (PreMacro*)&_tab[ds.magic()];
        PreMacro* m = (PreMacro*)prev->select();
        
        while( m && (m->getName() != ds) ) {
                prev = m;
                m = (PreMacro*)m->select();                
        }
        
        if( m ) {
                prev->select(m->select());
                delete m;
        }
}

// Initialize the macro manager.
void PreMacroManager::init (const char* file)
 {
    // Add special predefined macros to macro table.
        _Line = new PreMacro ("__LINE__", "");
    add (_Line);
        _Date = new PreMacro ("__DATE__", ""); 
    add (_Date);
        _Time = new PreMacro ("__TIME__", "");
    add (_Time);
        _IncLevel = new PreMacro ("__INCLUDE_LEVEL__", "0");
    add (_IncLevel);

         if (! file) {
                _File = new PreMacro ("__FILE__", "");
                _BaseFile = new PreMacro ("__BASE_FILE__", "");
         }
         else {
                char* basefile = new char[strlen (file) + 10];
                sprintf (basefile, "\"%s\"", file);
  
                _File = new PreMacro ("__FILE__", basefile);
                _BaseFile = new PreMacro ("__BASE_FILE__", basefile);
    
                delete[] basefile;
        }
         add (_File);
         add (_BaseFile);
 }


// Add a new macro to macro manager.
void PreMacroManager::addMacro (PreMacro *macro) //const
 {
    if (! macro) return;

    // Search for the macro. If found it's a redefinition of the macro.
    if (PreMacro* prev = (PreMacro*) get (macro->getName ())) 
    {
        // Special macro that cannot be (re)defined in the program.
        if (prev->isInhibited () || prev->isLocked ())
            return;

        // A redefinition normally causes a warning. But if the body
        // of the previous version of the macro is empty or not 
        // effectively different from the current macro body, the 
        // redefinition is accepted silently.
        if (! StrCol::onlySpaces (prev->getBody ())) 
        {
            // Effectively different means, that the two bodies differ
            // only in white spaces. It is not allowed to remove white 
            // spaces completely or to add white spaces where no one have
            // been before.
            if (StrCol::effectivelyDifferent (macro->getBody (), 
                                              prev->getBody ())) {
                *_err << macro->location () << sev_warning 
                      << "Redefinition of macro `" 
                      << macro->getName () << "'" << endMessage;
                if (~(prev->location ().filename ()))
                  *_err << prev->location () << "previous definition here" 
                        << endMessage;
            }
        }
            
        // Remove the previous version of the macro.
        kill (macro->getName ());
    }

    // Add the macro to the macro table.
    add (macro);
 }


// Remove macro `name' from macro table.
void PreMacroManager::removeMacro (const DString& name)
 { 
    if (name.empty())
        return;
        
    PreMacro *macro = (PreMacro*) get (name);
   
         assert(macro != getLineMacro() );
         assert(macro != getIncLevelMacro() );
         assert(macro != getDateMacro() );
         assert(macro != getTimeMacro() );
         assert(macro != getFileMacro() );
         
    // Special macro that cannot be undefined.
    if (macro && (macro->isLocked () || macro->isInhibited ()))
        return;
        
    kill (name); 
 }


// Get macro `name' from macro table.
PreMacro* PreMacroManager::getMacro (const DString& name) const
 { 
    if (name.empty())
        return (PreMacro*) 0;
        
    PreMacro *macro = (PreMacro*) get (name);

    // Special macro that cannot be defined and 
    // therefore also not be found.
    if (macro && macro->isInhibited ())
        return (PreMacro*) 0;
    
    return macro; 
 }


} // namespace Puma
