/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _GNU_SOURCE /* strndup() */

#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdbool.h>
#include <dirent.h> /* PATH_MAX */
#include <errno.h>

#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_file.h"
#include "gettext.h"
#include "cdw_fs.h"
#include "cdw_list_display.h"



/**
   \file cdw_file.c

   Data structure for storing information about files existing in file system.
   This module provides two basic functions:
   \li cdw_file_new() - constructor
   \li cdw_file_delete() - destructor

   two utility functions:
   \li cdw_file_duplicate()
   \li cdw_file_equal() - a predicate

   and two special functions:
   \li cdw_file_dealloc_files_from_list() - used when deallocating a list of files
   \li cdw_file_display_file() - used by list display widget

   List of arguments of cdw_file_new() constructor reflects its main usage.
   The function is used to create file object when caller has a path to
   a directory, and a result of scandir() call, i.e. a list of file names in
   the directory. Thus constructor is called with dirpath and filename.

   As a result values of type cdw_file_t should (must?) be used only for
   real files existing in file system.
*/



static cdw_file_t *cdw_file_new_base(void);
static cdw_rv_t    cdw_file_new_stat(cdw_file_t *file);
static cdw_rv_t    cdw_file_new_fullpath(cdw_file_t *file, const char *dirpath, const char *name);


/**
   \brief Create new "file" variable

   Create new variable describing file specified by \p dirpath and
   \p name. Fields of the variable describe the file.
   If given file is a regular file (or link) then \p dirpath should describe
   full directory path to directory, in which the file is located, and
   \p name should be the name of the file in this directory.
   If given file is a directory, then \p name should be name of this
   concrete directory, and \p dirpath should be path to its parent
   directory.

   The fact that there are two arguments: \p dirpath and \p name, is a result
   of context where the function is used. It is used in
   cdw_fs_copy_dirent_to_list(list, dirpath, struct dirent **eps, ),
   where there is one dirpath, and set of filenames stored in
   "struct dirent **eps". cdw_fs_copy_dirent_to_list() calls cdw_file_new()
   for each item in "eps" to create new file. This is why there are two
   arguments to cdw_file_new(), instead of one: fullpath.

   \param dirpath - full path to parent directory in which the file is located
   \param name - name of the file in the parent directory

   \return pointer to new "file" variable on success
   \return NULL on failure
*/
cdw_file_t *cdw_file_new(const char *dirpath, const char *name)
{
	cdw_assert (dirpath != (const char *) NULL, "ERROR: passed to the function NULL as dirpath\n");

	cdw_file_t *file = (cdw_file_t *) NULL;
	file = cdw_file_new_base();
	if (file == (cdw_file_t *) NULL) {
		cdw_vdm ("ERROR: failed to create new file with cdw_file_new()\n");
		return (cdw_file_t *) NULL;
	}

	cdw_rv_t crv = cdw_file_new_fullpath(file, dirpath, name);
	cdw_assert (crv == CDW_OK, "ERROR: call to cdw_file_new_fullpath() failed\n");

	crv = cdw_file_new_stat(file);
	cdw_assert (crv == CDW_OK, "ERROR: call to cdw_file_new_stat() failed\n");

	if (!strcmp(file->fullpath + file->name_start, "../")
	    || !strcmp(file->fullpath + file->name_start, "..")) {
		file->is_ref_to_parent_dir = true;
	} else {
		file->is_ref_to_parent_dir = false;
	}

	if (file->type == CDW_FS_DIR) {
		crv = cdw_fs_correct_dir_path_ending(&(file->fullpath));
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to assure correct ending of dirpath (\"%s\")\n", file->fullpath);
			cdw_file_delete(&file);
			return (cdw_file_t *) NULL;
		}
	}

	return file;
}





/**
   \brief Call lstat()and stat() for given file

   Call lstat() and stat() for given file to check its parameters, like
   file validity, file type (dir/file/link), file size.

   \return CDW_OK
*/
cdw_rv_t cdw_file_new_stat(cdw_file_t *file)
{
	/* 1: call lstat() to catch links: in case if fullpath is
	   link, the function examines links, not target of link  */

	struct stat stbuf;
	int rv = lstat(file->fullpath, &stbuf);
	if (rv != -1) {
		if (S_ISLNK(stbuf.st_mode)) {
			file->link = true;
			cdw_sdm ("IS a link: \"%s\"\n", file->fullpath);
		} else {
			file->link = false;
			cdw_sdm ("IN NOT a link: \"%s\"\n", file->fullpath);
		}
		file->invalid = false;
	} else {
		file->invalid = true;
		/* may happen when we create new file object that is not
		   related to any file in file system, i.e. there is a
		   fullpath (created for some reason), but the file itself
		   doesn't exist */
		/* perror("-1 (1) because"); */
		cdw_vdm ("ERROR: lstat() returns -1 for fullpath = \"%s\"\n", file->fullpath);
	}

	/* 2: stat() called for links examines targets */
	rv = stat(file->fullpath, &stbuf);
	int e = errno;
	if (rv == -1) {
		file->invalid = true;
		cdw_vdm ("WARNING: stat() failed for fullpath \"%s\" (calling stat() for links examines target); error = \"%s\"\n", file->fullpath, strerror(e));
		/* TODO: check how caller of this function behaves when CDW_SYS_ERROR is returned */
		/* return CDW_SYS_ERROR; */
	} else {
		file->invalid = false;
	}

	if (S_ISDIR(stbuf.st_mode)) {
		file->size = 0;
		file->type = CDW_FS_DIR;
	} else if (S_ISREG(stbuf.st_mode)) {
		file->size = stbuf.st_size;;
		file->type = CDW_FS_FILE;
	} else if (S_ISLNK(stbuf.st_mode)) {
		file->size = 0;
		file->type = CDW_FS_LINK;
	} else {
		/* char device, block device, named pipe, socket */
		file->size = 0;
		file->type = CDW_FS_OTHER;
	}

	return CDW_OK;
}





/**
   \brief Create correct fullpath for given file

   Function concatenates \p dirpath and \p name, making sure that there
   is exactly one slash between them. The function also sets correct value
   of "name_start" field and - if needed - of printable_fullpath field.

   Function does not check if ending slash is missing if result fullpath
   is a path to a dir, because the function can't tell if given file is
   a dir or not. So resulting fullpath is not always 100% correct.

   \param file - file in which a fullpath (and other fields) needs to me set
   \param dirpath - path to directory, one of two parts of fullpath that needs to be set
   \param name - name of file in directory specified by dirpath; one of two parts of fullpath that needs to be set

   \return CDW_GEN_ERROR on problems
   \return CDW_OK on success
*/
cdw_rv_t cdw_file_new_fullpath(cdw_file_t *file, const char *dirpath, const char *name)
{
	char *correct_dirpath = strdup(dirpath);
	cdw_rv_t crv = cdw_fs_correct_dir_path_ending(&correct_dirpath);
	cdw_assert (crv == CDW_OK, "ERROR: failed to assure correct ending of dirpath\n");
	file->fullpath = cdw_string_concat(correct_dirpath, name, (char *) NULL);
	free(correct_dirpath);
	correct_dirpath = (char *) NULL;
	if (file->fullpath == (char *) NULL) {
		cdw_assert (0, "ERROR: failed to concatenate dirpath and name: \"%s\" + \"%s\"\n", dirpath, name);
		return CDW_ERROR;
	}

	file->name_start = cdw_fs_get_filename_start(file->fullpath);
	if (file->name_start < 0) {
		cdw_vdm ("ERROR: failed to get start of file name in fullpath \"%s\"\n", file->fullpath);
		free(file->fullpath);
		file->fullpath = (char *) NULL;
		assert (0);
		return CDW_ERROR;
	}

	file->printable_fullpath = cdw_string_get_printable_if_needed(file->fullpath);

#ifndef NDEBUG
	if (file->printable_fullpath != (char *) NULL) {
		ssize_t i = cdw_fs_get_filename_start(file->fullpath);
		ssize_t j = cdw_fs_get_filename_start(file->printable_fullpath);
		cdw_assert (i == j, "ERROR: wrong assumption about file name start in path and printable path\n");
	}
#endif

	return CDW_OK;
}





/**
   \brief Allocate memory for "file" variable, initialize it

   Function allocates memory for "file" variable and
   initializes some of its fields to some default values.

   The function does not allocate space for file fullpath nor file name.

   \return pointer to newly allocated varible on success
   \return NULL on failure
*/
cdw_file_t *cdw_file_new_base(void)
{
	cdw_file_t *file = (cdw_file_t *) malloc (sizeof (cdw_file_t));
	if (file == (cdw_file_t *) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for new file\n");
		return (cdw_file_t *) NULL;
	} else {
		file->fullpath = (char *) NULL;
		file->printable_fullpath = (char *) NULL;
		file->name_start = -1;
		file->size = 0;
		file->invalid = true;
		file->link = false;
		file->type = CDW_FS_OTHER;
		file->is_ref_to_parent_dir = false;

		return file;
	}
}





/**
   \brief Duplicate "file" variable

   Function creates exact duplicate of given \file. Values of
   all fields are copied to new "file" variable, and pointer to
   this new variable is returned.

   \return pointer to new variable being copy of argument - on success
   \return NULL on failure
*/
cdw_file_t *cdw_file_duplicate(const cdw_file_t *file)
{
	cdw_assert (file != (cdw_file_t *) NULL, "ERROR: passing NULL pointer for duplication\n");
	cdw_assert (file->fullpath + file->name_start != (char *) NULL, "ERROR: passing file with NULL name for duplication\n");
	cdw_assert (file->fullpath != (char *) NULL, "ERROR: passing file with NULL fullpath for duplication\n");

	cdw_file_t *copy = (cdw_file_t *) malloc(sizeof(cdw_file_t));
	if (copy == (cdw_file_t *) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for duplicate of file\n");
		return (cdw_file_t *) NULL;
	}

	if (file->printable_fullpath != (char *) NULL) {
		copy->printable_fullpath = strdup(file->printable_fullpath);
	} else {
		copy->printable_fullpath = (char *) NULL;
	}
	copy->name_start = file->name_start;

	copy->fullpath = strdup(file->fullpath);
	if (copy->fullpath == (char *) NULL) {
		cdw_file_delete(&copy);

		cdw_vdm ("ERROR: failed to duplicate fullpath of file\n");
		return (cdw_file_t *) NULL;
	}

	copy->type = file->type;
	copy->size = file->size;
	copy->link = file->link;
	copy->invalid = file->invalid;
	copy->is_ref_to_parent_dir = file->is_ref_to_parent_dir;

	return copy;
}





/**
   \brief Deallocate all resources referenced to by "file" variable

   Function frees() all memory referenced to by fields of \p file, and frees
   \p file itself. NULL is assigned to \p file before return, so that caller
   doesn't have to do this himself.

   \return CDW_ERROR if \p file is a pointer to NULL file
   \return CDW_OK on success
*/
cdw_rv_t cdw_file_delete(cdw_file_t **file)
{
	cdw_assert (file != (cdw_file_t **) NULL, "ERROR: passing null pointer to the function\n");

	if (*file == (cdw_file_t *) NULL) {
		cdw_vdm ("ERROR: passed pointer to NULL file to function\n");
		return CDW_ERROR;
	}

	if ((*file)->printable_fullpath != (char *) NULL) {
		free((*file)->printable_fullpath);
		(*file)->printable_fullpath = (char *) NULL;
	}

	if ((*file)->fullpath == (char *) NULL) {
		cdw_vdm ("WARNING: passed to function file with null fullpath\n");
	} else {
		free((*file)->fullpath);
		(*file)->fullpath = (char *) NULL;
	}

	free(*file);
	*file = (cdw_file_t *) NULL;

	return CDW_OK;
}





/**
   \brief Compare all fields of two "file" variables

   Function compares values of all fields of given "file" variables.

   The function may be used as argument to cdw_dll_append()

   \return true if all fields of the variables have the same values
   \return false otherwise
*/
bool cdw_file_equal(const void *_file1, const void *_file2)
{
	cdw_assert (_file1 != (const cdw_file_t *) NULL, "ERROR: first argument is null\n");
	cdw_assert (_file2 != (const cdw_file_t *) NULL, "ERROR: second argument is null\n");

	const cdw_file_t *file1 = (const cdw_file_t *) _file1;
	const cdw_file_t *file2 = (const cdw_file_t *) _file2;

	cdw_assert (file1->name_start > 0, "ERROR: file1->name_start <= 0: %zd\n", file1->name_start);
	cdw_assert (file2->name_start > 0, "ERROR: file1->name_start <= 0: %zd\n", file2->name_start);
	cdw_assert (file1->fullpath != (char *) NULL, "ERROR: file1->fullpath is NULL\n");
	cdw_assert (file2->fullpath != (char *) NULL, "ERROR: file2->fullpath is NULL\n");

	if (file1 == (const cdw_file_t *) NULL || file2 == (const cdw_file_t *) NULL) {
		cdw_vdm ("ERROR: one (or two) file was null\n");
		return false;
	}

	if (file1->fullpath == (char *) NULL || file2->fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: one (or two) fullpaths was null\n");
		return false;
	} else {
		if (!strcmp(file1->fullpath, file2->fullpath)) {
			;
		} else {
			cdw_sdm ("INFO: files have different fullpaths\n%s\n%s\n", file1->fullpath, file2->fullpath);
			return false;
		}
	}


	/* values of file names were tested when testing fullpaths */
	if (file1->fullpath + file1->name_start == (char *) NULL
	    || file2->fullpath + file2->name_start == (char *) NULL) {

		cdw_vdm ("ERROR: one (or two) file names was null\n");
		return false;
	}

	if (file1->printable_fullpath == (char *) NULL && file2->printable_fullpath == (char *) NULL) {
		/* null "printable_fullpath" strings are allowed */
	} else if (file1->printable_fullpath != (char *) NULL && file2->printable_fullpath != (char *) NULL) {
		if (!strcmp(file1->printable_fullpath, file2->printable_fullpath)) {
			;
		} else {
			cdw_sdm ("INFO: files have different printable fullpaths\n");
			return false;
		}
	} else {
		cdw_sdm ("INFO: one printable fullpath is is null, the other is non-null\n");
		return false;

	}


	if (file1->name_start != file2->name_start) {
		cdw_sdm ("INFO: file differ in name starts\n");
		return false;
	}
	if (file1->is_ref_to_parent_dir != file2->is_ref_to_parent_dir) {
		cdw_sdm ("INFO: files differ in \"is ref to parent dir\" values\n");
		return false;
	}
	if (file1->type != file2->type) {
		cdw_sdm ("INFO: files differ in file types\n");
		return false;
	}
	if (file1->size != file2->size) {
		cdw_sdm ("INFO: files differ in file sizes\n");
		return false;
	}
	if (file1->link != file2->link) {
		cdw_sdm ("INFO: files differ in \"link\" values\n");
		return false;
	}
	if (file1->invalid != file2->invalid) {
		cdw_sdm ("INFO: files differ in \"invalid\" values\n");
		return false;
	}
	return true;
}





/**
   \brief Display in given window one file and its attributes

   Display in given \p row of given \p display one file. Set font/background
   of the file properly (based on value of \p isreverse), display additional
   file attributes in the same \p row.

   \p row is a number of row, in which given file should be displayed

   \p isreverse controls attributes of displayed item:
   \li if set to true then item is displayed with foreground and background
   attributes reversed compared to whole file selector window
   \li if set to false then item is displayed with the same foreground and
   background attributes as whole file selector window

   \param display - display to be used to show the file
   \param row - number of row, in which to display file
   \param data - directory entry (file or dir) item with file properties (name, type, size)
   \param isreverse - controls attributes of displayed item
*/
void cdw_file_display_file(void *display, void *data, size_t row, size_t h_offset, bool isreverse)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "display is NULL\n");
	CDW_LIST_DISPLAY *displ = (CDW_LIST_DISPLAY *) display;
	cdw_assert (displ->subwindow != (WINDOW *) NULL, "subwindow of display is NULL\n");

	int n_cols = getmaxx(displ->subwindow);
	int n_rows = getmaxy(displ->subwindow);
	cdw_assert (n_cols > 0, "we have subwindow with 0 columns, or null subwindow\n");
	cdw_assert (n_rows > (int) row, "row number larger than number of rows in subwindow\n");
	cdw_assert (data != (void *) NULL, "you try to display null file\n");
	cdw_file_t *file = (cdw_file_t *) data;

	/* this fills space between item name on the left side of the
	   window and item attributes on the right side of the window */
	mvwhline(displ->subwindow, (int) row, 0, ' ', n_cols);

	char *string = (char *) NULL;
	if (displ->display_format == CDW_LIST_DISPLAY_FORMAT_SHORT) {
		if (file->printable_fullpath != (char *) NULL) {
			string = file->printable_fullpath + file->name_start;
		} else {
			string = file->fullpath + file->name_start;
		}
	} else if (displ->display_format == CDW_LIST_DISPLAY_FORMAT_LONG) {
		cdw_assert (file->fullpath != (char *) NULL, "using file without fullpath\n");
		if (file->printable_fullpath != (char *) NULL) {
			string = file->printable_fullpath;
		} else {
			string = file->fullpath;
		}
	} else {
		cdw_assert (0, "ERROR: no code to handle display format %d\n", displ->display_format);
	}

	long unsigned int attr = 0;
	if (file->type == CDW_FS_DIR) {
		if (isreverse) {
			attr = A_BOLD | A_REVERSE | COLOR_PAIR(displ->colors);
		} else {
			attr = A_BOLD | A_NORMAL | COLOR_PAIR(displ->colors);
		}
	} else {
		if (isreverse) {
			attr = A_REVERSE | COLOR_PAIR(displ->colors);
		} else {
			attr = A_NORMAL | COLOR_PAIR(displ->colors);
		}
	}
	(void) wattrset(displ->subwindow, attr);

	char format[10];
	if (file->invalid) {
		strcpy(format, "!%.*s");
	} else {
		if (file->link == true) {
			strcpy(format, "~%.*s");
		} else {
			strcpy(format, "%.*s");
		}
	}
	if (strlen(string) >= h_offset) {
		mvwprintw(displ->subwindow, (int) row, 0, format, n_cols - 7, string + h_offset);
	} else {
		mvwprintw(displ->subwindow, (int) row, 0, " ");
	}

	/* testing code */
#if 0
	if (string[0] == 'S') { /* I have created local files with names with special chars, that start with 'S' */
		wchar_t dest[1000];
		size_t conv_len = mbstowcs(dest, string, 50);
		size_t len = strlen(string);
		fprintf(stderr, "analyzing \"%s\"\n", string);
		for (size_t i = 0; i < len; i++) {
			fprintf(stderr, " char: %u - > '%c'\n", (unsigned char) string[i], string[i]);
		}
		fprintf(stderr, " char: strlen(string) = %zd\n", strlen(string));
		for (size_t i = 0; i < len; i++) {
			fprintf(stderr, " wchar: %u - > '%c'\n", (unsigned char) dest[i], dest[i]);
			//perror("d");
		}
		fprintf(stderr, " wchar: wcslen(dest) = %zd, mbstowcs(): %zd\n\n\n", wcslen(dest), conv_len);
	}
#endif

	if (file->invalid) {
		/* 2TRANS: "E" stands for "Error" - this label marks invalid
		   item; please keep as short as possible */
		mvwprintw(displ->subwindow, (int) row, n_cols - 5, _("E"));
	} else {
		if (file->type == CDW_FS_DIR) {
			mvwprintw(displ->subwindow, (int) row, n_cols - 5,
				  /* 2TRANS: this is label indicating that labeled item
				     is a directory; please keep as short as possible */
				  _("DIR"));

		} else if (file->type == CDW_FS_FILE) {
			float size = (float) (file->size >= 1048576 ? /* 1024 * 1024 */
					      (float) file->size / 1048576.0 :
					      (float) file->size / 1024.0);
			mvwprintw(displ->subwindow, (int) row, n_cols - 8, "%6.1f%s", size,
				  (file->size >= 1048576 ?
				   /* 2TRANS: suffix for megabytes,
				      please keep as short as possible */
				   _("M") :
				   /* 2TRANS: suffix for kilobytes,
				      please keep as short as possible */
				   _("K")));
		} else { /* currently unsupported file type */
			/* 2TRANS: "n/a" means here that there is no label suitable
			   for description of given item; keep as short as possible */
			mvwprintw(displ->subwindow, (int) row, n_cols - 5, _("n/a"));
		}
	}
	(void) wattrset(displ->subwindow, A_NORMAL | COLOR_PAIR(displ->colors));

	return;
}







/**
   \brief Deallocate all files that are on given list

   Function frees all memory associated with all 'file' variables stored on
   given list. The list itself is not affected, only payload of
   the list is free()d.

   \param list - list of files, from which you want to dealloc all files

   \return CDW_OK
*/
cdw_rv_t cdw_file_dealloc_files_from_list(cdw_dll_item_t *list)
{
	if (list != (cdw_dll_item_t *) NULL) {
		cdw_dll_item_t *item = list;
		for ( ; item != (cdw_dll_item_t *) NULL; item = item->next) {
			cdw_file_delete((cdw_file_t **) &(item->data));
		}
	} else {
		cdw_vdm ("INFO: passed NULL list to the function\n");
	}

	return CDW_OK;
}






#ifdef CDW_UNIT_TEST_CODE


/* *********************** */
/* *** unit tests code *** */
/* *********************** */


static void     test_cdw_file_equal(void);
static void     test_cdw_file_new_base(void);
static void     test_cdw_file_new_file_delete(void);
static void     test_cdw_file_duplicate(void);
static cdw_rv_t test_cdw_file_new_helper(char **cwd, char **dirpath, char **filename);
static void     test_cdw_file_new_fullpath(void);


void cdw_file_run_tests(void)
{
	fprintf(stderr, "testing cdw_file.c\n");

	test_cdw_file_equal();
	test_cdw_file_new_base();
	test_cdw_file_new_file_delete();
	test_cdw_file_duplicate();
	test_cdw_file_new_fullpath();

	fprintf(stderr, "done\n\n");

	return;
}





void test_cdw_file_new_base(void)
{
	fprintf(stderr, "\ttesting cdw_file_new_base()... ");

	cdw_file_t *file = cdw_file_new_base();

	cdw_assert_test (file != (cdw_file_t *) NULL, "ERROR: function cdw_file_new_base() failed to allocate file\n");
	cdw_assert_test (file->fullpath == (char *) NULL, "ERROR: fullpath in new file base is not NULL\n");
	cdw_assert_test (file->name_start == -1, "ERROR: name start is -1 by default\n");

	cdw_assert_test (file->size == 0, "ERROR: size of new file is not 0 by default\n");
	cdw_assert_test (file->invalid == true, "ERROR: file is not invalid by default\n");
	cdw_assert_test (file->link == false, "ERROR: file is link by default\n");
	cdw_assert_test (file->type == CDW_FS_OTHER, "ERROR: file is not CDW_FS_OTHER by default\n");
	cdw_assert_test (file->is_ref_to_parent_dir == false, "ERROR: file is not ref to parent by default\n");

	cdw_file_delete(&file);

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_file_equal(void)
{
	fprintf(stderr, "\ttesting cdw_file_equal()... ");

	cdw_file_t file1, file2;
	bool test;
	cdw_rv_t crv;

	file1.fullpath = strdup("/home/acerion/");
	file2.fullpath = strdup("/home/acerion/");
	file1.printable_fullpath = (char *) NULL;
	file2.printable_fullpath = (char *) NULL;
	file1.name_start = 6;
	file2.name_start = 6;
	file1.type = CDW_FS_FILE;
	file2.type = CDW_FS_FILE;
	file1.size = 123456;
	file2.size = 123456;
	file1.link = true;
	file2.link = true;
	file1.invalid = true;
	file2.invalid = true;
	file1.is_ref_to_parent_dir = true;
	file2.is_ref_to_parent_dir = true;

	/* at the beginning the files are equal */
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: equal files recognized as not equal\n");


	/* testing with different "fullpath" field values */
	crv = cdw_string_set(&(file1.fullpath), "/my/test/path/1/");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't set new file fullpath\n");

	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different fullpaths are recognized as equal\n");

	crv = cdw_string_set(&(file1.fullpath), "/home/acerion/");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't restore old file fullpath\n");

	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: test failed after restoring original fullpath in file1\n");


	/* testing with different "printable_fullpath" field values */
	crv = cdw_string_set(&(file1.printable_fullpath), "my test string");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't set file1 printable fullpath\n");
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different printable fullpaths are recognized as equal (1)\n");

	crv = cdw_string_set(&(file2.printable_fullpath), "let there be chars");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't set file2 printable fullpath (1)\n");
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different printable fullpaths are recognized as equal (2)\n");

	crv = cdw_string_set(&(file2.printable_fullpath), "my test string");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't set file2 printable fullpath (2)\n");
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with equal printable fullpaths are recognized as non-equal\n");


	/* testing with different "type" field values */
	file1.type = CDW_FS_DIR;
	file2.type = CDW_FS_FILE;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different types aren't recognized as different (1)\n");

	file1.type = CDW_FS_DIR;
	file2.type = CDW_FS_DIR;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same types aren't recognized as equal (1)\n");

	file1.type = CDW_FS_LINK;
	file2.type = CDW_FS_LINK;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same types aren't recognized as equal (2)\n");

	file1.type = CDW_FS_LINK;
	file2.type = CDW_FS_DIR;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different types aren't recognized as different (2)\n");

	file1.type = CDW_FS_FILE;
	file2.type = CDW_FS_FILE;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with restored the same types aren't recognized as the same\n");


	/* testing with different "size" field values */
	file1.size = 123456;
	file2.size = 123456;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same sizes are recognized as different (1)\n");

	file1.size = 0;
	file2.size = 0;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same sizes are recognized as different (2)\n");

	file1.size = 654321;
	file2.size = 123456;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different sizes are recognized as the same (1)\n");

	file1.size = 0;
	file2.size = 123456;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different sizes are recognized as the same (2)\n");

	file1.size = 123456;
	file2.size = 123456;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with restored the same sizes are recognized as different\n");


	/* testing with different "link" field values */
	file1.link = true;
	file2.link = true;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same 'link' fields are recognized as different\n");

	file1.link = false;
	file2.link = true;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different 'link' fields are recognized as the same\n");

	file1.link = false;
	file2.link = false;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with restored the same 'link' fields are recognized as different\n");


	/* testing with different "invalid" field values */
	file1.invalid = true;
	file2.invalid = true;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same 'invalid' fields are recognized as different\n");

	file1.invalid = false;
	file2.invalid = true;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different 'invalid' fields are recognized as the same\n");

	file1.invalid = false;
	file2.invalid = false;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with restored the same 'invalid' fields are recognized as different\n");


	/* testing with different "is_ref_to_parent_dir" field values */
	file1.is_ref_to_parent_dir = true;
	file2.is_ref_to_parent_dir = true;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same 'is_ref_to_parent_dir' fields are recognized as different\n");

	file1.is_ref_to_parent_dir = false;
	file2.is_ref_to_parent_dir = true;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different 'is_ref_to_parent_dir' fields are recognized as the same\n");

	file1.is_ref_to_parent_dir = false;
	file2.is_ref_to_parent_dir = false;
	test = cdw_file_equal((const void *) &file1, (const void *) &file2);
	cdw_assert_test (test == true, "ERROR: files with restored the same 'is_ref_to_parent_dir' fields are recognized as different\n");


	/* end, clean up */
	free(file1.fullpath);
	file1.fullpath = (char *) NULL;
	free(file2.fullpath);
	file2.fullpath = (char *) NULL;

	free(file1.printable_fullpath);
	file1.printable_fullpath = (char *) NULL;
	free(file2.printable_fullpath);
	file2.printable_fullpath = (char *) NULL;

	fprintf(stderr, "OK\n");
	return;
}





cdw_rv_t test_cdw_file_new_helper(char **cwd, char **dirpath, char **filename)
{
	cdw_assert_test (*cwd == (char *) NULL, "ERROR: cwd should be pointer to NULL");
	cdw_assert_test (*dirpath == (char *) NULL, "ERROR: dirpath should be pointer to NULL");
	cdw_assert_test (*filename == (char *) NULL, "ERROR: filename should be pointer to NULL");

	/* first get some existing dir path */
	*cwd = (char *) malloc(PATH_MAX); /* PATH_MAX alreay includes ending null */
	cdw_assert_test (*cwd != (char *) NULL, "ERROR: failed to allocate memory for cwd\n");

	char *res = getcwd(*cwd, PATH_MAX - 1);
	cdw_assert_test (res != (char *) NULL, "ERROR: failed to get current working directory (1)\n");
	cdw_assert_test (res == *cwd, "ERROR: failed to get current working directory (2)\n");
	cdw_rv_t crv = cdw_fs_correct_dir_path_ending(cwd);
	cdw_assert_test (crv == CDW_OK, "ERROR: failed to ensure proper ending of cwd\n");


	ssize_t start = cdw_fs_get_filename_start(*cwd);
	cdw_assert_test (start >= 0, "ERROR: failed to get start of file name in path \"%s\"\n", *cwd);
	// cdw_vdm ("INFO: cwd = \"%s\", start = %d\n", *cwd, start);
	*filename = strdup(*cwd + start);
	*dirpath = strndup(*cwd, (size_t) start);
	cdw_assert_test (*filename != (char *) NULL && *dirpath != (char *) NULL,
			 "ERROR: failed to strdup() filename and dirpath\n");
	// cdw_vdm ("INFO: dirtpath = \"%s\"\n", *dirpath);
	// cdw_vdm ("INFO: filename = \"%s\"\n", *filename);

	return CDW_OK;
}




void test_cdw_file_new_file_delete(void)
{
	fprintf(stderr, "\ttesting cdw_file_new() and cdw_file_delete()... ");

	/* prepare arguments for cdw_file_new() */

	char *cwd = (char *) NULL;
	char *dirpath = (char *) NULL;
	char *filename = (char *) NULL;
	test_cdw_file_new_helper(&cwd, &dirpath, &filename);

	/* create file and test it */

	cdw_file_t *file = cdw_file_new(dirpath, filename);
	cdw_assert_test (file != (cdw_file_t *) NULL, "ERROR: function cdw_file_new_base() failed to allocate file\n");

	cdw_assert_test (file->fullpath != (char *) NULL, "ERROR: fullpath in new file base is NULL\n");
	cdw_assert_test (!strcmp(file->fullpath, cwd), "ERROR: incorrect fullpath: should be \"%s\", is: \"%s\"\n", cwd, file->fullpath);
	cdw_assert_test (!strcmp(file->fullpath + file->name_start, filename), "ERROR: incorrect name: should be \"%s\", is: \"%s\"\n", filename, file->fullpath + file->name_start);
	cdw_assert_test (file->size == 0, "ERROR: size of new file is not 0 for dir\n");
	cdw_assert_test (file->invalid == false, "ERROR: existing file is invalid\n");
	cdw_assert_test (file->link == false, "ERROR: dir is link\n");
	cdw_assert_test (file->type == CDW_FS_DIR, "ERROR: dir is not recognized as dir\n");
	cdw_assert_test (file->is_ref_to_parent_dir == false, "ERROR: current dir should not be reference to parent dir\n");

	/* now test cdw_file_delete(), but there isn't much to test */

	// char **f = &(file->fullpath);
	cdw_file_delete(&file);
	cdw_assert_test (file == (cdw_file_t *) NULL, "ERROR: failed to correctly delete file, file is not NULL\n");
	/* on some occasions this assertion fails, but valgrind doesn't
	   complain about leaking memory related to file->fullpath */
	// cdw_assert_test (*f == (char *) NULL, "ERROR: failed to correctly nullify file->fullpath, it doesn't point to NULL\n");


	free(cwd);
	cwd = (char *) NULL;
	free(filename);
	filename = (char *) NULL;
	free(dirpath);
	dirpath = (char *) NULL;

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_file_duplicate(void)
{
	fprintf(stderr, "\ttesting cdw_file_duplicate()... ");

	/* prepare arguments for cdw_file_new() */

	char *cwd = (char *) NULL;
	char *dirpath = (char *) NULL;
	char *filename = (char *) NULL;
	test_cdw_file_new_helper(&cwd, &dirpath, &filename);

	/* create file and test it (we tested creating file before, but just in case... */

	cdw_file_t *file = cdw_file_new(dirpath, filename);
	cdw_assert_test (file != (cdw_file_t *) NULL, "ERROR: function cdw_file_new_base() failed to allocate file\n");

	cdw_assert_test (file->fullpath != (char *) NULL, "ERROR: fullpath in new file base is NULL\n");
	cdw_assert_test (!strcmp(file->fullpath, cwd), "ERROR: incorrect fullpath: should be \"%s\", is: \"%s\"\n", cwd, file->fullpath);
	cdw_assert_test (!strcmp(file->fullpath + file->name_start, filename), "ERROR: incorrect name: should be \"%s\", is: \"%s\"\n", filename, file->fullpath + file->name_start);
	cdw_assert_test (file->size == 0, "ERROR: size of new file is not 0 for dir\n");
	cdw_assert_test (file->invalid == false, "ERROR: existing file is invalid\n");
	cdw_assert_test (file->link == false, "ERROR: dir is link\n");
	cdw_assert_test (file->type == CDW_FS_DIR, "ERROR: dir is not recognized as dir\n");
	cdw_assert_test (file->is_ref_to_parent_dir == false, "ERROR: current dir should not be reference to parent dir\n");

	/* here is a small trick to test duplicating file with non-zero size;
	   ('file' is a directory in this testcase, and 'size' is zero;
	   I don't want to test copying zero value) */
	file->size = 17345;

	/* now do the real testing */
	cdw_file_t *duplicate = cdw_file_duplicate(file);

	cdw_assert_test (duplicate != (cdw_file_t *) NULL, "ERROR: failed to create duplicate of a file\n");
	cdw_assert_test (!strcmp(file->fullpath, duplicate->fullpath),
			 "ERROR: file and its duplicate have different fullpaths:\nfile->fullpath      = \"%s\"\nduplicate->fullpath = \"%s\"\n",
			 file->fullpath, duplicate->fullpath);
	cdw_assert_test (file->name_start == duplicate->name_start,
			 "ERROR: file and its duplicate have different name starts:\nfile->name_start      = \"%zd\"\nduplicate->name_start = \"%zd\"\n",
			 file->name_start, duplicate->name_start);
	cdw_assert_test (file->size == duplicate->size,
			 "ERROR: file and its duplicate have different file sizes:\nfile->size      = %lld\nduplicate->size = %lld\n",
			 file->size, duplicate->size);
	cdw_assert_test (file->invalid == duplicate->invalid,
			 "ERROR: file and its duplicate have different \"invalid\" field values:\nfile->invalid      = %s\nduplicate->invalid = %s\n",
			 file->invalid ? "true" : "false", duplicate->invalid ? "true" : "false");
	cdw_assert_test (file->link == duplicate->link,
			 "ERROR: file and its duplicate have different \"link\" field values:\nfile->link      = %s\nduplicate->link = %s\n",
			 file->link ? "true" : "false", duplicate->link ? "true" : "false");
	cdw_assert_test (file->type == duplicate->type,
			 "ERROR: file and its duplicate have different \"type\" field values:\nfile->link      = %d\nduplicate->link = %d\n",
			 file->type, duplicate->type);
	cdw_assert_test (file->is_ref_to_parent_dir == duplicate->is_ref_to_parent_dir,
			 "ERROR: file and its duplicate have different \"is reference to parent dir\" field values\n");

	cdw_file_delete(&file);
	cdw_assert_test (file == (cdw_file_t *) NULL, "ERROR: failed to correctly delete file, file is not NULL\n");
	cdw_file_delete(&duplicate);
	cdw_assert_test (duplicate == (cdw_file_t *) NULL, "ERROR: failed to correctly delete file duplicate, it is not NULL\n");

	free(cwd);
	cwd = (char *) NULL;
	free(filename);
	filename = (char *) NULL;
	free(dirpath);
	dirpath = (char *) NULL;

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_file_new_fullpath(void)
{
	fprintf(stderr, "\ttesting cdw_file_new_fullpath()... ");

	/* preparation, set only fields that will be needed to test function */
	cdw_file_t file;
	file.printable_fullpath = (char *) NULL;
	file.name_start = -1;
	file.type = CDW_FS_DIR;
	char *dirpath = strdup("/this/is/a/test/");
	/* this assignment causes compiler warning, but it is intentional:
	   this char will be non-printable, but it is necessary if you want
	   to test creating printable fullpath */
	dirpath[12] = 0xFF;
	char *name = strdup("name");


	/* test */
	cdw_rv_t crv = cdw_file_new_fullpath(&file, dirpath, name);
	cdw_assert_test (crv == CDW_OK, "ERROR: call to cdw_file_new_fullpath() failed\n");
	cdw_assert_test (!strcmp(file.printable_fullpath, "/this/is/a/t?st/name"),
			 "ERROR: printable fullpath is incorrect: \"%s\"\n", file.printable_fullpath);
	cdw_assert_test (file.name_start != -1, "ERROR: name_start is -1\n");
	cdw_assert_test (file.name_start == (ssize_t) strlen(dirpath),
			 "ERROR: incorrect value of name_start: %zd\n", file.name_start);


	/* clean up */
	free(dirpath);
	dirpath = (char *) NULL;
	free(name);
	name = (char *) NULL;
	free(file.fullpath);
	file.fullpath = (char *) NULL;
	if (file.printable_fullpath != (char *) NULL) {
		free(file.printable_fullpath);
		file.printable_fullpath = (char *) NULL;
	}

	fprintf(stderr, "OK\n");
	return;
}


#endif /* CDW_UNIT_TEST_CODE */


