/*
 * Copyright 2010 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codenarc.rule.size

import org.codenarc.rule.AbstractAstVisitorRule
import org.gmetrics.metric.abc.AbcMetric

/**
 * Rule that calculates the ABC Complexity for methods/classes and checks against
 * configured threshold values.
 * <p/>
 * The <code>maxMethodComplexity</code> property holds the threshold value for the ABC complexity value
 * (magnitude) for each method. If this value is non-zero, a method with a cyclomatic complexity value greater than
 * this value is considered a violation. The value does not have to be an integer (i.e., 1.7 is allowed). The
 * <code>maxMethodComplexity</code> property defaults to 60.
 * <p/>
 * The <code>maxClassAverageMethodComplexity</code> property holds the threshold value for the average ABC
 * complexity value for each class. If this value is non-zero, a class with an average ABC complexity
 * value greater than this value is considered a violation. The value does not have to be an integer
 * (i.e., 1.7 is allowed). The <code>maxClassAverageMethodComplexity</code> property defaults to 60.
 * <p/>
 * The <code>ignoreMethodNames</code> property optionally specifies one or more (comma-separated) method
 * names that should be ignored (i.e., that should not cause a rule violation). The name(s) may optionally
 * include wildcard characters ('*' or '?'). Note that the ignored methods still contribute to the class
 * complexity value.
 * <p/>
 * This rule treats "closure fields" as methods. If a class field is initialized to a Closure (ClosureExpression),
 * then that Closure is analyzed and checked just like a method.
 *
 * @see <a href="http://www.softwarerenovation.com/ABCMetric.pdf">ABC Metric specification</a>.
 * @see <a href="http://jakescruggs.blogspot.com/2008/08/whats-good-flog-score.html">Blog post</a> describing guidelines for interpreting an ABC score.
 * @see <a href="http://gmetrics.sourceforge.net/gmetrics-AbcMetric.html">GMetrics ABC metric</a>.
 *
 * @author Chris Mair
 * @version $Revision: 344 $ - $Date: 2010-05-05 21:10:10 -0400 (Wed, 05 May 2010) $
 */
class AbcComplexityRule extends AbstractAstVisitorRule {
    String name = 'AbcComplexity'
    int priority = 2
    Class astVisitorClass = AbcComplexityAstVisitor
    int maxMethodComplexity = 60
    int maxClassAverageMethodComplexity = 60
    String ignoreMethodNames
}

class AbcComplexityAstVisitor extends AbstractMethodMetricAstVisitor  {

    protected Object createMetric() {
        return new AbcMetric()
    }

    protected String getMetricShortDescription() {
        return 'ABC score'
    }

    protected Object getMaxMethodMetricValue() {
        return rule.maxMethodComplexity
    }

    protected Object getMaxClassMetricValue() {
        return rule.maxClassAverageMethodComplexity
    }
}