# parameter.rb : Qt support for parameters.
# Copyright (C) 2006 Vincent Fourmond
 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA




module MetaBuilder

  # An exception raised when the user cancels an input dialog box.
  class CancelInput < Exception
  end
  
  class ParameterType

    # Creates a widget suitable for editing the value and returns it.
    # The widget will be a child of  _parent_ (can be nil).
    # The _target_ is provided for the Parameter system. It should contain
    # the value of the target whose parameter we should set. This can be
    # used for instance when the value has dynamic content depending on
    # some internals of _target_. Reimplementations should not
    # fail if it is +nil+. The _style_ parameter
    # is for now a placeholder. 
    #
    # The created widget must provide the following:
    # * <tt>text=|text</tt>: a writer|reader pair for the string value;
    # * <tt>value=|value</tt>: a writer|reader pair for the real value;
    # * <tt>value_changed()</tt> : a signal for when the value has changed.
    #   should be used in the same way as Qt::LineEdit::editingFinished
    #   in the sense that it should only trigger on complete output.
    # * <tt>label=</tt>: a way to set the label. Label could be basically
    #   anything accepted by Qt; if it is +nil+ or +false+, the display is
    #   switched off. The label display must be switched off by default.
    #
    def qt4_create_input_widget(parent = nil, target = nil, style = nil)
      return Qt4MB::GenericInputWidget.new(parent, self)
    end

    # Pops up a Qt4 dialog box to ask for the value of the parameter.
    # * _parent_ is the parent widget;
    # * _title_ is the title of the Dialog Box
    # * _label_ is the label of the widget used for querying
    # * _default_ is the default value (of the appropriate type).
    # * _target_ is the target for the parameters system. Defaults
    #   to +nil+.
    #
    # Returns directly a value in the right type. The default
    # implementation uses the #qt4_create_input_widget function.
    # As +false+ or +nil+ are appropriate return values in some
    # cases, #qt4_get raises a CancelInput exception. You *must*
    # handle it if you don't want nasty surprises...

    def qt4_get(parent, title, label, default, target = nil)
      return Qt4MB::GenericDialog.get_type(parent, self,
                                           title, label, default, target)
    end

    # Converts from the type to a QVariant
    def type_to_variant(value)
      return Qt::Variant.new(value)
    end

    # Converts from a QVariant to the appropriate (Ruby) type.
    def variant_to_type(variant)
      return variant.value
    end

  end

  class Parameter 
  end

  # This module holds all the classes descending from the Qt4 classes.
  module Qt4MB


    # THIS MUST BE REWRITTEN TO COMPLY WITH THE NEW SPECS !!!
    # A generic input for the types. Based on Qt::LineEdit.
    class GenericInputWidget < Qt::LineEdit

      # Emitted when the value is changed. Connected to
      # editingFinished().
      signals 'value_changed()' 

      # Creates a Generic widget. _type_ is the ParameterType child
      # instance for whom we should do conversion.
      def initialize(parent, type)
        super(parent)
        @type = type
        connect(self, SIGNAL('editingFinished()'),
                SIGNAL('value_changed()'))
      end

      def text=(t)
        setText(t)
      end

      # Returns the value currently displayed
      def value
        return @type.string_to_type(text)
      end

      # Sets the display.
      def value=(val)
        self.text = @type.type_to_string(val) 
      end
    end

    class GenericDialog < Qt::Dialog
      
      # Creates a generic input dialog for the type _type_.
      def initialize(parent, type, title, label, val, target)
        super(parent)
        self.window_title = title
        vbox = Qt::VBoxLayout.new(self)

        hbox = Qt::HBoxLayout.new
        @label = Qt::Label.new(label)
        hbox.add_widget(@label)
        @widget = type.qt4_create_input_widget(nil, target)
        hbox.add_widget(@widget)
        @widget.value = val
        vbox.add_layout(hbox)

        hbox = Qt::HBoxLayout.new
        # OK and Cancel buttons
        button  = Qt::PushButton.new("OK")
        connect(button, SIGNAL('clicked()'),
                SLOT('accept()'))
        hbox.add_widget(button)
        button  = Qt::PushButton.new("Cancel")
        connect(button, SIGNAL('clicked()'),
                SLOT('reject()'))
        hbox.add_widget(button)
        vbox.add_layout(hbox)
        # and that's all ! 
      end

      def exec
        res = super
        if res > 0
          return @widget.value
        else
          raise CancelInput,  "User cancelled input"
        end
      end

      # A convenience wrapper around the functionnality. Creates a
      # GenericDialog and executes it.
      def self.get_type(parent, type, title, label, val, target)
        dialog = new(parent, type, title, label, val, target)
        return dialog.exec
      end

    end

  end

end
