# description.rb : How do classes describe themselves
# Copyright (C) 2006 Vincent Fourmond

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

module SciYAG


  module Descriptions

    # A module to convert text input into an object of the given type.
    module Conversion

      # The regular expression that matches 'true' for boolean values:
      TRUE_RE = /^\s*(true|yes)\s*$/i

      # The regular expression that matches 'false'
      FALSE_RE = /^\s*(false|no(ne)?)\s*$/i

      # Converts +text+ to the given type. It support different schemes:
      # * if +type+ is String, Float, Array of Integer, the values are
      #   converted using the appropriate functions;
      # * if +type+ is a string, the type is considered to be String;
      #   this leaves a possibility to implement a more precise
      #   mechanism for choosing; see Descriptions::Parameter#type
      # * if +type+ is :bool, the text is interpreted as a true/false
      #   expression.
      # * if +type+ is an array (of symbols/strings), then a valid input
      #   is one of the elements of the arrays (it is converted to symbols
      #   if necessary)
      # * if +type+ is a hash, it behaves as if the keys had been
      #   specified as an array. The values can be used to provide
      #   a proper name;
      # * for any other input, +type+ is assumed to be a class, and its
      #   constructor should support build from 1 argument, a String.

      def text_to_type(text, type)
        # warning: case cannot be used, as it is the comparison
        # === which is used, that is is_a?.
        if type == String
          value = String(text)
        elsif type.is_a?(String) # some special specification;
          # always a string
          value = String(text)
        elsif type == Float
          value = Float(text)
        elsif type == Array
          value = Array(text)
        elsif type == Integer 
          value = Integer(text)
        elsif type.is_a?(Array)
          h = {}
          type.each do |a|
            h[a.to_s] = a
          end
          return h[text]        # No checking done...
        elsif type.is_a?(Hash)
          return text_to_type(text, type.keys)
        elsif type == :bool
          if text =~ TRUE_RE
            return true
          else
            return false
          end
        else
          value = type.new(text)
        end
        return value
      end

    end


    # A class that describes one parameter that will be fixed at
    # run-time by the user. 
    class Parameter 
      include Conversion
      # The short name of the parameter
      attr_accessor :name

      # The long name of the parameter, to be translated
      attr_accessor :long_name

      # The function names that should be used to set the symbol and
      # retrieve it's current value. The corresponding functions should
      # read or return a string, and writer(reader) should be a noop.
      attr_accessor :reader_symbol, :writer_symbol

      # The description
      attr_accessor :description

      # The type of the parameter. It can take several values;
      # see Descriptions::Conversion.text_to_type. Moreover, several values
      # for a string parameter can be interpreted by systems in charge
      # of querying the parameter:
      #
      # * <tt>"File: ..."</tt> represents a file, and the rest is the
      #   filter, Qt style;
      # * <tt>"Set"</tt> represents a data set, which can queried for by
      #   the sets_available function.
      attr_accessor :type

      def initialize(name, writer_symbol,
                     reader_symbol,
                     long_name, 
                     type,
                     description)
        @name = name
        @writer_symbol = writer_symbol
        @reader_symbol = reader_symbol
        @description = description
        @long_name = long_name
        @type = type  
      end

      # Sets the value of the given parameter in the _target_. It tries
      # to be clever somehow, using @type to know what should be
      # expected. See the text_to_type function.

      def set(target, value)
        value = text_to_type(value, @type)
        target.send(@writer_symbol, value) 
      end

      # Aquires the value from the backend, and returns it
      def get(target)
        target.send(@reader_symbol).to_s
      end

      def value(v)
        return text_to_type(v, @type)
      end
    end

    # The base class for all descriptions. A description describes
    # a "plugin" class. It has the following attributes:
    #
    # * the basic name, code-like, which is used mainly for internal
    #   purposes;
    # * the long name, which has to be translated
    # * the description itself, some small text describing the nature
    #   of the plugin
    # * a list of parameters the plugin can take, along with their
    #   description. These are Parameter .
    # 
    class Description
      # the class to instantiate.
      attr_accessor :class

      # The name of the backend (short, code-like)
      attr_accessor :name

      # (text) description !
      attr_accessor :description

      # Long name, the one for public display
      attr_accessor :long_name

      # The hash holding parameters. Useful mainly for subclasses
      attr_reader   :param_hash

      # The parameter list
      attr_reader   :param_list

      # The list of parameters that have to be fed into the initialize
      # function.
      attr_accessor :init_param_list


      # Initializes a Description
      def initialize(cls, name, long_name, description = "")
        @class = cls
        @name = name
        @long_name = long_name
        @description = description
        @param_list = []
        @param_hash = {}
        @init_param_list = []
      end

      # Creates an instance of the class, forwards parameters to the
      # initialize method.
      def instantiate(*args)
        return @class.new(*args)
      end

      # Prepares the argument list for instantiate based on
      # init_param_list and the text arguments given:
      def prepare_instantiate_arglist(*args)
        target = []
        for pars in @init_param_list
          target << pars.value(args.shift)
        end
        return target
      end

      def add_param(param)
        @param_list << param

        # Three different cross-linkings.
        @param_hash[param.reader_symbol] = param
        @param_hash[param.writer_symbol] = param
        @param_hash[param.name] = param
      end

      def param_set(i, p)
        param = p
        instance = i
        return proc { |x| param.set(instance,x) }
      end

      # Pushes the names of the params onto @init_list_param_list.
      # Arguments have to be strings.
      def init_params(*args)
        @init_param_list += args
      end

      # Fills an OptionParser with all the parameters the
      # Backend should be able to take.

      def fill_parser(instance, parser, uniquify = true)
        parser_banner(instance, parser)
        parser_options(instance, parser, uniquify)
      end

      # Fills a parser with options (and only that)
      def parser_options(instance, parser, uniquify = true)
        raise "The instance is not of the right class" unless
          instance.is_a? @class
        for param in @param_list
          if uniquify
            param_name = "--#{@name}-#{param.name} #{param.name.upcase}"
          else
            param_name = "--#{param.name} #{param.name.upcase}"
          end
          parser.on(param_name, param.description,
                    param_set(instance, param))
        end
      end

      # The parsers's banner.
      def parser_banner(instance, parser) 
        # nothing by default
      end

      # Creates a parser entry in +parser+ that creates a new instance of the
      # description's class and feed it to the +result+ method of
      # the +receiver+. You can specify an optionnal +prefix+ for the
      # option's name.
      def parser_instantiate_option(parser, receiver, result, prefix = "")
        op_name = "--#{prefix}#{@name}"
        for arg in @init_param_list
          op_name += " " + arg.name.upcase
        end
        # cool !
        parser.on(op_name, @description) do |*a|
          b = prepare_instantiate_arglist(*a)
          instance = instantiate(*b)
          receiver.send(result, instance)
        end
      end

    end

    # This module should be used with +include+ to provide the class with
    # descriptions functionnalities. You also need to +extend+
    # DescriptionExtend
    module DescriptionInclude
      # Returns the description associated with the backend object. Actually
      # returns the one associated with the class.
      def description
        return self.class.description
      end
      
      # Fills an OptionParser with their parameters. Most probably, the
      # default implementation should do for most cases. _uniquify_ asks
      # if we should try to make the command-line options as unique as
      # reasonable to do ?
      def fill_parser(parser, uniquify = true)
        description.fill_parser(self, parser, uniquify)
      end
      
      def parser_banner(parser)
        description.parser_banner(self,parser)
      end
      
      def parser_options(parser, uniquify = true)
        description.parser_options(self,parser, uniquify)
      end
      
      # Returns the long name of the backend:
      def long_name
        return description.long_name
      end
    end

    # This module should be used with +extend+ to provide the class with
    # descriptions functionnalities. You also need to +include+
    # DescriptionInclude. Please not that all the *instance* methods
    # defined here will become *class* methods there.
    module DescriptionExtend

      # Returns the description of the class.
      def description
        return @description
      end

      # Sets the description of the class. Classes should provide
      # an easier interface for that.
      def set_description(desc)
        @description = desc
      end

      # Like param, but doesn't define an accessor.
      def param_noaccess(writer, reader, name, long_name, type = String, 
                         desc = "")
        raise "Use describe first" if description.nil? 
        param = Descriptions::Parameter.new(name, writer, reader,
                                            long_name, 
                                            type, desc)
        description.add_param(param)
        return param
      end

      # Tells if the current object is a base ancestor ?
      def base_ancestor?
        if superclass.respond_to?(:lookup_description_extend_ancestor)
          return false
        else
          return true
        end
      end

      # Retrieves the most ancient ancestor that has included
      # DescriptionExtend.
      def lookup_description_extend_ancestor
        t = self
        while not t.base_ancestor?
          t = t.superclass
        end
        return t
      end

      # Valid only in the base class
      def description_list_base
        if base_ancestor?
          return @description_list
        else
          raise "This function should only be called from the " +
            "base ancestor"
        end
      end

      # Valid only in the base class
      def set_description_list_base()
        @description_list = []
      end


      # Returns the list of descriptions associated with the base
      # class
      def description_list
        return lookup_description_extend_ancestor.description_list_base
      end

      # Valid only in the base class
      def description_hash_base
        if base_ancestor?
          return @description_hash
        else
          raise "This function should only be called from the " +
            "base ancestor"
        end
      end

      # Valid only in the base class
      def set_description_hash_base()
        @description_hash = {}
      end

      # Returns the hash of descriptions associated with the base
      # class
      def description_hash
        return lookup_description_extend_ancestor.description_hash_base
      end


      # Registers a description in the base ancestor.
      def register_description(desc)
        description_list << desc
        description_hash[desc.name] = desc
      end

      # Redefined so that we can add the description lists only in the
      # base ancestor.
      def DescriptionExtend.extend_object(t)
        # First call super, to make sure the methods are here !
        super
        if t.base_ancestor?
          t.set_description_list_base
          t.set_description_hash_base
        end
      end

      
      # This shortcut creates an accessor for the given symbol and registers
      # it at the same time. Use if *after* describe. Something like
      #
      #  param :size, "size", "Size", Integer , "The size of the backend"
      #
      # should be perfectly fine.
      def param(symbol, name, long_name, type = String, 
                desc = "")
        # We use the accessor = symbol.
        param_noaccess((symbol.to_s + "=").to_sym, 
                       symbol, name, long_name, 
                       type, desc)
        # Creates an accessor
        attr_accessor symbol
      end

      # The parameters the class should inherit from its direct parents
      # (which could have in turn inherited them...).
      def inherit(*names)
        if self.superclass.respond_to?(:description)
          parents_params = self.superclass.description.param_hash
          for n in names
            if parents_params.key?(n)
              description.add_param(parents_params[n])
            else
              warn "Param #{n} not found"
            end
          end
        else
          warn "The parent class has no description"
        end
      end

      # Adds the params to the list of init params. Better use init_param
      def init_params(*params)
        description.init_params(*params)
      end

      # Creates a parameter which has to be used for instancitation
      def init_param(name, long_name, type, desc)
        p = param_noaccess(nil, nil, name, long_name, type, desc)
        init_params(p)
      end

    end

  end
end

