# strings.rb : Different ParameterTypes to deal with different styles items,
# such as colors, line types, fonts ???
# Copyright (C) 2006 Vincent Fourmond
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA


module MetaBuilder

  # The module ParameterTypes should be used for all subclasses of
  # ParameterType, to keep the place clean and tidy.
  module ParameterTypes

    # A HTML color: the value is returned in the form of an array
    # of three integers between 0 and 255. Strings can be in the form
    # of a standard HTML color like #FF0000 for red, or it can be a
    # name corresponding to a constant in one of the modules given
    # in the :namespaces array.
    class HTMLColorParameter < ParameterType

      type_name :color, 'color'
      
      def string_to_type_internal(str)
        if str =~ /#([0-9a-fA-F]{6})/
          return $1.scan(/../).map {|i| i.to_i(16)}
        elsif @type.has_key?(:namespaces) 
          for mod in @type[:namespaces]
            if mod.const_defined?(str)
              return normalize(mod.const_get(str))
            end
          end
        else
          raise IncorrectInput, "#{str} is not a valid color"
        end
      end

      def type_to_string_internal(color)
        return "##{color.map {|i| sprintf("%02x",i)}.join}"
      end

      private
      # Returns a normalized color, that is, if everything is below or equal
      # to 1.0, multiply by 255 -- and always convert to int.
      def normalize(ar)
        if ar.max <= 1.0
          ret = ar.map {|c| c * 255.0}
        else
          ret = ar.dup
        end
        ret.map! {|i| i.to_i}
        return ret
      end
    end

    # A set of margins - that is, four floats representing left top
    # right bottom coordinates.
    class MarginsParameter < ParameterType

      type_name :margins, 'margins'
    end

  end
end
