# factory.rb : few classes to handle factories
# Copyright (C) 2006 Vincent Fourmond
 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

# The MetaBuilder module contains the classes necessary to create classes
# with a simple meta-information system that lets you save/export the
# state of an object (under certain assumptions), query parameters in
# different ways (command-line, Qt widgets, and others if necessary).
module MetaBuilder

  # The Factories module contains several classes useful for dealing
  # with Description factories:
  #
  # * UniqueFactory: only one instance of each class described, and
  # the user can choose which one is the current.
  module Factories

    # A factory for which there is only one instance of each
    # class described, and only one selected instance at any moment.
    # User can switch freely from one instance to another.
    class UniqueFactory

      attr_reader :current, :current_name

      # Creates a UniqueFactory using the factory in _base_class_.
      # The selected description is named by _default_, or is
      # the first one in the list in case _default_ is missing.
      def initialize(base_class, default = nil)
        @base_class = base_class
        @instances = {}
        for name, desc in @base_class.factory_description_hash
          @instances[name] = desc.instantiate
        end
        if default.nil?
          default = @base_class.factory_description_list.first.name
        end
        self.current=default
      end

      # Sets the current class to the named class.
      def current=(str)
        if @instances.has_key?(str)
          @current_name = str
          @current = @instances[str]
        else
          raise "Incorrect name given: #{str}"
        end
      end

      def instance_list
        return @instances.values
      end

      def instances
        return @instances
      end

      # Fills the Optionparser _parser_ with all the options necessary to
      # select a class among others and set options to it. Note that
      # options don't need to apply to the current class;  they are set
      # but the factory doesn't change the current class in this case.
      # 
      # The options will be preceded by _banner_. If _uniquify_ is true, the
      # names of the options will be made as unique as possible.
      def option_parser_factory(parser, banner = "", uniquify = true)
        parser.separator banner
        # A for loop won't work here as name would be overwritten
        # for every iteration.
        @instances.each do |name, instance|
          name
          parser.on("--#{name}", instance.description.banner(instance)) do 
            self.current = name
          end
          instance.option_parser_options(parser, uniquify)
          parser.separator ""
        end
      end
    end
  end

end
