# debug.rb, copyright (c) 2006 by Vincent Fourmond: 
# The debugging functions for PlotMaker
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/log'


module CTioga

  # This module contains code to be included by PlotMaker for debugging.
  # Even if it doesn't deserve a real module of it's own, it makes sense
  # to have it separated anyway.
  module Debug

    include Tioga::FigureConstants
    include Log

    # These variables are scoped at the module level so
    # that they are shared among all bits of code - as they can only be
    # set in one place.
    @@debug = false
    @@debug_indent = 0
    @@debug_indent_string = "  "

    # The function to use to display debugging information.
    # Checks if we are in debug mode.
    def debug_puts(str)
      if @@debug
        debug @@debug_indent_string * @@debug_indent +
          str
      end
    end

    # A function using the hooks defined by Bill to display what is
    # happenning inside Tioga
    def debug_figmaker(t)
      t.def_enter_context_function { 
        debug_puts "Tioga: enter new context\t" + 
        frames_str(t)
        @@debug_indent += 1
      }
      t.def_exit_context_function { 
        @@debug_indent -=1
        debug_puts "Tioga: exit context\t" + frames_str(t)
      }
        
      t.def_enter_show_plot_function { |bounds| 
        debug_puts "Tioga: enter show_plot #{bounds[0]} #{bounds[1]} " +
        "#{bounds[2]} #{bounds[3]}" 
        @@debug_indent += 1
      }
        
      t.def_exit_show_plot_function { |bounds| 
        @@debug_indent -=1
        debug_puts "Tioga: exit show_plot #{bounds[0]} #{bounds[1]} " +
        "#{bounds[2]} #{bounds[3]}" 
      }
        
      t.def_enter_subfigure_function { |margins| 
        debug_puts "Tioga: enter subfigure #{margins[0]} #{margins[1]} " +
        "#{margins[2]} #{margins[3]}" 
        debug_puts "Tioga: frames " + frames_str(t)
      }
        
      t.def_enter_subplot_function { |margins| 
        debug_puts "Tioga: enter subplot #{margins[0]} #{margins[1]} " +
        "#{margins[2]} #{margins[3]}" 
        debug_puts "Tioga: frames " + frames_str(t)
      }
    end

    def debug_patterns(t)
      t.def_figure('Test_patterns') { 
        setup_real_size_bop(t)
        test_pattern(t) 
      }
      t.make_preview_pdf(t.figure_index('Test_patterns'))

      t.def_figure('Test_patterns_right') { 
        setup_real_size_bop(t)
        test_pattern_right(t) 
      }
      t.make_preview_pdf(t.figure_index('Test_patterns_right'))
    end

    # This function is taken from Tioga; it is used to check the alignements
    # of the TeX text and PDF image.
    def test_pattern(t)
      t.line_width = 1
      t.stroke_frame
      # we make it asymmetric to catch other kinds of problems
      t.set_subframe('left' => 0.1 , 'right' => 0.05,
                     'top' => 0.05, 'bottom' => 0.1)
      centerx = t.bounds_xmin + 0.5 * t.bounds_width
      chr_ht = t.default_text_height_dy
      t.stroke_color = Red
      t.line_width = 0.2
      t.stroke_line(t.bounds_xmax, t.bounds_ymin, t.bounds_xmax, t.bounds_ymax)
      t.stroke_line(centerx, t.bounds_ymin, centerx, t.bounds_ymax)
      t.stroke_line(t.bounds_xmin, t.bounds_ymin, t.bounds_xmin, t.bounds_ymax)
      y = t.bounds_ymin; dy = t.bounds_height / 10
      scale = 0.85;
      10.times do
        t.stroke_line(t.bounds_xmin, y, t.bounds_xmax, y)
        t.show_text('text'=>"Left", 'x'=> t.bounds_xmin, 'y' => y,
                    'scale'=>0.5+scale, 'justification' => LEFT_JUSTIFIED)
        t.show_text('text'=>"Center", 'x'=> centerx, 'y' => y,
                    'scale'=>1.8-scale, 'justification' => CENTERED)
        t.show_text('text'=>"Right", 'x'=> t.bounds_xmax, 'y' => y,
                    'scale'=>0.5+scale, 'justification' => RIGHT_JUSTIFIED)
        scale = scale - 0.1
        y = y + dy
      end 
      t.stroke_line(t.bounds_xmin, t.bounds_ymax, t.bounds_xmax, t.bounds_ymax)
      t.justification = CENTERED
      t.show_text('text'=>"TeX and PDF Alignment Test Pattern",
                  'side'=> TOP, 'pos' => 0.5, 'shift'=>0.3)
      t.show_text('text'=>"Check for Proper Registration of Text and Graphics",
                  'side'=>BOTTOM, 'pos' => 0.5, 'shift'=> 1.6, 'scale'=>0.4)
      t.show_text(
                  'text'=>'If needed, adjust by changing ' + 
                  '{\sffamily FigureMaker.tex\_xoffset} ' +
                  'and {\sffamily FigureMaker.tex\_yoffset}.',
                  'side'=>BOTTOM, 'pos' => 0.5, 'shift'=> 3.9, 'scale'=>0.3)
    end

    # This function is taken from Tioga; it is used to check the alignements
    # of the TeX text and PDF image.
    def test_pattern_right(t)
      t.line_width = 1
#       t.stroke_frame
      margin = 0.1
      # we make it asymmetric to catch other kinds of problems
      t.set_subframe('left' => 0.1 , 'right' => 0.05,
                     'top' => 0.05, 'bottom' => 0.1)
      centerx = t.bounds_xmin + 0.5 * t.bounds_width
      chr_ht = t.default_text_height_dy
      t.stroke_color = Red
      t.line_width = 1
      t.stroke_line(t.bounds_xmax, t.bounds_ymin, t.bounds_xmax, t.bounds_ymax)
      t.stroke_line(centerx, t.bounds_ymin, centerx, t.bounds_ymax)

      y = t.bounds_ymin; dy = t.bounds_height / 10
      scale = 0.85;
      10.times do
        t.stroke_line(centerx, y, t.bounds_xmax, y)
        t.show_text('text'=>"Left", 'x'=> t.bounds_xmin, 'y' => y,
                    'scale'=>0.5+scale, 'justification' => LEFT_JUSTIFIED)
        t.show_text('text'=>"Center", 'x'=> centerx, 'y' => y,
                    'scale'=>1.8-scale, 'justification' => CENTERED)
        t.show_text('text'=>"Right", 'x'=> t.bounds_xmax, 'y' => y,
                    'scale'=>0.5+scale, 'justification' => RIGHT_JUSTIFIED)
        scale = scale - 0.1
        y = y + dy
      end 
      t.stroke_line(centerx, t.bounds_ymax, t.bounds_xmax, t.bounds_ymax)
      t.justification = CENTERED
      t.show_text('text'=>"TeX and PDF Alignment Test Pattern",
                  'side'=> TOP, 'pos' => 0.5, 'shift'=>0.3)
      t.show_text('text'=>"Check for Proper Registration of Text and Graphics",
                  'side'=>BOTTOM, 'pos' => 0.5, 'shift'=> 1.6, 'scale'=>0.4)
      t.show_text(
                  'text'=>'If needed, adjust by changing ' + 
                  '{\sffamily FigureMaker.tex\_xoffset} ' +
                  'and {\sffamily FigureMaker.tex\_yoffset}.',
                  'side'=>BOTTOM, 'pos' => 0.5, 'shift'=> 3.9, 'scale'=>0.3)
    end

    # Returns a string representing the 'options' of the FigureMaker object
    # given.
    def figmaker_options(t)
      return '' unless @@debug
      s = ""
      for meth in t.methods.sort
        if t.respond_to?((meth + '=').to_sym)
          begin
            s += "\n#{meth}: " + t.send(meth).to_s
          rescue
            # In case the method did need some arguments...
          end
        end
      end
      return s
    end

  end

end
