/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: medussa.c,v 1.38 2003/02/05 04:40:14 kos Exp $
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <pthread.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "common.h"
#include "array.h"
#include "configfile.h"
#include "llog.h"
#include "hashpool.h"
#include "net.h"
#include "cli.h"
#include "xmalloc.h"
#include "keyspace.h"
#include "medussa.h"
#include "console.h"
#include "glob.h"

/*
 * name: ascii string
 * function: implementation
 * minargs
 * maxargs
 * state: any=at any time, otherwise required state
 */

static directive_t directive[] = {
  { "help",      do_help, 0, 0, any, "Recursion" },
  { "quit",      do_quit, 0, 0, any, "Ends the connection" },
  { "fetch",     do_fetch, 0, 0, any, "Requests a file" },
  { "motd",      do_motd, 0, 0, any, "Requests the message of the day" },
  { "ident",     do_ident, 0, 0, any, "Requests the server to identify itself" },
  { "version",   do_version, 0, 0, any, "Produces version information" },
  { "client",    do_client, 1, 1, any, "Switches to client mode" },
  { "admin",     do_admin, 1, 1, any, "Switches to administration mode" },

  { "gimme",     do_gimme, 1, 1, client, "Requests a new keyspace fraction" },
  { "zilch",     do_zilch, 3, 3, client, "Unsuccessful update on keyspace fraction" },
  { "gotit",     do_gotit, 5, 5, client, "Successful update on keyspace fraction" },
  { "cancel",    do_cancel, 3, 3, client, "Node originated cancellation request" },

  { "schedule",  do_schedule, 1, 0, admin, "Manages the scheduler" },
  { "hash",      do_hash, 1, 0, admin, "Manages the hashes" },
  { "pool",      do_pool, 1, 0, admin, "Manages the hash pool" },
  { "stats",     do_stats, 1, 0, admin, "Statistics calculator" },
  { "node",      do_node, 1, 0, admin, "Manages the nodes" },
  { "parameter", do_parameter, 1, 0, admin, "Manages the scheduler parameters" },
  { "", (dfunction_t)NULL, 0, 0},
};

static int
filename_notok(medussa_t *m, char *fname) {
  
  if(config_class_char_get(m->conf, "downloadable", fname))
    return 0;
  return 1;
}

static void
tokenize(char *usrc, char **tokens, int *ntokens) {
  int i;
  static char src[MED_LINELEN];
  
  strncpy(src, usrc, MED_LINELEN);
  for(i=1,tokens[0] = strtok(src, " "); (tokens[i]=strtok(NULL, " ")); i++)
    ;
  *ntokens = i;
}

static int
harvest_params(msg *params, msg *p, int sindex) {
  int i;
  char *name;
  char *value;

  for(i=sindex; i<msg_nelems(p); i+=2) {
    name = msg_get(p, i);
    value = msg_get(p, i+1);
    msg_add_str(params, name, value, msg_getlen(p, i+1));
  }
  return 0;
}

static int
packet_send(net_conn_t *c, int code, int count, ...) {
  char type[MED_LINELEN];
  char *pp;
  va_list vl;
  msg *packet;
  int i;

  va_start(vl, count);
  packet = msg_new();
  snprintf(type, MED_LINELEN, "%03d", code);
  msg_add(packet, type, 0);
  if(count>0) {
    for(i=0; i<count; i++) {
      pp = va_arg(vl, char *);
      msg_add(packet, pp, 0);
    }
  } else {
    while((pp = va_arg(vl, char *)))
      msg_add(packet, pp, 0);
  }
  va_end(vl);

  net_send(c, packet);
  msg_destroy(packet);
  return 0;
}

int
do_motd(medussa_t *m, net_conn_t *c, msg *p) {
  char *motd;
  
  if(!(motd = config_char_get(m->conf, "motd"))) 
    packet_send(c, P_OK, 1, "None");
  else
    packet_send(c, P_OK, 1, motd);
  return 0;
}

int
do_ident(medussa_t *m, net_conn_t *c, msg *p) {
  char *ident;
  
  if(!(ident = config_char_get(m->conf, "ident")))
    packet_send(c, P_OK, 1, "None");
  else
    packet_send(c, P_OK, 1, ident);
  return 0;
}

int
do_help(medussa_t *m, net_conn_t *c, msg *p) {
  int i;

  for(i=0; directive[i].func; i++)
    packet_send(c, P_MORE, 0, directive[i].name, directive[i].help, NULL);
  packet_send(c, P_OK, 0, "Satisfied, Sherlock?", NULL);
  return 0;
}

int
do_client(medussa_t *m, net_conn_t *c, msg *p) {
  m->state = client;
  packet_send(c, P_OK, 2, msg_get(p, 1), " is a cool name.");
  return 0;
}

int
do_gimme(medussa_t *m, net_conn_t *c, msg *p) {
  workload_t *w;
  msg *packet;
  char type[MED_LINELEN];

  w = workload_init();

  if(hashpool_node_register(m->h, msg_get(p, 1), w)) {
    workload_destroy(w);
    llog(2, "hashpool_register(%s): %s\n", msg_get(p, 1), hashpool_geterr(m->h));
    packet_send(c, P_ERR, 1, hashpool_geterr(m->h));
    return 0;
  }

  packet = msg_new();
  snprintf(type, MED_LINELEN, "%03d", P_OK);
  msg_add(packet, type, 0);
  msg_add(packet, w->generator, 0);
  msg_add(packet, w->generator_params, 0);
  msg_add(packet, w->hash.p, w->hash.l);
  msg_add(packet, w->method, 0);
  msg_add(packet, w->start, 0);
  msg_add(packet, w->finish, 0);
  net_send(c, packet);
  msg_destroy(packet);
  workload_destroy(w);
  return 0;
}

int
do_zilch(medussa_t *m, net_conn_t *c, msg *p) {

  if(hashpool_node_update(m->h, msg_get(p, 1), msg_get(p, 2), msg_get(p, 3), NULL, 0, NULL)) {
    llog(2, "hashpool_update(%s): %s\n", msg_get(p, 1), hashpool_geterr(m->h));
    packet_send(c, P_ERR, 1, hashpool_geterr(m->h));
    return 0;
  }
  packet_send(c, P_OK, 1, "Thanks for trying");
  return 0;
}

int
do_gotit(medussa_t *m, net_conn_t *c, msg *p) {

  if(hashpool_node_update(m->h, msg_get(p, 1), msg_get(p, 2), msg_get(p, 3), msg_get(p, 4), msg_getlen(p, 4), msg_get(p, 5))) {
    llog(2, "hashpool_update(%s): %s\n", msg_get(p, 1), hashpool_geterr(m->h));
    packet_send(c, P_ERR, 2, "hashpool_update:", hashpool_geterr(m->h));
    return 0;
  }
  packet_send(c, P_OK, 1, "You rule");
  return 0;
}

int
do_cancel(medussa_t *m, net_conn_t *c, msg *p) {
  msg *params;

  params = msg_new();
  msg_add_str(params, "name", msg_get(p, 1), msg_getlen(p, 1));
  if(hashpool_manage(m->h, "node", "delete", params)) {
    llog(2, "hashpool_manage(%s): %s\n", msg_get(p, 1), hashpool_geterr(m->h));
    packet_send(c, P_ERR, 1, hashpool_geterr(m->h));
    msg_destroy(params);
    return 0;
  }
  packet_send(c, P_OK, 1, "Retracted.");
  msg_destroy(params);
  return 0;
}

int
do_version(medussa_t *m, net_conn_t *c, msg *p) {

  packet_send(c, P_OK, 1, PROTO_VERSION);
  return 0;
}

int
do_quit(medussa_t *m, net_conn_t *c, msg *p) {
  m->state = initial;
  packet_send(c, P_OK, 0, "Good hunting", NULL);
  return 1;
}

/* XXX: broken, use net_wfp */
int
do_fetch(medussa_t *m, net_conn_t *c, msg *p) {
  char buf[MED_LINELEN];
  FILE *fp;
  int len;
  msg *sp;
  struct stat st;  
  
  if(filename_notok(m, msg_get(p, 1))) {
    packet_send(c, P_ERR, 1, "This ain\'t ftp, homeboy.");
    return 1;
  }

  if(!(fp = fopen(msg_get(p, 1), "r"))) {
    packet_send(c, P_ERR, 2, "fopen(r):", strerror(errno));
    return 0;
  }

  fstat(fileno(fp), &st);
  snprintf(buf, MED_LINELEN, "%lu", st.st_size);
  packet_send(c, P_MORE, 2, buf, "octets coming your way");

  /*net_conn_setflags(c, NET_F_BINARY);*/
  while((len = fread(buf, 1, MED_LINELEN, fp)) > 0) {
    sp = msg_new();
    msg_add(sp, buf, len);
    net_send(c, sp);
    msg_destroy(sp);
  }
  fclose(fp);
  /*net_conn_setflags(c, NET_F_DEFAULT);*/
  packet_send(c, P_OK, 1, "OK.");

  return 0;
}

int
do_admin(medussa_t *m, net_conn_t *c, msg *p) {
  bstring response;
  char *adminpassword;
  
  if(!(adminpassword = config_char_get(m->conf, "adminpassword"))) {
    packet_send(c, P_ERR, 0, "Remote administration is disabled.", NULL);
    return 0;
  }

  chresp_init(adminpassword, &m->nonce, &m->adminhash);
  response.l = MIN(BSTRING_MAXLEN, msg_getlen(p, 1));
  memcpy(response.p, msg_get(p, 1), response.l);

  if(chresp_authenticate(&m->adminhash, &response)) {
    m->state = admin;
    packet_send(c, P_OK, 0, "I\'m bracing myself", NULL);
    llog(2, "Successful admin login from %s\n", c->ident);
  } else {
    packet_send(c, P_ERR, 0, "Bzzzt.", NULL);
    llog(2, "Failed admin login from %s\n", c->ident);
    return 1;
  }
  return 0;
}

int
do_schedule(medussa_t *m, net_conn_t *c, msg *p) {
  int retv=0;
  char retstr[1024];
  char name[MED_LINELEN];
  char value[MED_LINELEN];
  int valuel;
  int i;  
  msg *params;

  params = msg_new();
  if(harvest_params(params, p, 2)) {
    packet_send(c, P_ERR, 0, "harvest_params: malformed argument", NULL);
    msg_destroy(params);
    return 0;
  }

  if((retv = hashpool_manage(m->h, "schedule", msg_get(p, 1), params))) {
    snprintf(retstr, 1024, "hashpool_manage(schedule, %s) = %d", msg_get(p, 1), retv);
    packet_send(c, P_ERR, 0, retstr, hashpool_geterr(m->h), NULL);
    msg_destroy(params);
    return 0;
  }

  for(i=0; i<msg_nelems(params); i++) {
    if(msg_get_index(params, i, name, MED_LINELEN, value, MED_LINELEN, &valuel))
      continue;
    packet_send(c, P_MORE, 0, name, value, NULL);
  }

  packet_send(c, P_OK, 0, "OK.", NULL);
  msg_destroy(params);
  return 0;
}

int
do_hash(medussa_t *m, net_conn_t *c, msg *p) {
  int retv;
  char retstr[1024];
  char name[MED_LINELEN];
  char value[MED_LINELEN];
  int valuel;
  int i;
  msg *params;

  params = msg_new();
  if(harvest_params(params, p, 2)) {
    packet_send(c, P_ERR, 0, "harvest_params: malformed argument", NULL);
    msg_destroy(params);
    return 0;
  }

  if((retv = hashpool_manage(m->h, "hash", msg_get(p, 1), params))) {
    snprintf(retstr, 1024, "hashpool_manage(hash, %s) = %d", msg_get(p, 1), retv);
    packet_send(c, P_ERR, 0, retstr, hashpool_geterr(m->h), NULL);
    msg_destroy(params);
    return 0;
  }

  for(i=0; i<msg_nelems(params); i++) {
    if(msg_get_index(params, i, name, MED_LINELEN, value, MED_LINELEN, &valuel))
      continue;
    packet_send(c, P_MORE, 0, name, value, NULL);
  }

  packet_send(c, P_OK, 0, "OK.", NULL);
  msg_destroy(params);
  return 0;
}

int
do_pool(medussa_t *m, net_conn_t *c, msg *p) {
  int retv;
  char retstr[1024];
  char name[MED_LINELEN];
  char value[MED_LINELEN];
  int valuel;
  int i;
  msg *params;

  params = msg_new();
  if(harvest_params(params, p, 2)) {
    packet_send(c, P_ERR, 0, "harvest_params: malformed argument", NULL);
    msg_destroy(params);
    return 0;
  }

  if((retv = hashpool_manage(m->h, "pool", msg_get(p, 1), params))) {
    snprintf(retstr, 1024, "hashpool_manage(pool, %s) = %d", msg_get(p, 1), retv);
    packet_send(c, P_ERR, 0, retstr, hashpool_geterr(m->h), NULL);
    msg_destroy(params);
    return 0;
  }

  for(i=0; i<msg_nelems(params); i++) {
    if(msg_get_index(params, i, name, MED_LINELEN, value, MED_LINELEN, &valuel))
      continue;
    packet_send(c, P_MORE, 0, name, value, NULL);
  }
  
  packet_send(c, P_OK, 0, "OK.", NULL);
  msg_destroy(params);
  return 0;
}

int
do_stats(medussa_t *m, net_conn_t *c, msg *p) {
  char retstr[1024];
  int retv;
  char name[MED_LINELEN];
  char value[MED_LINELEN];
  int valuel;
  int i;
  msg *params;

  params = msg_new();
  if(harvest_params(params, p, 2)) {
    packet_send(c, P_ERR, 0, "harvest_params: malformed argument", NULL);
    msg_destroy(params);
    return 0;
  }

  if((retv = hashpool_manage(m->h, "stats", msg_get(p, 1), params))) {
    snprintf(retstr, 1024, "hashpool_manage(pool, %s) = %d", msg_get(p, 1), retv);
    packet_send(c, P_ERR, 0, retstr, hashpool_geterr(m->h), NULL);
    msg_destroy(params);
    return 0;
  }

  for(i=0; i<msg_nelems(params); i++) {
    if(msg_get_index(params, i, name, MED_LINELEN, value, MED_LINELEN, &valuel))
      continue;
    packet_send(c, P_MORE, 0, name, value, NULL);
  }
  
  msg_destroy(params);
  packet_send(c, P_OK, 0, "OK.", NULL);
  return 0;
}

int
do_node(medussa_t *m, net_conn_t *c, msg *p) {
  int retv;
  char retstr[1024];
  char name[MED_LINELEN];
  char value[MED_LINELEN];
  int valuel;
  int i;
  msg *params;

  params = msg_new();
  if(harvest_params(params, p, 2)) {
    packet_send(c, P_ERR, 0, "harvest_params: malformed argument", NULL);
    return 0;
  }

  if((retv = hashpool_manage(m->h, "node", msg_get(p, 1), params))) {
    snprintf(retstr, 1024, "hashpool_manage(node, %s) = %d", msg_get(p, 1), retv);
    packet_send(c, P_ERR, 0, retstr, hashpool_geterr(m->h), NULL);
    msg_destroy(params);
    return 0;
  }

  for(i=0; i<msg_nelems(params); i++) {
    if(msg_get_index(params, i, name, MED_LINELEN, value, MED_LINELEN, &valuel))
      continue;
    packet_send(c, P_MORE, 0, name, value, NULL);
  }
  
  packet_send(c, P_OK, 0, "OK.", NULL);
  msg_destroy(params);
  return 0;
}

int
local_param_list(medussa_t *m, msg *p) {
  int i;
  config_elem_t *el;

  for(i=0; i<config_nelems(m->conf); i++) {
    el = config_getbynum(m->conf, i);
    msg_add_strn(p, el->lhs, el->lhs);
  }
  return 0;
}

int
local_param_get(medussa_t *m, char *which, msg *p) {
  char *res;

  if(!which)
    return 1;
  if(!(res = config_char_get(m->conf, which)))
    return 2; 
  msg_zero(p);
  msg_add_strn(p, which, res);
  return 0;
}

int
local_param_set(medussa_t *m, char *lhs, char *rhs, msg *p) {
  msg_zero(p);
  config_set(m->conf, lhs, rhs);
  return 0;
}

/* XXX: so this is hackish. fix it */
int
do_parameter(medussa_t *m, net_conn_t *c, msg *p) {
  int retv;
  char retstr[1024];
  char name[MED_LINELEN];
  char value[MED_LINELEN];
  int valuel;
  int i;
  msg *params;
  char *which;
  char *towhat;

  params = msg_new();

  if(harvest_params(params, p, 2)) {
    packet_send(c, P_ERR, 0, "harvest_params: malformed argument", NULL);
    msg_destroy(params);
    return 0;
  }

  if(!strcmp(msg_get(p, 1), "list")) {
    if((retv = hashpool_manage(m->h, "parameter", msg_get(p, 1), params))) {
      snprintf(retstr, 1024, "hashpool_manage(parameter, %s) = %d", msg_get(p, 1), retv);
      packet_send(c, P_ERR, 0, retstr, hashpool_geterr(m->h), NULL);
      msg_destroy(params);
      return 0;
    }
    local_param_list(m, params);

  } else if(!strcmp(msg_get(p, 1), "get")) {
    if(!(which = msg_get_strp(params, "name"))) {
      packet_send(c, P_ERR, 0, "no parameter specified", NULL);
      msg_destroy(params);
      return 0;
    }
    if((retv = hashpool_manage(m->h, "parameter", msg_get(p, 1), params))) {
      snprintf(retstr, 1024, "hashpool_manage(parameter, %s) = %d", msg_get(p, 1), retv);
      if((retv = local_param_get(m, which, params))) {
	packet_send(c, P_ERR, 0, retstr, hashpool_geterr(m->h), NULL);
	msg_destroy(params);
	return 0;
      }
    }
  } else if(!strcmp(msg_get(p, 1), "set")) {
    if(!(which = msg_get_strp(params, "name"))) {
      packet_send(c, P_ERR, 0, "no parameter specified", NULL);
      msg_destroy(params);
      return 0;
    }
    if(!(towhat = msg_get_strp(params, "value"))) {
      packet_send(c, P_ERR, 0, "no value specified", NULL);
      msg_destroy(params);
      return 0;
    }

    if(!msg_get_strp(m->h->param, which))
      retv = local_param_set(m, which, towhat, params);
    else
      retv = hashpool_manage(m->h, "parameter", "set", params);

    if(retv) {
      snprintf(retstr, 1024, "hashpool_manage(parameter, %s) = %d", msg_get(p, 1), retv);
      packet_send(c, P_ERR, 0, retstr, hashpool_geterr(m->h), NULL);
      msg_destroy(params);
      return 0;
    }
  } else {
    if((retv = hashpool_manage(m->h, "parameter", msg_get(p, 1), params))) {
      snprintf(retstr, 1024, "hashpool_manage(parameter, %s) = %d", msg_get(p, 1), retv);
      packet_send(c, P_ERR, 0, retstr, hashpool_geterr(m->h), NULL);
      msg_destroy(params);
      return 0;
    }
  }

  for(i=0; i<msg_nelems(params); i++) {
    if(msg_get_index(params, i, name, MED_LINELEN, value, MED_LINELEN, &valuel))
      continue;
    packet_send(c, P_MORE, 0, name, value, NULL);
  }
  
  msg_destroy(params);
  packet_send(c, P_OK, 0, "OK.", NULL);
  return 0;
}

void *
serve_single(void *p) {
  int i;
  msg *packet;
  char *comm;
  medussa_t *m;
  char type[MED_LINELEN];
  
  m = (medussa_t *)p;
  llog(5, "Servicing client \"%s\"\n", net_conn_id(m->conn));

  /* fill up the rest of the thread information */
  m->state = initial;
  chresp_generate_nonce(&m->nonce, NONCE_LEN);
  
  snprintf(type, MED_LINELEN, "%03d", P_OK);
  packet = msg_new();
  msg_add(packet, type, 0);
  msg_add(packet, "medussa", 0);
  msg_add(packet, ""PROTO_VERSION"", 0);
  msg_add(packet, "Implementation "MEDUSSA_VERSION" ["HOSTTYPE"], ready and willing", 0);
  msg_add(packet, m->nonce.p, m->nonce.l);
  net_send(m->conn, packet);
  msg_destroy(packet);
  
  while(1) {
    if(!(packet = net_recv(m->conn)))
      break;
    if(!msg_nelems(packet)) {
      packet_send(m->conn, P_ERR, 1, "No command specified; Wake up, grandad.");
      goto again;
    }

    comm = msg_get(packet, 0);
    for(i=0; directive[i].func; i++) {

      if(!strncmp(directive[i].name, comm, MED_LINELEN)) {
	if(directive[i].minargs && directive[i].minargs > msg_nelems(packet)-1) {
	  packet_send(m->conn, P_ERR, 2, "Not enough arguments for ", comm);
	  goto again;
	}
	if(directive[i].maxargs && directive[i].maxargs < msg_nelems(packet)-1) {
	  packet_send(m->conn, P_ERR, 2, "Too many arguments for ", comm);       
	  goto again;
	}

	if(directive[i].state != any && directive[i].state != m->state) {
	  packet_send(m->conn, P_ERR, 2, "Wrong state for ", comm);
	  goto again;
	}

	if(directive[i].func(m, m->conn, packet)) {
	  msg_destroy(packet);
	  goto out;
	}
	goto again;
      }
    }

    if(!directive[i].func) {
      packet_send(m->conn, P_ERR, 1, "Unknown directive");
      continue;
    }
    
  again:
    msg_destroy(packet);
  }

 out:
  net_conn_destroy(m->conn);
  xfree(m);
  return NULL;
}

void *
run_console(void *p) {
  medussa_t *m;
  console_t *c;

  m = (medussa_t *)p;
  if(!(c = console_new(m))) {
    llog(1, "console_new: failed\n");
    return NULL;
  }

  console_main(c);
  console_destroy(c);
  return NULL;
}

void *
run_scheduler(void *p) {
  medussa_t *m;
  int checkpoint;

  checkpoint = 0;
  m = (medussa_t *)p;

  while(1) {
    if(checkpoint == config_int_get(m->conf, "checkpointperiod")) {
      hashpool_checkpoint(m->h, config_char_get(m->conf, "checkpointfile"));
      checkpoint = 0;
    }
    hashpool_schedule(m->h);    
    sleep(1);
    checkpoint++;
  }
  return NULL;
}

void *
run_listener(void *p) {
  medussa_t *m;
  net_conn_t *c;
  pthread_attr_t attr;
  pthread_t thrid;
  medussa_t *lm;  
  
  m = (medussa_t *)p;

  while(1) {
    if(!(c = net_accept(m->n))) {
      llog(1, "accept: %s\n", strerror(errno));
      sleep(1);
      continue;
    }

    lm = xcalloc(1, sizeof(medussa_t));
    memcpy((void *)lm, (void *)m, sizeof(medussa_t));
    lm->conn = c;
    
    pthread_attr_init(&attr);
    pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    pthread_create(&thrid, &attr, serve_single, (void *)lm);
    pthread_attr_destroy(&attr);
  }
  
  return 0;
}

void
create_pidfile(char *file) {
  FILE *fp;

  if(!file || !(fp = fopen(file, "w")))
    return;

  fprintf(fp, "%u\n", getpid());
  fclose(fp); 
}

int
serve_multiple(medussa_t *m) {
  pthread_t console;
  pthread_t scheduler;
  pthread_t listener;
  pthread_attr_t attr;
  char *user;

  if((user = config_char_get(m->conf, "uid")))
    setreuid(atoi(user), atoi(user));
  if((user = config_char_get(m->conf, "gid")))
    setregid(atoi(user), atoi(user));

  /* Pretend we're an application */
  if(!config_int_get(m->conf, "interactive")) {
    if(fork())
      exit(0);
    else {
#ifdef SETPGRP_VOID
      setpgrp();
#else
      setpgrp(0, getpid());
#endif

      close(0);
      close(1);
      close(2);
    }
    create_pidfile(config_char_get(m->conf, "pidfile"));
  } else {    
    /* Spawn the console thread */
    pthread_attr_init(&attr);
    pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    pthread_create(&console, &attr, run_console, (void *)m);
    pthread_attr_destroy(&attr);    
  }
  
  /* Spawn the scheduler thread */
  pthread_attr_init(&attr);
  pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);
  pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
  pthread_create(&scheduler, &attr, run_scheduler, (void *)m);
  pthread_attr_destroy(&attr);

  /* Spawn the listener */
  pthread_attr_init(&attr);
  pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);
  pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
  pthread_create(&listener, &attr, run_listener, (void *)m);
  pthread_attr_destroy(&attr);

  if(config_int_get(m->conf, "interactive")) { 
    pthread_join(console, NULL);
    pthread_kill(scheduler, SIGTERM);
    pthread_join(scheduler, NULL);
    pthread_kill(listener, SIGTERM);      
  } else {
    pthread_join(scheduler, NULL);
  }

  return 0;
}

/* XXX: update this */
void
usage(char *argv0, int code) {
  fprintf(stderr, "\
Medussa version %s [%s] usage options\n\
\t-V Print version string\n\
\t-v verbose level (default: %s)\n\
\t-b bind address (default: %s)\n\
\t-p bind port (default: %s)\n\
\t-i server name (default: %s)\n\
\t-T configuration dump\n\
\t-d Background operation\n\
\t-s Filename to use when checkpointing (default: %s)\n\
\t-S Time period in seconds for checkpointing (default: %s)\n\
\t-r Restore from checkpointed configuration\n\
",
	  MEDUSSA_VERSION,
	  HOSTTYPE,
	  DEF_VERBOSE,
	  DEF_BINDADDR,
	  DEF_PORT,
	  "hostname",
	  DEF_CHECKPOINTFILE,
	  DEF_CHECKPOINTPERIOD
	  );
  exit(code);
}

void
logging_init(config_t *c) {

  if(!strcmp(config_char_get(c, "logmethod"), "file"))
    llog_init(LLOG_FILE, config_char_get(c, "logfile"), "w");
  else if(!strcmp(config_char_get(c, "logmethod"), "syslog"))
    llog_init(LLOG_SYSLOG, "medussa", LOG_PID, LOG_DAEMON, LOG_NOTICE);
  else
    llog_init(LLOG_STDERR);
  llog_level(config_int_get(c, "verbose"));
}

int
load_schedule(medussa_t *m) {
  char *tokens[16];
  int ntokens;
  msg *params;
  int i;
  char *name;
  char *schedule;
  int j;
  char line[MED_LINELEN];
  
  params = msg_new();
  
  for(i=0; i<config_nclasses(m->conf); i++) {
    name = config_class_name(m->conf, i);    
    if(!strcmp(name, DEFAULT_CLASS))
      continue;
    
    msg_zero(params);
    msg_add_strn(params, "name", name);
    
    if(!config_class_char_get(m->conf, name, "type") ||
       !config_class_char_get(m->conf, name, "hash"))
      continue;

    msg_add_strn(params, "method", config_class_char_get(m->conf, name, "type"));
    msg_add_strn(params, "hash", config_class_char_get(m->conf, name, "hash"));
    if(hashpool_manage(m->h, "hash", "add", params))
      llog(1, "hashpool_manage(hash): %s\n", hashpool_geterr(m->h));
    
    for(j=0; j<config_class_nelems(m->conf, name); j++) {
      if(glob((schedule = config_class_elem_name(m->conf, name, j)), "schedule*")) {
	msg_zero(params);
	msg_add_strn(params, "name", schedule);
	msg_add_strn(params, "hash", name);
	strncpy(line, config_class_char_get(m->conf, name, schedule), MED_LINELEN);
	tokenize(line, tokens, &ntokens);
	msg_add_strn(params, "generator", tokens[0]);
	msg_add_strn(params, "generator_params", tokens[1]);
	if(hashpool_manage(m->h, "schedule", "add", params))
	  llog(1, "hashpool_manage(schedule): %s\n", hashpool_geterr(m->h));
      }
    }
  }

  msg_destroy(params);
  return 0;
}

int
main(int argc, char **argv) {
  medussa_t m;
  char c;
  char configclass[MED_LINELEN];
  char configfile[MED_LINELEN];
  char ident[MED_LINELEN];
  
  /* basic defaults */
  strncpy(configclass, DEF_CONFIGCLASS, MED_LINELEN);
  strncpy(configfile, DEF_CONFIGFILE, MED_LINELEN);

  /* defaults */
  if(!(m.conf = config_init(configclass))) {
    fprintf(stderr, "config_init(%s): Failed\n", configclass);
    exit(12);
  }

  gethostname(ident, MED_LINELEN);
  config_set(m.conf, "ident", ident);
  config_set(m.conf, "bindaddress", "0.0.0.0");
  config_set(m.conf, "bindport", DEF_PORT);
  config_set(m.conf, "configclass", configclass);
  config_set(m.conf, "verbose", DEF_VERBOSE);
  config_set(m.conf, "uid", DEF_UID);
  config_set(m.conf, "gid", DEF_GID);
  config_set(m.conf, "interactive", DEF_INTERACTIVE);
  config_set(m.conf, "pidfile", DEF_PIDFILE);
  config_set(m.conf, "checkpointfile", DEF_CHECKPOINTFILE);
  config_set(m.conf, "checkpointperiod", DEF_CHECKPOINTPERIOD);
  config_set(m.conf, "logmethod", DEF_LOGMETHOD);

  /* config file options */
  if(config_load(m.conf, configfile))
    fprintf(stderr, "config_load(%s): %s\n", configfile, config_perror(m.conf));

  /* command line options */
  while((c = getopt(argc, argv, "df:p:b:i:v:s:S:r:TV")) != EOF) {    
    switch(c) {
    case 'd':
      config_set(m.conf, "interactive", "0");
      config_set(m.conf, "logmethod", "syslog");
      break;
    case 'f':
      config_set(m.conf, "configfile", optarg);
      break;
    case 'v':
      config_set(m.conf, "verbose", optarg);
      break;
    case 'b':
      config_set(m.conf, "bindaddress", optarg);
      break;
    case 'p':
      config_set(m.conf, "bindport", optarg);
      break;
    case 'i':
      config_set(m.conf, "ident", optarg);
      break;
    case 's':
      config_set(m.conf, "checkpointfile", optarg);
      break;
    case 'S':
      config_set(m.conf, "checkpointperiod", optarg);
      break;
    case 'r':
      config_set(m.conf, "restorefile", optarg);
      break;
    case 'T':
      config_dump(m.conf);
      exit(0);
    case 'V':
      printf("medussa version %s\n", VERSION);
      exit(0);
    case '?':
      usage(argv[0], 1);
      break;
    }
  }

  m.pid = getpid();
  logging_init(m.conf);
  
  if(!(m.h = hashpool_init())) {
    llog(1, "hashpool_init failed. What-what-what?\n");
    exit(1);
  }

  if(config_char_get(m.conf, "restorefile")) {
    if(hashpool_restore(m.h, config_char_get(m.conf, "restorefile"))) {
      llog(1, "Restore from %s failed: %s\n", config_char_get(m.conf, "restorefile"), hashpool_geterr(m.h));
      llog(1, "Continuing ...\n");
    }
  } else {
    load_schedule(&m);
  }

  if(!(m.n = net_init(NET_SERVER, 
		      config_char_get(m.conf, "bindaddress"), 
		      atoi(config_char_get(m.conf, "bindport"))))) {
    llog(1, "net_init(NET_SERVER, %s, %s) failed\n",
	 config_char_get(m.conf, "bindaddress"),
	 config_char_get(m.conf, "bindport"));
    hashpool_destroy(m.h);
    exit(2);
  }
  
  llog(1, "Medussa %s starting up, loglevel %d, pid %d\n", 
       MEDUSSA_VERSION, 
       config_int_get(m.conf, "verbose"), 
       m.pid);
  serve_multiple(&m);

  hashpool_checkpoint(m.h, config_char_get(m.conf, "checkpointfile"));
  hashpool_destroy(m.h);
  net_destroy(m.n);
  config_destroy(m.conf);
  llog_close();
  exit(0);
}
