/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _WF_POPUP_H_
#define _WF_POPUP_H_

#include "mforms/popup.h"
#include "wf_base.h"
#include "wf_mforms.h"

using namespace System;
using namespace Windows::Forms;
using namespace System::Drawing::Drawing2D;

namespace MySQL {
  namespace Forms {

    // The actual popup window
    private ref class PopupControl : Windows::Forms::Form
    {
    protected:
      int modalResult;

      virtual void OnMouseDown(MouseEventArgs ^e) override;
      virtual void OnMouseUp(MouseEventArgs ^e) override;
      virtual void OnMouseClick(MouseEventArgs ^e) override;
      virtual void OnMouseDoubleClick(MouseEventArgs ^e) override;
      virtual void OnMouseMove(MouseEventArgs ^e) override;
      virtual void OnMouseEnter(EventArgs ^e) override;
      virtual void OnMouseLeave(EventArgs ^e) override;

      virtual void OnKeyPress(KeyPressEventArgs^ e) override;
    public:
      PopupControl();

      virtual void DoRepaint();
      virtual int Show(int x, int y);

      virtual property MySQL::Geometry::Rect DisplayRect
      {
        MySQL::Geometry::Rect get()
        {
          System::Drawing::Rectangle content_area= ClientRectangle;
          content_area.X += Padding.Left;
          content_area.Y += Padding.Top;
          content_area.Width -= Padding.Horizontal;
          content_area.Height -= Padding.Vertical;

          return MySQL::Geometry::Rect(content_area.Left, content_area.Top, content_area.Width, content_area.Height);
        }
      }

      property int ModalResult
      {
        int get() { return modalResult; }
        void set(int value) { modalResult = value; }
      }
    };

    private ref class StandardPopupControl: PopupControl
    {
    protected:
      virtual void OnPaint(PaintEventArgs ^e) override;
    public:
      StandardPopupControl();
    };

    private ref class TransparentPopupControl: PopupControl
    {
    private:
      System::Drawing::Bitmap^ contentBitmap;
      System::Drawing::Size baseSize;
      float borderSize;
      int shadowSize;
      System::Drawing::Color shadowColor;
      int shadowOffset;
      int cornerSize;
      int animationSteps;
      bool animated;
      System::Drawing::Point hotSpot;
    protected:
      void RunLoop();

      void UpdateAndShowPopup(bool doAnimated);
      void HidePopup();
      GraphicsPath^ GetPath();
      void PrepareBitmap();

      virtual void OnKeyPress(KeyPressEventArgs^ e) override;
      virtual void OnLostFocus(EventArgs^ e) override;
      virtual void OnMouseDown(MouseEventArgs^ e) override;
    public:
      TransparentPopupControl();

      virtual void DoRepaint() override;
      virtual int Show(int x, int y) override;

      virtual property MySQL::Geometry::Rect DisplayRect
      {
        MySQL::Geometry::Rect get() override
        {
          MySQL::Geometry::Rect rect(0, 0, baseSize.Width, baseSize.Height);
          rect.pos.x += shadowSize + shadowOffset + Padding.Left;
          rect.pos.y += shadowSize + shadowOffset + Padding.Top;
          rect.size.width -= Padding.Horizontal;
          rect.size.height -= Padding.Vertical;

          return rect;
        }
      }

      virtual property bool ShowWithoutActivation
      {
        bool get() override { return true; }
      }

      virtual property Windows::Forms::CreateParams^ CreateParams	
      {
        Windows::Forms::CreateParams^ get() override
        {
          Windows::Forms::CreateParams^ cp = PopupControl::CreateParams;

          cp->ExStyle |= (int) MySQL::Utilities::SysUtils::WS::EX_LAYERED;
          cp->ExStyle |= (int) MySQL::Utilities::SysUtils::WS::EX_NOACTIVATE;
          return cp;
        }
      }
    };

    public ref class PopupImpl : public ObjectImpl
    {
    protected:
      PopupImpl(mforms::Popup *self);
      ~PopupImpl();

      static bool create(mforms::Popup *self, mforms::PopupStyle style);
      static void set_needs_repaint(mforms::Popup *self);
      static void set_size(mforms::Popup *self, int width, int height);
      static int show(mforms::Popup *self, int spot_x, int spot_y);
      static MySQL::Geometry::Rect get_content_rect(mforms::Popup *self);
      static void set_modal_result(mforms::Popup *self, int result);

    public:
      static void init(Manager ^mgr)
      {
        mforms::ControlFactory *f= mforms::ControlFactory::get_instance();

        DEF_CALLBACK2(bool, mforms::Popup*, mforms::PopupStyle, mgr, f->_popup_impl, PopupImpl, create);
        DEF_CALLBACK1(void, mforms::Popup*, mgr, f->_popup_impl, PopupImpl, set_needs_repaint);
        DEF_CALLBACK3(void, mforms::Popup*, int, int, mgr, f->_popup_impl, PopupImpl, set_size);
        DEF_CALLBACK3(int, mforms::Popup*, int, int, mgr, f->_popup_impl, PopupImpl, show);
        DEF_CALLBACK1(MySQL::Geometry::Rect, mforms::Popup*, mgr, f->_popup_impl, PopupImpl, get_content_rect);
        DEF_CALLBACK2(void, mforms::Popup*, int, mgr, f->_popup_impl, PopupImpl, set_modal_result);
      }
    };

  };
};

#endif // _WF_POPUP_H_
