/*
 * Solitaire game (name unknown)
 *
 * Copyright (C) Evan Harris, 1994, 1995.
 *
 * Permission is granted to freely redistribute and modify this code,
 * providing the author(s) get credit for having written it.
 */

#include <stdlib.h>
#include <stdio.h>
#include <vga.h>
#include <vgamouse.h>

#include "vga16.h"
#include "mouse.h"
#include "key.h"
#include "solitaire.h"


#define SO_SCREENMODE G640x480x16
#define SO_SCREENWIDTH 640
#define SO_SCREENHEIGHT 480

#define SO_TEXTFG 1
#define SO_TEXTBG 0
#define SO_BUTTONFG 2
#define SO_BUTTONBG 6
#define SO_MOUSEFG 5
#define SO_MOUSEDARKFG 2
#define SO_OUTLINE 6

#define NUMCARDIMAGES 52
#define CARDACTUALWIDTH 53
#define CARDIMAGEWIDTH 56
#define CARDIMAGEHEIGHT 80

#define VERSION "SOLITAIRE\n  v1.3"
#define VERSIONLEFT (SO_SCREENWIDTH - 76)

#ifndef CARDSFILE
#define CARDSFILE "Cards56x80"
#endif
#ifndef CARDSMOUSESAMPLERATE
#define CARDSMOUSESAMPLERATE MOUSE_DEFAULTSAMPLERATE
#endif

#define USLEEP_TIME 30000


#define CARDTOIMAGE(card)	(card % NUMCARDIMAGES)

#define BITWIDTH CARDIMAGEWIDTH /* multiple of 8 */
#define BITHEIGHT CARDIMAGEHEIGHT
#define BITSHOWLEFT 24		/* multiple of 8 */
#define BITTOP 24
#define COLWIDTH (BITWIDTH + 8)
#define TOPROWHEIGHT	0


#define SO_QUITLEFT (SO_SCREENWIDTH - 80)
#define SO_QUITRIGHT (SO_QUITLEFT + 80)
#define SO_QUITBOTTOM 132
#define SO_QUITTOP (SO_QUITBOTTOM - 16)

#define SO_NEWGAMELEFT SO_QUITLEFT
#define SO_NEWGAMERIGHT SO_QUITRIGHT
#define SO_NEWGAMEBOTTOM 48
#define SO_NEWGAMETOP (SO_NEWGAMEBOTTOM - 16)

#define SO_RESTARTLEFT SO_QUITLEFT
#define SO_RESTARTRIGHT SO_QUITRIGHT
#define SO_RESTARTBOTTOM 76
#define SO_RESTARTTOP (SO_RESTARTBOTTOM - 16)

#define SO_UNDOLEFT SO_QUITLEFT
#define SO_UNDORIGHT SO_QUITRIGHT
#define SO_UNDOBOTTOM 104
#define SO_UNDOTOP (SO_UNDOBOTTOM - 16)


#define SO_PALETTESIZE 7

int palette[SO_PALETTESIZE * 3] = {
    0x00, 0x20, 0x00,		/* green */
    0x3f, 0x3f, 0x3f,		/* white */
    0x00, 0x00, 0x00,		/* black */
    0x37, 0x00, 0x00,		/* red */
    0x00, 0x00, 0x20,		/* blue */
    0x3f, 0x3f, 0x00,		/* yellow */
    0x1a, 0x1a, 0x1a,		/* grey */
};

#define MIN(a, b) ((a) < (b) ? (a) : (b))
static void LoadCards(char *file);
static long GetMouseButton(void);
static long WhileMouseButton(void);
static int ParsePos(long pos, int downcard, int buttonup);

static unsigned char ***imageCard;
static unsigned char **backCard;
static unsigned char **outlineCard;
static unsigned char *blankLine;


void
InitDisplay(int argc, char **argv)
{
    vga_disabledriverreport();
    vga_init();
#if !defined(CARDSMOUSE)
    vga_setmousesupport(1);
#endif
    
    if (vga_setmode(SO_SCREENMODE) != 0) {
	fprintf(stderr, "Mode %s not available!\n",
		vga_getmodename(SO_SCREENMODE));
	exit(1);
    }

#if defined(CARDSMOUSE)
    mouse_init("/dev/mouse", vga_getmousetype(), CARDSMOUSESAMPLERATE);
    mouse_setxrange(0, SO_SCREENWIDTH - 1);
    mouse_setyrange(0, SO_SCREENHEIGHT - 1);
    mouse_setwrap(MOUSE_NOWRAP);
#endif

    vga16_init();

    LoadCards(CARDSFILE);

    blankLine = (unsigned char *)calloc(SO_SCREENWIDTH, sizeof(unsigned char));
    if (blankLine == NULL) {
	fprintf(stderr, "Error: cannot get memory for blankLine\n");
	exit(1);
    }

    vga_setpalvec(0, SO_PALETTESIZE, &palette[0]);

    vga16_text(SO_NEWGAMELEFT, SO_NEWGAMEBOTTOM, " NEW GAME ",
	       SO_BUTTONFG, SO_BUTTONBG);
    vga16_text(SO_RESTARTLEFT, SO_RESTARTBOTTOM, " RESTART  ",
	       SO_BUTTONFG, SO_BUTTONBG);
    vga16_text(SO_UNDOLEFT, SO_UNDOBOTTOM, "   UNDO   ",
	       SO_BUTTONFG, SO_BUTTONBG);
    vga16_text(SO_QUITLEFT, SO_QUITBOTTOM, "   QUIT   ",
	       SO_BUTTONFG, SO_BUTTONBG);

    vga16_text(VERSIONLEFT, 200, VERSION, SO_TEXTFG, SO_TEXTBG);
}


static void
LoadCards(char *file)
{
    int i, j, k, l, c, colour;
    FILE *f;

    f = fopen(file, "r");
    if (f == NULL) {
	fprintf(stderr, "Cannot find '%s'\n", file);
	exit(1);
    }

    imageCard = (unsigned char ***)malloc(NUMCARDIMAGES
					  * sizeof(unsigned char **));
    if (imageCard == NULL) {
	fprintf(stderr, "Error: cannot get memory for imageCard\n");
	exit(1);
    }
    for (i = 0; i < NUMCARDIMAGES; i++) {
	imageCard[i] = (unsigned char **)malloc(CARDIMAGEHEIGHT
						* sizeof(unsigned char *));
	if (imageCard == NULL) {
	    fprintf(stderr, "Error: cannot get memory for imageCard[%d]\n",
		    i);
	    exit(1);
	}
	for (j = 0; j < CARDIMAGEHEIGHT; j++) {
	    imageCard[i][j] = (unsigned char *)malloc(CARDIMAGEWIDTH
						      * sizeof(unsigned char));
	    if (imageCard[i][j] == NULL) { 
		fprintf(stderr,
			"Error: cannot get memory for imageCard[%d][%d]\n",
			i, j);
		exit(1);
	    }
	    
	    if (SUIT(i) == SPADES || SUIT(i) == CLUBS) {
		colour = 2;
	    } else {
		colour = 3;
	    }
	    if (TYPE(i) < JACK) {
		for (k = 0; k < CARDIMAGEWIDTH / 8; k++) {
		    if ((c = getc(f)) == EOF) {
			fprintf(stderr, "Unexpected EOF in '%s'\n", file);
			exit(1);
		    }
		    for (l = 0; l < 8; l++) {
			imageCard[i][j][8 * k + l] =
			    (c & 1 << (7 - l)) ? colour : 1;
		    }
		}
		for (k = CARDACTUALWIDTH; k < CARDIMAGEWIDTH; k++) {
		    imageCard[i][j][k] = 0;
		}
	    } else {
		for (k = 0; k < CARDIMAGEWIDTH / 2; k++) {
		    if ((c = getc(f)) == EOF) {
			fprintf(stderr, "Unexpected EOF in '%s'\n", file);
			exit(1);
		    }
		    imageCard[i][j][2 * k] = (unsigned char)c >> 4;
		    imageCard[i][j][2 * k + 1] = (unsigned char)c & 0xf;
		}
	    }
	}
    }

    fclose(f);

    backCard = malloc(CARDIMAGEHEIGHT * sizeof(unsigned char *));
    outlineCard = malloc(CARDIMAGEHEIGHT * sizeof(unsigned char *));
    if (backCard == NULL || outlineCard == NULL) {
	fprintf(stderr, "Error: cannot get memory for cards");
	exit(1);
    }
    for (i = 0; i < CARDIMAGEHEIGHT; i++) {
	backCard[i] = malloc(CARDIMAGEWIDTH * sizeof(unsigned char));
	outlineCard[i] = calloc(CARDIMAGEWIDTH, sizeof(unsigned char));
	if (backCard[i] == NULL || outlineCard[i] == NULL) {
	    fprintf(stderr, "Error: cannot get memory for cards");
	    exit(1);
	}
    }
    for (i = 0; i < CARDACTUALWIDTH; i++) {
	outlineCard[0][i] = SO_OUTLINE;
	outlineCard[CARDIMAGEHEIGHT - 1][i] = SO_OUTLINE;
    }
    for (i = 0; i < CARDIMAGEHEIGHT; i++) {
	outlineCard[i][0] = SO_OUTLINE;
	outlineCard[i][CARDACTUALWIDTH - 1] = SO_OUTLINE;
    }
    for (i = 0; i < CARDIMAGEHEIGHT; i++) {
	for (j = 0; j < CARDACTUALWIDTH; j++) {
	    if ((i / 4) % 2 == (j / 4) % 2) {
		backCard[i][j] = 4;
	    } else {
		backCard[i][j] = 1;
	    }
	}
	for (; j < CARDIMAGEWIDTH; j++) {
	    backCard[i][j] = 0;
	}
    }
}


static unsigned char columnTop[COLUMNS];

/* We redisplay a whole column, rather than do anything tricky. */
void
DisplayColumn(short col)
{
    unsigned char **image;
    int card, length, row, height, line;
    int bittop = BITTOP;
	
    for (card = column[col], length = 0; card != NOCARD; card = next[card])
	length++;

    if (length > 0 && (SO_SCREENHEIGHT - TOPROWHEIGHT) / length < bittop)
	bittop = (SO_SCREENHEIGHT - TOPROWHEIGHT) / length;
    columnTop[col] = bittop;

    card = column[col];
    row = TOPROWHEIGHT;
    col = col * COLWIDTH;

    while (card != NOCARD && row < SO_SCREENHEIGHT - 1) {
	if (hidden[card])
	    image = backCard;
	else
	    image = imageCard[CARDTOIMAGE(card)];
	if (next[card] == NOCARD)
	    height = BITHEIGHT;
	else
	    height = bittop;
	if (row + height >= SO_SCREENHEIGHT)
	    height -= row + height - SO_SCREENHEIGHT + 1;

	for (line = 0; line < height; line++) {
	    vga16_drawscansegment(image[line], col, row + line, BITWIDTH);
	}

	row += height;
	card = next[card];
    }
    for (; row < SO_SCREENHEIGHT; row++) {
	vga16_drawscansegment(blankLine, col, row, BITWIDTH);
    }

    return;
}


short
GetCmd()
{
    int c, c1;
    
    for (c = NOCARD; c == NOCARD;) {
	c = GetMouseButton();
	c1 = WhileMouseButton();
	if (c >= 0) {
	    c = ParsePos(c, -1, 0);
	    c1 = ParsePos(c1, c, 1);

	    if (c == c1 && c >= 0) {
		return c;
	    }
	    c = NOCARD;
	}
    }

    return c;
}


static int oldx = -1, oldy, oldcolour[40];

static long
GetMouseButton()
{
    int x, y, button, key;

    x = mouse_getx();
    y = mouse_gety();
    if (x != oldx || y != oldy) {
	if (oldx != -1) {
	    RestoreUnderMousePointer(oldx, oldy, SO_SCREENWIDTH, SO_SCREENHEIGHT, oldcolour);
	}
	SaveUnderMousePointer(x, y, SO_SCREENWIDTH, SO_SCREENHEIGHT, oldcolour);
	RenderMousePointer(x, y, SO_MOUSEFG, SO_MOUSEDARKFG, SO_SCREENWIDTH, SO_SCREENHEIGHT);
	oldx = x;
	oldy = y;
    }
	
    for (;;) {
	usleep(USLEEP_TIME);	/* don't chew up as much CPU */

	if (mouse_update()) {
	    x = mouse_getx();
	    y = mouse_gety();
	    button = mouse_getbutton();

	    if (x != oldx || y != oldy) {
		if (oldx != -1) {
		    RestoreUnderMousePointer(oldx, oldy, SO_SCREENWIDTH, SO_SCREENHEIGHT, oldcolour);
		}
		SaveUnderMousePointer(x, y, SO_SCREENWIDTH, SO_SCREENHEIGHT, oldcolour);
		RenderMousePointer(x, y, SO_MOUSEFG, SO_MOUSEDARKFG, SO_SCREENWIDTH, SO_SCREENHEIGHT);
		oldx = x;
		oldy = y;
	    }
	
	    if (button & MOUSE_LEFTBUTTON) {
		break;
	    }
	}
	if ((key = key_getkey()) != -1) {
	    switch (key) {
	      case 'n':
	      case 'N':
		return NEWGAME;
	      case 'r':
	      case 'R':
		return RESTART;
	      case 'u':
	      case 'U':
		return UNDO;
	      case 'q':
	      case 'Q':
		return QUIT;
	      case '\014':
		vga16_redrawscreen();
		break;
	      default:
		break;
	    }
	}
    }

    if (x >= SO_NEWGAMELEFT && x < SO_NEWGAMERIGHT
	&& y > SO_NEWGAMETOP && y <= SO_NEWGAMEBOTTOM) {
	return NEWGAME;
    }
    if (x >= SO_RESTARTLEFT && x < SO_RESTARTRIGHT
	&& y > SO_RESTARTTOP && y <= SO_RESTARTBOTTOM) {
	return RESTART;
    }
    if (x >= SO_UNDOLEFT && x < SO_UNDORIGHT
	&& y > SO_UNDOTOP && y <= SO_UNDOBOTTOM) {
	return UNDO;
    }
    if (x >= SO_QUITLEFT && x < SO_QUITRIGHT
	&& y > SO_QUITTOP && y <= SO_QUITBOTTOM) {
	return QUIT;
    }

    return (x << 9) | y;
}


static long
WhileMouseButton()
{
    int x, y, button;

    for (;;) {
	usleep(USLEEP_TIME);	/* don't chew up as much CPU */

	mouse_update();
	x = mouse_getx();
	y = mouse_gety();
	button = mouse_getbutton();

	if (x != oldx || y != oldy) {
	    if (oldx != -1) {
		RestoreUnderMousePointer(oldx, oldy, SO_SCREENWIDTH, SO_SCREENHEIGHT, oldcolour);
	    }
	    SaveUnderMousePointer(x, y, SO_SCREENWIDTH, SO_SCREENHEIGHT, oldcolour);
	    RenderMousePointer(x, y, SO_MOUSEFG, SO_MOUSEDARKFG, SO_SCREENWIDTH, SO_SCREENHEIGHT);
	    oldx = x;
	    oldy = y;
	}
	
	if (!(button & MOUSE_LEFTBUTTON)) {
	    break;
	}
    }

    if (oldx != -1) {
	RestoreUnderMousePointer(oldx, oldy, SO_SCREENWIDTH, SO_SCREENHEIGHT, oldcolour);
    }
    oldx = -1;

    return (x << 9) | y;
}


static int
ParsePos(long pos, int downcard, int buttonup)
{
    int x, y, card, row, bittop;
    
    x = pos >> 9;
    y = pos & 0x1ff;
    if (x / COLWIDTH > 6 || y < TOPROWHEIGHT ||
	(x % COLWIDTH) / BITWIDTH > 0) {
	return NOCARD;
    }

    card = column[x / COLWIDTH];
    bittop = columnTop[x / COLWIDTH];
    row = TOPROWHEIGHT;

    while (card != NOCARD) {
	if ((next[card] != NOCARD && y >= row && y < row + bittop)
	    || (next[card] == NOCARD && y >= row && y < row + BITHEIGHT)) {
	    if (!hidden[card] && (downcard == -1 || downcard == card)) {
		return card;
	    }
	}
	card = next[card];
	row += bittop;
    }

    return NOCARD;
}
